/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/on",
    "dojo/dom-class",
    "dojo/dom-construct",
    "app/widgets/UReleaseWidget"
], function (
    declare,
    array,
    on,
    domClass,
    domConstruct,
    UReleaseWidget
) {
    /**
     *
     */
     return declare("app/widgets/utilities/PageSelector",
        [UReleaseWidget], {

            /**
             *
             */
            templateString:
                '<div class="pagination-selector">' +
                    '<div class="inline-block no-double-click-select" data-dojo-attach-point="leftArrow"></div>' +
                    '<div class="page-index inline-block no-double-click-select" data-dojo-attach-point="contents"></div>' +
                    '<div class="inline-block no-double-click-select" data-dojo-attach-point="rightArrow"></div>' +
                '</div>',

           /**
            * options: Define to use an option selector. Array of options to show. Option = {label, value}.
            *
            * numberOfPages: Define to use a page selector. Number of pages to show.
            *
            * displayCount: number of indexes to show on a page selector. Should be an odd number greater than 5.
            *
            * className: Additional class name to add to the pagination selector.
            */
            options: [],
            model:null,
            numberOfPages: null,
            selectedValue: null,
            displayCount: 9,
            className: null,

            /**
             *
             */
            postCreate: function() {
                this.inherited(arguments);
                var _this = this;
                if (this.displayCount < 5){
                    this.displayCount = 5;
                }
                else {
                    this.displayCount -= 2;
                }
                this._buildPageIndex();
                if (this.className){
                    domClass.add(this.domNode, this.className);
                }
                if(this.model) {
                    this.model.addChangeListener(function () {
                        _this.refresh();
                    }, this);
                }
            },

            /**
             * Rebuilds the selector.
             * TODO: eliminated this method.
             */
            refresh: function(selectedValue, numberOfPages){
                // if we are using a model to track page numbers, set them here.
                if(this.model) {
                    this.numberOfPages = this.model.get('totalPages');
                    this.selectedValue = this.model.get('pageNumber');
                } else {
                    //legacy behavior, getting this data from the table.
                    if (selectedValue){
                        this.selectedValue = selectedValue;
                    }
                    if (numberOfPages){
                        this.numberOfPages = numberOfPages;
                    }
                }
                this._buildPageIndex();
            },

            /**
             * Returns the current selected value or 0 if there is none.
             */
            getValue: function(){
                return this.selectedValue || 0;
            },

            /**
             * Builds the page or option selector.
             */
            _buildPageIndex: function(){
                var _this = this;
                domConstruct.empty(this.contents);
                if (this.numberOfPages){
                    domClass.add(this.leftArrow, "pagination-arrow");
                    domClass.add(this.rightArrow, "pagination-arrow");
                    this._setArrowStatus();
                    var firstMidOption = 0;
                    var lastMidOption = this.displayCount;

                    // Default selected page to 1 if not defined.
                    if (!this.selectedValue){
                        this.selectedValue = 1;
                    }
                    // The number of page links to show to the left and right of the current selected.
                    var extraShowValue = Math.floor(this.displayCount/2);
                    // The indexes to show an ellipsis when there are more pages that the display count.
                    firstMidOption = this.selectedValue - extraShowValue;
                    lastMidOption = this.selectedValue + extraShowValue;
                    if (this.selectedValue - (extraShowValue + 1) <= 1){
                        lastMidOption += (this.selectedValue - extraShowValue - 2) * - 1;
                        firstMidOption = 1;
                    }
                    else if (this.selectedValue + (extraShowValue + 1) >= this.numberOfPages){
                        firstMidOption -= (this.selectedValue + extraShowValue) - this.numberOfPages + 1;
                        lastMidOption = this.numberOfPages;
                    }
                    var i = 1;
                    for (i; i < this.numberOfPages + 1; i++){
                        // The selected value is not a link.
                        if (this.selectedValue === i){
                            this._createLink("div", i, null, "selected-page");
                        }
                        // Always have a link to the first and last pages.
                        // If the number of pages is equal to the display count, set all pages to be a link.
                        else if (i === 1 || i === this.numberOfPages || this.numberOfPages === (this.displayCount + 2)){
                            this._createLink("a", i);
                        }
                        // Create an ellipsis on a page overflow.
                        else if (i === firstMidOption || i === lastMidOption){
                            this._createLink("div", i18n("..."));
                        }
                        // Create page links between the ellipses.
                        else if (i > firstMidOption && i < lastMidOption){
                            this._createLink("a", i);
                        }
                    }
                }
                else if (this.options) {
                    // Option selector shows all options inline.
                    array.forEach(this.options, function(option){
                        var type = "a";
                        var className = "";
                        if (!_this.selectedValue && _this.defaultValue){
                            _this.selectedValue = _this.defaultValue;
                        }
                        else if (!_this.selectedValue){
                            _this.selectedValue = option.value;
                        }
                        if (_this.selectedValue === option.value){
                            type = "div";
                            className = "selected-page";
                        }
                        _this._createLink(type, option.value, option.label, className);
                    });
                }

            },

            /**
             * Creates a link or div of a page or option.
             */
            _createLink: function(type, value, label, className){
                var _this = this;
                var pageNumber = domConstruct.create(type, {
                    className: "page-number-link inline-block " + className,
                    innerHTML: label || value,
                    value: value
                }, this.contents);
                // Refresh the selector only when you click on a link.
                if (type === "a"){
                    on(pageNumber, "click", function(){
                        _this.selectedValue = value;
                        _this.onClick(_this.selectedValue);
                        _this.refresh();
                    });
                }
            },

            /**
             * Determines the state of the page arrows depending on the current selected page.
             */
            _setArrowStatus: function(){
                var _this = this;
                domClass.remove(this.leftArrow, "page-left-arrow-disabled");
                domClass.add(this.leftArrow, "page-left-arrow");
                domClass.remove(this.rightArrow, "page-right-arrow-disabled");
                domClass.add(this.rightArrow, "page-right-arrow");
                if (this.selectedValue === 1){
                    domClass.remove(this.leftArrow, "page-left-arrow");
                    domClass.add(this.leftArrow, "page-left-arrow-disabled");
                }
                if (this.selectedValue === this.numberOfPages){
                    domClass.remove(this.rightArrow, "page-right-arrow");
                    domClass.add(this.rightArrow, "page-right-arrow-disabled");
                }
                // Shift pages when clicking on the page arrows.
                if (!this.setArrows){
                    this.setArrows = true;
                    on(this.leftArrow, "click", function(){
                        if (_this.selectedValue !== 1){
                            _this.selectedValue -= 1;
                            _this.onClick(_this.selectedValue);
                            _this.refresh();
                        }
                    });
                    on(this.rightArrow, "click", function(){
                        if (_this.selectedValue !== _this.numberOfPages){
                            _this.selectedValue += 1;
                            _this.onClick(_this.selectedValue);
                            _this.refresh();
                        }
                    });
                }
            },

            /**
             * Function to call when clicking on a page number or option
             */
            onClick: function(value){
            },

            /**
             * Change the options when using the option selector
             */
            changeOptions: function(options){
                this.options = options;
            },

            /**
             * With a given domNode, place the selector within that node.
             */
            placeAt: function(node){
                domConstruct.place(this.domNode, node);
            },
            destroy: function () {
                if(this.model) {
                    this.model.removeChangeListeners(this);
                }
                this.inherited(arguments);
            }
        }
    );
});
