/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Cloud Sync
* (c) Copyright IBM Corporation 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals define, i18n, uReleaseConfig*/
define([
    "dojo/_base/declare",
    "dojo/dom-construct",
    "dojo/dom-class",
    "dojo/date/locale",
    "dojo/aspect",
    "dojo/on",
    "dojo/_base/array",
    "dojo/_base/xhr",
    "dojo/request/xhr",
    "dojo/json",
    "dojo/data/ItemFileWriteStore",
    "gridx/Grid",
    "gridx/core/model/cache/Async",
    "gridx/modules/SingleSort",
    "gridx/modules/ColumnResizer",
    "gridx/modules/CellWidget",
    "gridx/modules/Edit",
    'gridx/modules/Focus' ,
    "dijit/form/Button",
    "js/webext/widgets/Dialog",
    "js/webext/widgets/Alert",
    "app/widgets/UReleaseWidget"
], function(
    declare,
    domConstruct,
    domClass,
    dateLocale,
    aspect,
    on,
    array,
    xhr,
    xhrRequest,
    json,
    ItemFileWriteStore,
    Grid,
    Cache,
    Sort,
    ColumnResizer,
    CellWidget,
    Edit,
    Focus,
    Button,
    Dialog,
    Alert,
    UReleaseWidget) {
    return declare("app/widgets/sync/IntegrationProviderDialog", [UReleaseWidget], {
        templateString:'<div></div>',

        postCreate : function() {
            // create dialog
            var _this=this;
            this.dialog = new Dialog();
            this.executed=false;

            // title
            domConstruct.create("div", {
                innerHTML: i18n("Integration Providers")
            }, this.dialog.titleNode);

            domClass.add(this.dialog.containerNode, "integration-dialog");
            this.gridContainer = domConstruct.create("div", {
            }, this.dialog.containerNode);

            this.logContainer = domConstruct.create("div", {
                style: {
                    border: "1px solid lightGray",
                    marginTop: "3px",
                    padding: "10px"
                },
                "class": "hidden"
            }, this.dialog.containerNode);

            this.buttonContainer = domConstruct.create("div", {
                className: "underField"
            }, this.dialog.containerNode);

            // grid
            this.createGrid();

            // close button
            var closeButton = new Button({
                label: i18n("Close"),
                onClick: function(){
                    _this.dialog.hide().then(function(){
                        if (_this.executed && _this.refresh) {
                            _this.refresh();
                        }
                    });
                }
            });
            closeButton.placeAt(this.buttonContainer);

            this.dialog.show();

            // cleanup
            aspect.after(this.dialog, "onHide", function() {
                _this.grid.destroy();
                _this.dialog.destroy();
                _this.destroy();
            });
        },

        createGrid: function() {
            var _this=this;
            // create grid data
            var gridData = {
                identifier: "id",
                items: []
            };
            var id=1;
            array.forEach(this.model, function(ip,index){
                gridData.items.push({
                    id: id,
                    name: ip.get("name"),
                    ipIndex: index
                });
                id+=1;
            });

            // create grid
            this.ipUpdateMap = {};
            this.ipStatusMap = {};
            var store = new ItemFileWriteStore({data: gridData});
            var layout = [
                 {'name': i18n('Integration Provider'), 'field': 'name', 'width':  '150px'},
                 {'name': i18n('Last Update'), 'field': 'ipIndex', 'width':  '125px',
                     widgetsInCell: true,
                     decorator: function(){
                         return '<label data-dojo-attach-point="lastUpdate"></label>';
                     },
                     setCellValue: function(ipIndex){
                         var ip = _this.model[ipIndex];
                         _this.ipUpdateMap[ip.get("id")]=this.lastUpdate;
                         this.lastUpdate.innerHTML = dateLocale.format(
                                 new Date(ip.get("lastExecution")), {formatLength : "short"});
                     }
                 },
                 {'name': '', 'field': 'ipIndex', 'width': '100px',
                     widgetsInCell: true,
                     decorator: function(){
                         return [
                                 '<div data-dojo-attach-point="icon"',
                                    ' class="inline-block integration-status-icon row"',
                                    ' style="vertical-align: middle;margin:0;">',
                                    '</div>',
                                '<a data-dojo-attach-point="anchor">'+i18n("View Log")+'</a>'
                             ].join('');
                     },
                     setCellValue: function(ipIndex){
                         _this.ipStatusMap[_this.model[ipIndex].get("id")]=this.icon;
                         var status = _this.model[ipIndex].get("lastExecutionStatus");
                         if (status){
                             if (status === "Success"){
                                 domClass.remove(this.icon, "failed");
                                 domClass.add(this.icon, "success");
                             }
                             else {
                                 domClass.remove(this.icon, "success");
                                 domClass.add(this.icon, "failed");
                              }
                         }
                         on(this.anchor, "click", function() {
                             _this.showLog(_this.model[ipIndex]);
                         });
                     }
                 },
                 {'name': '', 'field': 'ipIndex', 'width': '225px',
                     widgetsInCell: true,
                     decorator: function(){
                         return [
                             '<span data-dojo-type="dijit.form.Button" ',
                                 'data-dojo-attach-point="btn" ',
                             '>'+i18n("Run Integration")+'</span>'
                         ].join('');
                     },
                     setCellValue: function(ipIndex){
                         var __this=this;
                         on(this.btn, "click", function() {
                             _this.runIntegrationProvider(__this.btn, _this.model[ipIndex]);
                         });
                     }
                 }
            ];
            this.grid = new Grid({
                store: store,
                cacheClass: Cache,
                style: {
                    height: "90px"
                },
                structure: layout,
                selectRowTriggerOnCell: true,
                modules: [
                          Sort,
                          ColumnResizer,
                          CellWidget,
                          Edit,
                          Focus
                 ]
            }, this.gridContainer);
            this.grid.startup();
        },

        runIntegrationProvider: function(btn, ip) {
            var _this = this;
            this.executed = true;
            btn.set("disabled", true);
            btn.set("label", i18n("Running..."));
            var postId = {"integrationProviderId": ip.get("id")};
            xhrRequest.post(uReleaseConfig.urls.base+"integrationProvider/execute",{
                data: postId,
                handleAs: "json"}).then(function(data) {
                    btn.set("disabled", false);
                    btn.set("label", i18n("Run Integration"));
                    _this.showLog(ip);
                },
                function(error) {
                    _this.errorAlert = new Alert({
                        forceRawMessages:true,
                        message: i18n("Failed to update: ") + "<br>" + error.response.text
                    });
                }
            );
        },

        showLog: function(ip) {
            var _this = this;
            this.updateLastReport(ip).then(function(){
                // update grid values
                _this.ipUpdateMap[ip.get("id")].innerHTML = dateLocale.format(
                        new Date(ip.get("lastExecution")), {formatLength : "short"});
                var status = ip.get("lastExecutionStatus");
                if (status){
                    if (status === "Success"){
                        domClass.remove(_this.ipStatusMap, "failed");
                        domClass.add(_this.ipStatusMap, "success");
                    }
                    else {
                        domClass.remove(_this.ipStatusMap, "success");
                        domClass.add(_this.ipStatusMap, "failed");
                     }
                }

                // show report
                var reportLogs = json.parse(ip.get("lastExecutionReport"));
                if (_this.logCloseButton) {
                    _this.logCloseButton.destroy();
                }
                domConstruct.empty(_this.logContainer);
                domClass.remove(_this.logContainer, "hidden");
                domConstruct.create("div", {
                    style: {
                        fontWeight: "bold",
                        marginBottom: "4px"
                    },
                    innerHTML: i18n("Last Execution Report of ") + ip.get("name")
                }, _this.logContainer);

                var reportHtml = domConstruct.create("div", {
                    className: "integration-row-report columnForm",
                    style: "line-height: 20px;"
                }, _this.logContainer);

                var reportHtmlContainer = domConstruct.create("div", {
                    className: "table"
                }, reportHtml);

                array.forEach(reportLogs, function(log) {
                    domConstruct.create("div", {
                        className: log.className,
                        innerHTML: log.text
                    },reportHtmlContainer);
                });

                var closeButtonContainer = domConstruct.create("div", {
                }, _this.logContainer);
                _this.logCloseButton = new Button({
                    label: i18n("Close"),
                    onClick: function(){
                        domClass.add(_this.logContainer, "hidden");
                        _this.dialog.resize();                    }
                });
                _this.logCloseButton.placeAt(closeButtonContainer);
                _this.dialog.resize();
            });
        },

        updateLastReport: function(ip) {
            var deferred = xhr.get({
                url: uReleaseConfig.urls.base+"integrationProvider/"+ip.resourceId+"/updateExecutionStatus",
                handleAs: "json",
                load: function(execution) {
                    ip.set("lastExecutionStatus",execution.status);
                    ip.set("lastExecutionReport",execution.report);
                    ip.set("lastExecution",execution.lastUpdate);
                }
            });
            return deferred.promise;
        }



    });
});
