/** Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*global define, uReleaseConfig, util, i18n */
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/_base/lang",
    "app/widgets/UReleaseWidget",
    "app/widgets/select/FilterableCheckedMultiSelect"
], function(
    declare,
    array,
    lang,
    _UReleaseWidget,
    FilterableCheckedMultiSelect
) {

        /**
          FilterableCheckedMultiSelect
         * provides a multi select checkbox, with a text box above it.  Options in the multi-select
         * {
         *
         *  CONSTRUCTOR ARGUMENTS:
         *  name: {String} the html name attribute of the CheckedMultiSelect, used if this widget is submitted with a form.
         *  multiple: {Boolean} whether or not multiple items can be selected at once.
         *  placeHolderText: {String} the title text of the type of thing that we are filtering, literally shows up as "Filter {placeHolderText}"
         *  onChange: {Function} a function to be fired whenever an item is selected
         *
         *  EXISTING PROPERTIES AND METHODS:
         *  addOption(Object|Array): a single option or list of options of the form {label:"", value:""}
         *
         */
    return declare("app/widgets/select/FilterableMultipleListCheckedMultiSelect", [_UReleaseWidget],{

        /*******************************************************************************
         * PROPERTIES
        *******************************************************************************/
        templateString:'<div></div>',

        /**
         * base class name appended to the widgets top most dom node
         * @see app/widgets/UReleaseWidget
         */
        className:'filterable-multiple-list-checked-multi-select',

        /**
         * whether or not there are multiple separate lists.
         * if false, there will only be 1 list of options
         * with many lists of values for each key
         */
        multipleLists: true,

        /**
         * the widget used to show a single options list
         */
        _selectWidget:null,

        /**
         * the widget used to show a single options list
         */
        _visibleListName:null,

        /**
         * contains all options lists that this widget uses
         */
        _value: null,
        /**
         * contains all selected values from all lists in the form
         * <pre>
         * {
         *     listName: [{
         *             label:"foo",
         *             value:"bar"
         *         },{
         *             label:"foo",
         *             value:"bar"
         *     }],
         *     otherListName:[]
         *     ...
         * }
         */
        _lists: null,


        /*******************************************************************************
         * PUBLIC METHODS
        *******************************************************************************/
        /**
         *
         */
        setList: function (name, value) {
            if(!lang.isArray(value)) {
                throw new Error("a list must be an array of objects");
            }
            if(!this.multipleLists) {
                name = 'masterList';
            }
            this._lists[name] = value;
        },
        /**
         *
         */
        show: function (name) {
            this._setVisibleList(name);
        },
        /**
         *
         */
        refresh: function () {
            this.show(this._visibleListName);
        },
        /**
         * @abstract
         */
        onChange: function () {
            //no-op by default.
        },

        /*******************************************************************************
         * PRIVATE METHODS
        *******************************************************************************/
        postCreate: function () {
            this.inherited(arguments);
            this._initLists();
            this._initUI();
        },

        /**
         * initialize the empty lists and values containers
         */
        _initLists: function () {
            this._lists = {};
            this._value = {};
        },

        /**
         * draw the multi-select widget that the user will interact with
         */
        _initUI: function () {
            var _this = this;
            this._selectWidget = new FilterableCheckedMultiSelect({
                multiple: this.multipleList,
                // fire user specified on change before setting values internally
                // so that the onChange can be used to alter stored data if the
                // api user wishes
                onChange: function (value) {
                    // XXX test this.  Maybe we can pass a value in to be modified
                    // without returning it?
                    this.value = value;
                    _this.onChange(this.value);
                    _this._storeValue(this.value);
                }

            });
            this._selectWidget.placeAt(this.domNode);

        },

        _getList: function (listName) {
            if(!this.multipleLists) {
                // if we are basing multiple options off of the same list, then
                // always user 'masterList'
                return this._lists.masterList;
            }
            return this._lists[listName];
        },
        /**
         * replace the currently shown list with another one from the set of lists
         */
        _setVisibleList: function (listName) {
            var list = this._getList(listName);
            // if the api user tries to set a list that hasn't been created
            // do nothing.
            if(lang.isArray(list)) {
                this._clearDisplayedOptions();
                var currentList = this._setSelectedItems(listName);
                this._selectWidget.addOption(currentList);
                this._visibleListName = listName;
            } else {
                // it may not be an error to try to select a list that doesn't exist
                console.warn("Tried to show list:", listName, "that was",
                    this._getList(listName), "but should have been an array" );
            }
        },
        _setSelectedItems: function (listName) {
            function setSelected(listItem, selectedValue) {
                if(listItem.value === selectedValue) {
                    listItem.selected = true;
                }
                //else {
                    //listItem.selected = false;
                //}
            }
            var selectedValues = this._value[listName];

            var list = lang.clone(this._getList(listName));
            if(selectedValues) {
                array.forEach(list, function (listItem) {
                        if(lang.isArray(selectedValues)) {
                            array.forEach(selectedValues, function (selectedValue) {
                                setSelected(listItem, selectedValue);
                            });
                        }
                        //else {
                            //// selectedValues is actually a string, just use it
                            //setSelected(listItem, selectedValues);
                        //}
                });
            } // else selected defaults to undefined (not selected)
            return list;
        },

        /**
         * remove all displayed options from the select widget
         */
        _clearDisplayedOptions: function () {
            var _this = this,
                currentOptions = this._selectWidget.getOptions();
            array.forEach(currentOptions, function (option) {
                _this._selectWidget.removeOption(option);
            });
        },

        /**
         * stores the values selected for the current list.
         * @param value an array of all selected values from the currently
         * visible list
         */
        _storeValue: function (value) {
            // do not make a check for array here, since single items will be strings,
            // not string arrays of length 1
            this._value[this._visibleListName] = value;
        },

        /**
         * //XXX unnecessary?
         */
        //_valueGetter: function () {
            //return this._value;

        //},
        /**
         *
         */
        _valueSetter: function (valueObject) {

            var listName,
                list,
                updatingVisibleList = false;
            for(listName in valueObject) {
                if(valueObject.hasOwnProperty(listName)) {
                    //convenience variables
                    list = valueObject[listName];

                    // set the list value
                    this._value[listName] = list;
                    if(this._visibleListName === listName) {
                        updatingVisibleList = true;
                    }
                }
            }
            if(updatingVisibleList) {
                this.refresh();
            }

        }
    });
});
