/** Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*global define, uReleaseConfig, util, i18n */
define([
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/_base/event",
    "dojo/_base/array",
    "dojo/on",
    "dojo/dom-class",
    "dojo/query",
    "dijit/form/TextBox",
    "dojox/form/CheckedMultiSelect",
    "app/widgets/UReleaseWidget",
    "dojo/dom-construct"
], function(
    declare,
    lang,
    event,
    array,
    on,
    domClass,
    query,
    TextBox,
    CheckedMultiSelect,
    UReleaseWidget,
    domConstruct
) {

        /**
          FilterableCheckedMultiSelect
         * provides a multi select checkbox, with a text box above it.  Options in the multi-select
         * {
         *
         *  CONSTRUCTOR ARGUMENTS:
         *  name: {String} the html name attribute of the CheckedMultiSelect, used if this widget is submitted with a form.
         *  title: {String} title of the CheckedMultiSelect
         *  description: {String} description of the CheckedMultiSelect
         *  preSelectedOptions: {Array} array of items to mark as selected
         *  multiple: {Boolean} whether or not multiple items can be selected at once.
         *  placeHolderText: {String} the title text of the type of thing that we are filtering, literally shows up as "Filter {placeHolderText}"
         *  onChange: {Function} a function to be fired whenever
         *
         *  EXISTING PROPERTIES AND METHODS:
         *  addOption(Object|Array): a single option or list of options of the form {label:"", value:""}
         *
         */
    return declare("app/widgets/FilterableCheckedMultiSelect", [UReleaseWidget],{

        /**
         * the containing template string for the multiselect
         */
        templateString:"<div>"+
            "<div data-dojo-attach-point=\"headerContainer\"></div>" +
            "</div>",
        /**
         * the base classname of the widget
         * @see app/widgets/UReleaseWidget
         */
        className:"filterable-checked-multi-select",
        /**
         * the html name of the checked multi select
         */
        name:null,
        /**
         * the title of the checked multi select
         */
        title: null,
        /**
         * the description of the checked multi select
         */
        description: null,
        /**
         * the list of already selected options
         */
        preSelectedOptions: null,
        /**
         * whether or not multiple items can be selected.
         */
        multiple:true,

        /**
         * the display name of the kind of content that this multiselect is showing.
         */
        placeHolderText:null,

        /**
         * expose the onChange method for the multiSelect
         */
        onChange: function (arg) {
            // no-op
        },

        /**
         * expose the addOption method from the multiSelect
         * note: this is private, but without underscore to keep the api the same as the CheckedMultiSelect
         */
        addOption: function (arg) {
            this._multiSelect.addOption(arg);
        },

        /**
         * Removes an option from the multi-select
         * note: this is private, but without underscore to keep the api the same as the CheckedMultiSelect
         */
        removeOption: function (arg) {
            this._multiSelect.removeOption(arg);
        },

        /**
         * Returns the options corresponding with the given value or index
         * note: this is private, but without underscore to keep the api the same as the CheckedMultiSelect
         */
        getOptions: function (arg) {
            return this._multiSelect.getOptions(arg);
        },

        /**
         * Method that rebuild the list of options and marks a value as selected
         */
        refreshValue: function (value) {
            var _this = this;
            var optionList = this._multiSelect.getOptions();
            this._multiSelect.removeOption(optionList);

            var options = [];
            array.forEach(optionList, function (item) {
                options.push({
                    label: item.label,
                    value: item.value,
                    selected:item.value === value
                });
            });
            this._multiSelect.addOption(options);
        },

        /**
         * the textbox acting as a filter for this widget
         */
        _filterBox:null,

        /**
         * the checked multi select acting as part of this widget
         */
        _multiSelect:null,

        /**
         * the string aganst which the multi select will be filtered.
         */
        _filterString:null,


        /**
         * initialization method (see _WidgetBase
         */
        postCreate: function() {
            this.inherited(arguments);
            this.initializeUI();
            var _this = this;

            if (_this.title && _this.description) {
                var mappingSection = domConstruct.create("div", {
                    className: "header-section"
                }, _this.headerContainer);

                var titleHtml = domConstruct.create("h2", {
                    innerHTML: _this.title
                }, mappingSection);

                var info = domConstruct.create("p", {
                    innerHTML: _this.description
                }, mappingSection);
            }
        },
        initializeUI: function () {
            this._filterBox = new TextBox({
                placeholder: this.placeHolderText || i18n("Filter items")
            });
            domClass.add(this._filterBox.domNode, "filterable-checked-multi-select-text-box");

            this.own(
                on(this._filterBox, "keyup", lang.hitch(this,"_updateFilter")),
                on(this._filterBox, "submit", function (e) { event.stop(e);})
            );
            this.watch("_filterString", lang.hitch(this,"_doFilterOptions"));


            this._multiSelect = new CheckedMultiSelect({
                name: this.name,
                multiple: this.multiple,
                onChange: this.onChange
            });
            // uncomment these commented lines and style the widget properly, and selection inversion will work.
            //this._invertButton = new Button({
                //label:"invert",
                //onClick: function () {
                    //_this._multiSelect.invertSelection();
                //}
            //});
            //domClass.add(this._invertButton.domNode, "filterable-checked-multi-select-button");

            this._filterBox.placeAt(this.domNode);
            //this._invertButton.placeAt(this.domNode);
            this._multiSelect.placeAt(this.domNode);
        },
        _updateFilter: function () {
            this.set('_filterString', this._filterBox.get('value'));
        },
        _doFilterOptions: function (name, oldVal, newVal) {
            array.forEach(query(".dojoxMultiSelectItem", this._multiSelect.domNode), function (item) {
                var node = query(".dojoxMultiSelectItemLabel", item)[0];
                var value = node.innerHTML;
                if(value.trim().toLowerCase().indexOf(newVal.trim().toLowerCase()) > -1) {
                    domClass.remove(item, "hidden");
                } else {
                    domClass.add(item, "hidden");
                }
            });
        }
    });
});
