/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals define */
define([
    "dojo/_base/declare",
    "dojo/on",
    "dojo/_base/lang",
    "dojo/_base/event",
    "dojo/_base/array",
    "dojo/dom-construct",
    "dijit/form/Button",
    "dijit/Dialog",
    "app/widgets/UReleaseWidget",
    "app/widgets/mixins/_ResourceMixin",
    "js/webext/widgets/table/TreeTable",
    "js/webext/widgets/GenericConfirm",
    "dojo/text!./templates/ResourceTreeTable.html"
], function (
    declare,
    on,
    lang,
    event,
    array,
    domConstruct,
    Button,
    Dialog,
    _UReleaseWidget,
    _ResourceMixin,
    TreeTable,
    Confirm,
    template
) {

    /**
     * Table variant of ResourceList which shows the members as rows in a table
     * Takes an object following:
     *  {
     *      model: Resource instance containing a collection
     *      widgetClass: The dojo "class" object to instantiate for each member of the collection
     *      showHeaders: boolean, indicates whether to show column headers in the table
     *  }
     */

    return declare("app/widgets/ResourceTreeTable",
            [_UReleaseWidget, TreeTable, _ResourceMixin], {

        templateString: template,
        /**
         * whether or not pagination and other things are processed server-side
         * @see TreeTable
         */
        serverSideProcessing: false,

        /**
         * the array of columns to show in the ResourceTreeTable
         * @see TreeTable
         */
        columns: [],
        /**
         * whether or not to hide the footer
         * @see TreeTable
         */
        hideFooter: true,

        /**
         * the property of each returned xhr object in the list to use as the 'id'
         * @see TreeTable
         */
        idAttribute: "resourceId",

        /**
         * whether or not to show expand / collapse buttons
         * @see TreeTable
         */
        hideExpandCollapse: true,

        /**
         * the resource class backing this TreeTable
         * @see Resource
         * @see ResourceRegistry
         */
        Resource: null,

        /**
         * the resource instance backing this TreeTable
         * @see Resource
         */
        resource: null,

        /**
         * The text that we want to see for the add button
         * @type {String}
         */
        addLabel: "",

        /**
         * The node that we want to attach the add button to.
         * @type {Node}
         */
        labelNode: null,

        /**
         * The text we want for the title of the dialog opened with the add button
         * @type {String}
         */
        addDialogLabel:"",

        /**
         * custom name of the input button.
         * @type {String}
         */
        buttonName: "button",


        constructor: function () {
            var _this = this;

            /**
             * a list of columnDefinitions that can be used by all subclasses to DRYly add columns to a ResourceTreeTable
             */
            this.columnDefinitions = {
                name: {
                    name: i18n("Name"),
                    formatter: function(item) {
                        var container = domConstruct.create("div", {
                            className: "resource-tree-item-name-container"
                        });

                        domConstruct.create("a",{
                            innerHTML: item.getHTML("name"),
                            className: "resource-tree-item-name"
                        }, container);

                        var description = item.get("description");
                        if (description){
                            domConstruct.create("div",{
                                innerHTML: description,
                                className: "resource-tree-item-description"
                            }, container);
                        }
                        return container;
                    }
                },
                options: {
                        name: i18n("Actions"),
                        formatter: function(item) {
                            var optionsDiv = domConstruct.create("div");
                            if (item.canWrite()) {
                                var writeDiv = domConstruct.create("div", {
                                    className: "resource-button edit-button",
                                    style: {
                                        position: "relative",
                                        left: "0",
                                        display: "inline-block"
                                    }
                                }, optionsDiv);

                                on(writeDiv, "click", function(e) {
                                    event.stop(e);
                                    _this.edit(item);
                                });
                            }
                            if (item.canDelete()) {
                                var deleteDiv = domConstruct.create("div", {
                                    className: "resource-button delete-button",
                                    style: {
                                        position: "relative",
                                        left: "5px",
                                        display: "inline-block"
                                    }
                                }, optionsDiv);

                                on(deleteDiv, "click", function(e) {
                                    event.stop(e);
                                    var confirmDialog = new Confirm({
                                        message: i18n("Are you sure you want to delete?"),
                                        showUnderField: true,
                                        action: function() {
                                            item.remove();
                                            _this.hideRow(_this.rowObjects[item.resourceId]);
                                            _this.onDelete();
                                        }
                                    });
                                });
                            }
                            return optionsDiv;
                        }
                    }
                };
        },

        /**
         * MAIN method
         */
        postCreate: function () {
            this.setColumns();
            this.inherited(arguments);
            this.createAddButton();
        },

        _initI18n : function () {
            this.inherited(arguments);
            this.addLabel = i18n("Add New");
            this.addDialogLabel = i18n("Add New");
        },

        /**
         * initialize the resource, and then call loadTable as implemented in TreeTable
         * @see TreeTable
         */
        loadTable: function() {
            this._initializeResource().then(
                lang.hitch(this, "inherited", arguments)
            );
        },

        onDelete: function() {
            //no op
        },

        /**
         * a method to set the columns to be displayed in the ResourceTreeTable This can be overridden by extensions of this class to add or remove columns
         * @see addColumnDefinition also
         */
        setColumns: function () {
            this.columns = [
                this.columnDefinitions.name,
                this.columnDefinitions.options
            ];
        },

        /**
         * describes how to get item objects to be displayed in the ResourceTreeTable's table
         * @see TreeTable
         */
        getData: function() {
            return this.resource.getMembers();
        },

        /**
         * select a row of the ResourceTreeTable.
         */
        selectRowForResource: function (resource) {
            var row = this._getRowObjectForResource(resource);
            this.selectRow(row.domNode);
        },

        /**
            * Get the internal object tracking tree data for a given resource.
            */
        _getRowObjectForResource: function(resource) {
            var rowObject;
            array.some(this.rowObjectList, function(thisRowObject) {
                if (thisRowObject.item.resourceId === resource.resourceId) {
                    rowObject = thisRowObject;
                    return true;
                }
            });

            return rowObject;
        },

        /**
         * describes how to get child items of item objects displayed in the ResourceTreeTable's table
         * @see TreeTable
         */
        getItemChildren: function(item) {
            var children = item.get && item.get("children");
            if(children) {
                return children.getMembers();
            }
        },

        /**
         *
         */
        addColumn: function (column, afterIndex) {
            var i = 0;
            if(!afterIndex) {
                this.columns.push(column);
            } else {
                if(lang.isObject(afterIndex)) {
                    while(afterIndex.name !== this.columns[i].name) {
                        i++;
                    }
                    afterIndex = i;
                }
                afterIndex++;

                this.columns.splice(afterIndex, 0, column);
            }
        },

        /**
         *
         */
        removeColumn: function (name) {
            var index;
            array.some(this.columns, function (col, i) {
                index = i;
                return col.name.toLowerCase() === name.toLowerCase();
            });
            this.columns.splice(index, 1);
        },

        /**
         * add a definition to the list of the column definitions available to show
         * @param columnObject the an object containing the definition of a column @see TreeTable
         */
        addColumnDefinition: function (columnObject) {
            var x;
            // if the column(s) to add is/are passed in a hash, read the hash into the columndefinitions object.
            if (!columnObject instanceof Object) {
                throw new Error(i18n("add Column only takes an object of the form `{columnName:{/*col object here*/},otherName:{/*col object here*/}}` as an argument"));
            }
            for(x in columnObject) {
                if (columnObject.hasOwnProperty(x)) {
                    this.columnDefinitions[x] = columnObject[x];
                }

            }
            return columnObject;
        },

        createAddButton: function() {
            var _this = this;

            if(_this.labelNode) {
                domConstruct.empty(_this.labelNode);
            }
            if(_this.resource.canAddMember() && !!_this.labelNode) {
                // create the button
                var addButton = new Button({
                    name: _this.buttonName,
                    label: _this.addLabel,
                    baseClass: "dijit dijitReset dijitInline dijitButton idxButtonSpecial"
                });

                addButton.placeAt(_this.labelNode);

                on(addButton, "click", function() {
                    var dialog = new Dialog({
                        title: _this.addDialogLabel
                    });
                    on(dialog, "hide", function() {
                        _this.resource.load().then(function() {
                            _this.resource.members.sort(function(a, b) {
                                return b.serviceData.name.toUpperCase() < a.serviceData.name.toUpperCase() ? 1 : -1;
                            });
                            _this.refresh();
                            dialog.destroy();
                        });
                        // anythign to do here?
                    });
                    _this.showEditForm(dialog);
                    dialog.show();
                });
            }
        }
    });
});
