/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
/*globals uReleaseConfig, define, dijit */
define([
    "dojo/_base/declare",
    "dojo/on",
    "dojo/dom-construct",
    "dijit/form/Button",
    "dijit/Dialog",
    "app/model/pipemodel/LifecycleModel",
    "app/widgets/resourceTreeTable/ResourceTreeTable",
    "js/webext/widgets/ColumnForm",
    "js/webext/widgets/RestSelect"
], function(
    declare,
    on,
    domConstruct,
    Button,
    Dialog,
    LifecycleModel,
    ResourceTreeTable,
    ColumnForm,
    RestSelect
) {

    /**
     * Table variant of ResourceList which shows the members as rows in a table
     * Takes an object following:
     *  {
     *      model: Resource instance containing a collection
     *      widgetClass: The dojo "class" object to instantiate for each member of the collection
     *      showHeaders: boolean, indicates whether to show column headers in the table
     *  }
     */
    return declare("app/widgets/resourceTreeTable/LifecycleTreeTable", [ResourceTreeTable], {

        /**
         * the resource class backing this TreeTable
         */
        Resource: LifecycleModel,

        postCreate: function() {
            this.addColumnDefinition({
                "derived": {
                    "name": i18n("Extends"),
                    "formatter": function(item) {
                        var result = domConstruct.create("div", {className: "derived-from-container"});

                        var lifecycleModelData = item.get("parentLifecycleModel");
                        if (lifecycleModelData) {
                            domConstruct.create("span", {
                                "innerHTML": String(lifecycleModelData.get("name")).escape()
                            }, result);

                            if (item.isNoncompliant()) {
                                var noncompliantLink = domConstruct.create("a", {
                                    "innerHTML": i18n("( Non-compliant )"),
                                    "class": "derived-from-container-label inlineBlock",
                                    "style": {
                                        "marginLeft": "8px"
                                    }
                                }, result);
                                on(noncompliantLink, "click", function() {
                                    item.showNoncompliantPhaseModels();
                                });
                            }
                        }

                        return result;
                    }
                }
            });
            this.inherited(arguments);
        },

        /**
         *
         */
        setColumns: function () {
            this.columns = [
                this.columnDefinitions.name,
                this.columnDefinitions.derived,
                this.columnDefinitions.options
            ];
        },
        /**
         * @extends MasterDetailView
         */
        getData: function() {
            this.resource.sortMembers();
            return this.resource.getMembers();
        },

        edit: function(item) {
            var _this = this;
            var dialog = new Dialog({
                title: i18n("Edit Lifecycle")
            });
            on(dialog, "hide", function() {
                _this.resource.load().then(function() {
                    _this.resource.members.sort(function(a, b) {
                        return b.serviceData.name.toLowerCase() < a.serviceData.name.toLowerCase() ? 1 : -1;
                    });
                    _this.refresh();
                    dialog.destroy();
                });
            });
            this.showEditForm(dialog, item);
        },

        /**
         * XXX: This pattern is appearing in Milestone, Lifecycle, and
         * Application tree table. We may want to DRY this up.
         */
        showEditForm: function(dialog, model) {
            var _this = this;
            domConstruct.empty(dialog.containerNode);

            var newUrl = uReleaseConfig.urls.base + "lifecycleModel/" + (model ? model.get('id') : "");
            var form = new ColumnForm({
                submitUrl: newUrl,
                submitMethod: model ? "PUT" : "POST",
                submitFormat: "JSON",
                postSubmit: function(data, ioArgs) {
                    dialog.hide();

                    _this.resource.load().then(function() {
                        _this.loadTable();
                    });
                },
                onCancel: function() {
                    dialog.hide();
                }
            });

            form.addField({
                name: "name",
                value: model ? model.get('name') : "",
                label: i18n("Name"),
                required: true,
                type: "Text"
            });

            form.addField({
                name: "description",
                value: model ? model.get('description') : "",
                label: i18n("Description"),
                type: "Text Area",
                style: "width: 284px; font: inherit;"
            });
            form.addField({
                name: "id",
                value: model ? model.get('id') : "",
                type: "hidden"
            });

            var parentLifecycleModel = model ? model.get('parentLifecycleModel') : null;
            var parentLifecycleModelId = parentLifecycleModel ? parentLifecycleModel.get('id') : null;

            var lifecycleModelSelect = new RestSelect({
                "value": parentLifecycleModelId || "",
                "name": "parentLifecycleModelId",
                "restUrl": uReleaseConfig.urls.base + "lifecycleModel/",
                "getLabel": function(item) {
                    return item.name;
                },
                "getValue": function(item) {
                    return item.id;
                },
                "isValid": function(item) {
                    // if this is backed by a model, make sure we can't try to make
                    //it depend on itself.  If it's new, anything goes.
                    return model ? ( !! model && model.get('id') !== item.id) : true;
                }
            });

            form.addField({
                name: "parentLifecycleModelId",
                label: i18n("Extends"),
                widget: lifecycleModelSelect
            });

            var milestoneSequence = model ? model.get('milestoneSequence') : null;
            var milestoneSequenceId = milestoneSequence ? milestoneSequence.get('id') : null;

            var templateSelect = new RestSelect({
                "autoSelectFirst": false,
                "allowNone": true,
                "value": milestoneSequenceId || "",
                "name": "milestoneSequenceId",
                "restUrl": uReleaseConfig.urls.base + "milestoneSequence/",
                "getLabel": function(item) {
                    return item.name;
                },
                "getValue": function(item) {
                    return item.id;
                },
                "onChange": function(value, item) {
                    milestoneSequence = value;
                }
            });

            form.addField({
                name: "milestoneSequenceId",
                label: i18n("Process Checklist Template"),
                widget: templateSelect
            });

            form.placeAt(dialog.containerNode);
            dialog.set("focused", true);
            dialog.show();
        },

        /**
         * override getter for child items
         * //FIXME:  TreeTable expects data to be plain java objects,
         * not Resources, so we need to find a real solution long term, or stop using WebExt
         */
        getItemChildren: function() {
            return [];
        }
    });
});
