/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals uReleaseConfig, define, i18n*/
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/on",
    "dojo/request/xhr",
    "dijit/Dialog",
    "app/model/application/Application",
    "app/widgets/resourceTreeTable/ResourceTreeTable",
    "js/webext/widgets/ColumnForm"
], function (
    declare,
    array,
    on,
    xhr,
    Dialog,
    Application,
    ResourceTreeTable,
    ColumnForm
) {

    /**
     * Table variant of ResourceList which shows the members as rows in a table
     * Takes an object following:
     *  {
     *      model: Resource instance containing a collection
     *      widgetClass: The dojo "class" object to instantiate for each member of the collection
     *      showHeaders: boolean, indicates whether to show column headers in the table
     *  }
     */
    return declare("app/widgets/resourceTreeTable/ApplicationTreeTable",
            [ResourceTreeTable], {

        /**
         * the resource class backing this TreeTable
         */
        Resource: Application,

        /**
         * Edit the currently selected item in the tree table.
         */
        edit: function(item) {
            var _this = this;
            var dialog = new Dialog({
                title: i18n("Edit Application")
            });
            on(dialog, "hide", function() {
                _this.resource.load().then(function() {
                    _this.resource.members.sort(function(a, b) {
                        return b.serviceData.name.toLowerCase() < a.serviceData.name.toLowerCase() ? 1 : -1;
                    });
                    _this.refresh();
                    dialog.destroy();
                });
            });
            this.showEditForm(dialog, item);
        },

        /**
         * XXX: This pattern is appearing in Milestone, Lifecycle, and
         * Application tree table. We may want to DRY this up.
         */
        showEditForm: function(dialog, item) {
            var newUrl = uReleaseConfig.urls.base+"application/";
            if (item) {
                newUrl = newUrl + item.get("id");
            }

            var form = new ColumnForm({
                submitUrl: newUrl,
                submitMethod: item ? "PUT" : "POST",
                submitFormat: "JSON",
                postSubmit: function() {
                    dialog.hide();
                },
                onCancel: function() {
                    dialog.hide();
                }
            });

            //User can edit team associations, so we want to allow canEdit(), but for apps from integrations,
            //they can't edit the name and description.
            var readOnlyMode = false;

            if (item) {
                if (item.isAutomatedApplication()) {
                    readOnlyMode = true;
                }
            }

            form.addField({
                name: "name",
                label: i18n("Name"),
                value: item ? item.get("name") : null,
                required: true,
                readOnly: readOnlyMode,
                type: "Text"
            });

            form.addField({
                name: "description",
                label: i18n("Description"),
                value: item ? item.get("description") : null,
                readOnly: readOnlyMode,
                type: "Text Area",
                style: "width: 284px; font: inherit;"
            });

            form.addField({
                name: "level",
                type: "hidden",
                value: "APPLICATION"
            });

            if (item) {
                form.addField({
                    name: "id",
                    value: item.get('id'),
                    type: "hidden"
                });

                item.parent = null;
                item.load().then(function() {
                    xhr.get(uReleaseConfig.urls.base + "teams/", {
                        headers: {
                            "Accept": "application/json"
                        },
                        handleAs: "json"
                    }).then(function(data) {
                        var teamArray = [];
                        var selectedTeamArray = [];
                        array.forEach(item.get("teams"), function(team) {
                            selectedTeamArray.push(team.id);
                        });
                        array.forEach(data, function(team) {
                            var teamObject = {
                                label: (team.name ? (String(team.name)).escape() : i18n("(Error: No team)")),
                                value: team.id,
                                selected: selectedTeamArray.indexOf(team.id) > -1
                            };
                            teamArray.push(teamObject);
                        });
                        form.addField({
                            name: "teams",
                            label: i18n("Team"),
                            type: "CHECKED-MULTI-SELECT",
                            required: true,
                            allowedValues: teamArray,
                            style: "width: 295px; font: inherit;"
                        });

                    });


                    // RELEASE ENVIRONEMENT ASSOCIATION MUTLI-SELECT (existing application)
                    // Only supported for manual applications
                    if (!item.isAutomatedApplication()) {
                        // Get Environments
                        xhr.get(uReleaseConfig.urls.base + "environment/", {
                            headers: {
                                "Accept": "application/json"
                            },
                            handleAs: "json"
                        }).then(function(data) {
                            //Then build out the multi-select
                            var releaseEnvArray = [];
                            var selectedReleaseEnvArray = item.get("releaseEnvironments");
                            // Iterate through the options and see if it was already in the model
                            // If so, set it to selected
                            array.forEach(data, function(releaseEnv) {
                                var releaseEnvObject = {
                                    label: (releaseEnv.name ? (String(releaseEnv.name)).escape() : i18n("(Error: No release environment)")),
                                    value: releaseEnv.id,
                                    selected: selectedReleaseEnvArray.indexOf(releaseEnv.id) > -1
                                };
                                releaseEnvArray.push(releaseEnvObject);
                            });

                            // Actually add the field
                            form.addField({
                                name: "releaseEnvironments",
                                label: i18n("Release Environments"),
                                type: "CHECKED-MULTI-SELECT",
                                allowedValues: releaseEnvArray,
                                style: "width: 295px; font: inherit;"
                            });
                        });
                    }
                });
            } else {
                xhr.get(uReleaseConfig.urls.base + "teams/", {
                    headers: {
                        "Accept": "application/json"
                    },
                    handleAs: "json"
                }).then(function(data) {
                    var teamArray = [];
                    array.forEach(data, function(team) {
                        var teamObject = {
                            label: (team.name ? (String(team.name)).escape() : i18n("(Error: No team)")),
                            value: team.id
                        };
                        teamArray.push(teamObject);
                    });
                    form.addField({
                        name: "teams",
                        label: i18n("Team"),
                        type: "CHECKED-MULTI-SELECT",
                        required: true,
                        allowedValues: teamArray,
                        style: "width: 295px; font: inherit;"
                    });
                });

                // RELEASE ENVIRONEMENT ASSOCIATION MUTLI-SELECT (new application)
                // Don't need to check for automated flag, because we cannot create automated apps
                // Get Release Envs from the environment end point
                xhr.get(uReleaseConfig.urls.base + "environment/", {
                    headers: {
                        "Accept": "application/json"
                    },
                    handleAs: "json"
                }).then(function(data) {
                    // Then start building out the multi select
                    var releaseEnvArray = [];
                    array.forEach(data, function(releaseEnv) {
                        var releaseEnvObject = {
                            label: (releaseEnv.name ? (String(releaseEnv.name)).escape() : i18n("(Error: No Release Environments)")),
                            value: releaseEnv.id
                        };
                        releaseEnvArray.push(releaseEnvObject);
                    });

                    // Add the field
                    form.addField({
                        name: "releaseEnvironments",
                        label: i18n("Release Environment"),
                        type: "CHECKED-MULTI-SELECT",
                        allowedValues: releaseEnvArray,
                        style: "width: 295px; font: inherit;"
                    });
                });
            }

            form.placeAt(dialog.containerNode);
            dialog.show();
        }
    });
});
