/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
/*global define, window*/
define([
    "dojo/_base/declare",
    "dojo/date/locale",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-style",
    "dojo/on",
    "dojo/mouse",
    "app/widgets/Popup",
    "app/widgets/UReleaseWidget"
], function(
    declare,
    locale,
    domClass,
    domConstruct,
    domStyle,
    on,
    mouse,
    Popup,
    _URWidget
) {
    /**
     * The widget to represent the risk icon (small color circle + text) for the
     * red-yellow-green application status.
     * @class
     * @augments {_URWidget}
     * @lends RiskIcon#
     */
    return declare("app/widgets/icon/RiskIcon", [_URWidget], {
        templateString:
            '<div data-dojo-attach-point="containerAP" class="inline-block risk-icon">' +
                '<div data-dojo-attach-point="badgeAP" class="badge"></div>' +
                '<span data-dojo-attach-point="contentAP" class="content"></span>' +
            '</div>',
        count: null,
        status: {
            red: "RED",
            yellow: "YELLOW",
            green: "GREEN"
        },

        /**
         * Function overriden from _URWidget#postCreate
         */
        postCreate: function() {
            if (!this.value || !this.value.status) {
                throw new Error("Invalid value passed to RiskIcon widget");
            }
            var status = this.value.status.toUpperCase();
            this._determineColorAndMessage(status);

            if (this.value && this.value.username && this.value.date) {
                this._populatePopup();
            }
        },

        /**
         * Helper function for #postCreate to determine what the color of the risk
         * icon should be. Also creates the dom nodes and the supplimental text.
         * @param  {string} status The status of the risk involved, with a value
         *                         of either "red", "yellow", or "green."
         */
        _determineColorAndMessage: function(status) {
            if (status === this.status.red) {
                domClass.add(this.badgeAP, "red");
                this.contentAP.innerHTML = i18n("High");
            } else if (status === this.status.yellow) {
                domClass.add(this.badgeAP, "yellow");
                this.contentAP.innerHTML = i18n("Medium");
            } else if (status === this.status.green) {
                domClass.add(this.badgeAP, "green");
                this.contentAP.innerHTML = i18n("Low");
            }
        },

        /**
         * Helper function for #postCreate to make the popup that has the
         * history information for the involved risk.
         */
        _populatePopup: function() {
            var riskContainer = domConstruct.create("div", {
                className: "inline-block risk-info popup-info-container"
            });

            domConstruct.create("div", {
                className: "inline-block risk-changed-by-info",
                innerHTML: i18n("<span class='bold user-name'> %1 </span><br />%2<br />",
                    this.value.username,
                    locale.format(new Date(this.value.date), {formatLength: "long"}))
            }, riskContainer);

            domConstruct.create("div", {
                className: "separator-bar"
            }, riskContainer);

            domConstruct.create("div", {
                className: "risk-description",
                innerHTML: this.value.description || ""
            }, riskContainer);

            var riskPopup = new Popup({
                attachPoint: this.containerAP,
                contents: riskContainer,
                align: "left",
                textAlign: "left"
            });

            this.own(
                on(this.containerAP, mouse.enter, function() {
                    riskPopup.show();
                }),
                on(this.containerAP, mouse.leave, function() {
                    riskPopup.hide();
                }),
                on(riskPopup, mouse.leave, function() {
                    riskPopup.hide();
                })
            );
        }
    });
});