/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*global define */
define([
        "dojo/_base/declare",
        "dojo/_base/lang",
        "app/widgets/UReleaseWidget",
        "dojo/dom-construct",
        "dojo/dom-class",
        "dojo/dom-style",
        "dojo/on",
        "dojo/mouse",
        "dojo/_base/array"
], function(
        declare,
        lang,
        _UReleaseWidget,
        domConstruct,
        domClass,
        domStyle,
        on,
        mouse,
        array
) {
    return declare( [_UReleaseWidget], {
        templateString: '<div class="icon-picker" />',

        value: null,
        selected: null,

        icons: {},
        iconMasterClass: "",    // Class name to apply to all icons
        iconList: null,         // Format: Array of arrays(Rows) holding icon class names.
                                // [["iconClass", "iconsClass"], [...], [...]]

        colorRowList: null,     // Assign a color to the selected icon box in a row. Default: Blue.
                                // ["red", "orange", "yellow", ...]
        showPreview: true,

        postCreate: function(){
            this._createIconBoxes();
            this.selectIcon(this.value);

            if (this.showPreview){
                this.iconPreview = domConstruct.create("div", {
                    className: "icon-picker-preview inline-block"
                }, this.domNode);
                this.previewIcon();
            }
        },

        /**
         * Creates the icon boxes to choose the colors.
         */
        _createIconBoxes: function(){
            var _this = this;
            var icons = {};

            array.forEach(this.iconList, function(iconRow){
                icons = _this._createIconRow(iconRow, _this.iconMasterClass, _this.colorRowList, _this.domNode);
                _this.icons = lang.mixin(_this.icons, icons);
            });
        },

        /**
         * Create a row of icons
         * @param icons: An array of icon class names.
         * @param iconMasterClass : Class name to apply to all icons in the row.
         * @param iconRowColors : Array of colors for each icon in the row.
         * @param attachPoint: Node to attach the row of icons.
         * @return An object of all the icons in the icon row.
         */
        _createIconRow: function(icons, iconMasterClass, iconRowColors, attachPoint){
            var _this = this;
            var iconRow = {};

            var iconRowContainer = domConstruct.create("div", {
                className: "icon-picker-row"
            }, attachPoint);

            array.forEach(icons, function(icon, i){
                var containerColor = null;
                if (iconRowColors && iconRowColors.length === icons.length){
                    containerColor = iconRowColors[i];
                }
                iconRow[icon] = _this._createIconBox(icon, iconMasterClass, containerColor, iconRowContainer);
            });
            return iconRow;
        },

        /**
         * Create an icon
         * @param iconClass: Class name of the icon.
         * @param additionalClass: Additional class for the icon.
         * @param iconContainerColor : Color of the icon container.
         * @param attachPoint: Node to attach icon to
         * @return The created icon.
         */
        _createIconBox: function(iconClass, additionalClass, iconContainerColor, attachPoint){
            var _this = this;
            var iconContainer = domConstruct.create("div", {
                className: "icon-container inline-block" + (iconContainerColor ? " " + iconContainerColor : "")
            }, attachPoint);

            var iconClassFull = iconClass + (additionalClass ? " " + additionalClass : "");
            var icon = domConstruct.create("div", {
                className: iconClassFull
            }, iconContainer);

            this.own(
                on(iconContainer, "click", function(){
                    _this.selectIcon(iconClass);
                    domClass.add(iconContainer, "selected");
                })
            );
            this.own(
                on(iconContainer, mouse.enter, function(){
                    _this.previewIcon(iconClassFull);
                })
            );
            this.own(
                on(iconContainer, mouse.leave, function(){
                    _this.previewIcon();
                })
            );
            return iconContainer;
        },

        /**
         * Selects Icon
         * @param iconClass: Class name of icon to select.
         */
        selectIcon: function(iconClass){
            if (this.icons){
                if (this.value){
                    var currentSelected = this.icons[this.value];
                    if (currentSelected){
                        domClass.remove(currentSelected, "selected");
                    }
                }
                this.value = iconClass;
                var selected = this.icons[iconClass];
                if (selected){
                    domClass.add(selected, "selected");
                }
            }
        },

        /**
         * Preview icon in preview pane.
         * @param iconClass: Class name of icon to show.
         */
        previewIcon: function(iconClass){
            if (this.iconPreview){
                domConstruct.empty(this.iconPreview);
                if (!iconClass){
                    if (this.value){
                        iconClass = this.value + " " + this.iconMasterClass;
                    }
                    else {
                        iconClass = "blank-icon " + this.iconMasterClass;
                    }
                }
                domConstruct.create("div", {
                    className: iconClass + " " + "large"
                }, this.iconPreview);
            }
        }
    });
});
