/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013, 2015. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals define, window, document, require, i18n, uReleaseConfig*/
define([
    "dojo/_base/declare",
    "dojo/_base/kernel",
    "dojo/_base/lang",
    "dijit/_WidgetBase",
    "dijit/_TemplatedMixin",
    "dojox/widget/Standby",
    "app/model/Resource"
], function (
    declare,
    kernel,
    lang,
    _WidgetBase,
    _TemplatedMixin,
    Standby,
    Resource
) {
    var x = 0;
    /**
     * @returns class UReleaseWidget
     * @description This is the base mixin for all widgets that have a displayed
     * component on the page.  It condenses _WidgetBase and _TemplatedMixin into a single include.
     * It provides a way to show and hide a spinner covering the entire widget, to disable access
     * to it during refreshes.
     * It provides a hook method `_initI18n` that should be overridden to initialize translated
     * text after the i18n method has been initialized.
     * It provides a method that throws an error unless the widget is backed by a * specific
     * resource type.
     * It lays the groundwork for the patch updating system whereby snippets of the dom may be
     * updated without redrawing the widget.
     *
     *
     */
    return declare("app/widgets/UReleaseWidget",
                [_WidgetBase, _TemplatedMixin], {

            templateString: "<div></div>",
            postCreate: function () {
                this._deprecateModelProp();
                this.inherited(arguments);
                this._initI18n();
            },

            /**
             * method used to group activity related to the translation of strings on the widget
             */
            _initI18n: function() {
                // no-op hook
            },

            /**
             * call useStandBy once when generating a widget in order to provide this.standby.show() and this.standby.hide()
             * in order to restrict interaction with the widget and indicate changing state to the user with a spinner.
             */
            useStandby : function () {
                if(!this.standby) {
                    this.standby = new Standby({
                        target:this.domNode,
                        image: uReleaseConfig.urls.staticContent + "images/anim/loading-spinner-small.gif",
                        opacity: 0.35,
                        color: "lightGray"
                    });
                    document.body.appendChild(this.standby.domNode);
                    this.standby.startup();
                } else {
                    console.warn('WARNING: you are attempting to initialize standby multiple times (see: app/widgets/UReleaseWidget:useStandBy)');
                }
            },

            /**
             * deselect anything the user has selected
             */
            deselectCursorText: function () {
                // prevent selecting text on double click.
                if (window.getSelection){
                    window.getSelection().removeAllRanges();
                }
                else if (document.selection) {
                    document.selection.empty();
                }
            },
            /**
             * @param resourceName String. must be the fully qualified name of a resource
             * e.g. 'app/model/scheduledDeployment/ScheduledDeployment' NOT 'ScheduledDeployment' NOT 'Scheduled Deployment'
             */
            mustBeBackedByResource: function (resourceName) {
                var resource = this.resource || this.model;
                if(!resource || !(resource instanceof lang.getObject(resourceName))) {
                    throw new Error(i18n("%s must be backed by a Resource ", this.declaredClass) + (resourceName ? (i18n("of type %s", resourceName)) : ""));
                }
            },


            /**
             * a list of all functions that will be called when the backing resource is updated
             */
            _patchFunctions:null,

            /**
             * a method to add a new function to the list of functions that are executed when a
             * resource is modified
             */
            addPatchFunction: function (ap, domNode, func) {
                if(!this._patchFunctions) {
                    this._patchFunctions = {};
                }
                this._patchFunctions[ap] = {
                    "domNode":domNode,
                    "callback":lang.hitch(this, func)
                };
            },

            /**
             * method that is called when a resource is updated, or when we want to run all patch functions in our `_patchFunctions` list
             */
            patch:function () {
                var patchFunction;

                // if we have any patch functions to run
                if(this._patchFunctions && !util.isEmpty(this._patchFunctions)) {
                    var functionReference;
                    for( functionReference in this._patchFunctions) {
                        if(this._patchFunctions.hasOwnProperty(functionReference)) {
                            patchFunction = this._patchFunctions[functionReference];

                            // if there is no domNode associated with the patch function, and the widget is not being destroyed
                            if(patchFunction.domNode === undefined && !this._beingDestroyed) {
                                // unset the reference to the patch function.
                                patchFunction = undefined;
                            } else {
                                // call the patch function, passing in the domNode specific where the patchFunction was added, a reference to this widget, and a reference to this model
                                patchFunction.callback(patchFunction.domNode, this, this.model || this.resource); // this.model AND/OR this.resource may be null/undefined
                            }
                        }
                    }
                }
            },
            /************************************
             * PRIVATE METHODS
             * *********************************/

            /**
             * @private
             */
            _deprecateModelProp : function () {
                if(!this.resource && this.model instanceof Resource) {
                    this.resource = this.model;
                    kernel.deprecated(i18n('property "this.model"'), i18n("Use this.resource instead"));
                }
            },

            /**
             * clean up the patch functions we are tracking
             */
            destroy:function() {
                var func;
                for(func in this._patchFunctions) {
                    if(this._patchFunctions.hasOwnProperty(func)) {
                        delete this._patchFunctions[func];
                    }
                }
                this.inherited(arguments);
            }
    });
});
