/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
define(["dojo/_base/declare",
        "dojo/_base/array",
        "dojo/_base/lang",
        "dojo/dom-class",
        "dojo/dom-construct",
        "dojo/on",
        "dojo/_base/event",
        "dijit/_WidgetBase",
        "dijit/_TemplatedMixin",
        "dijit/Dialog",
        "dijit/form/Button",
        "js/webext/widgets/ColumnForm",
        "js/webext/widgets/RestSelect",
        "app/model/ResourceRegistry",
        "app/widgets/resourceTreeTable/ResourceTreeTable",
        "app/widgets/ResourceTable",
        "app/widgets/DetailView"],
function (declare, array, lang, domClass, domConstruct, on, event, Widget, TemplatedMixin, Dialog, Button, ColumnForm, RestSelect, Registry, ResourceTreeTable, ResourceTable, DetailView) {

    /**
     *
     * an example extension can be found at OrganizationDetailView, and an implementation can be found on the Organization index.jsp
     *
     * arguments:
     * {
         model : Resource
         *
     *   treeTable : ResourceTreeTable or extension of ResourceTreeTable
     *
     *   childPropertyName : String the name of the property that is an array of child elements
     *
     *   hookAddDialogFields: Function, exposes the ColumnForm so that fields can be added with form.addField() if this is not defined, the form will default to a name and description.
     *   Note that
     *
     *   detailView : an extension of the DetailView class, custonmized to work with the model passed to this TreeDetailView
     *
     *   columns: Array of objects containing name and formatter properties.  See TreeTable::columns
     *
     *   getData: Function
     *
     * }
     */
    return declare([Widget, TemplatedMixin], {

        templateString :
            '<div class="tree-detail-view">' +
                '<div class="inlineBlock tree-table-container">' +
                    '<div class="tree-table-add-button">' +
                        '<div data-dojo-attach-point="addButtonAP"></div>' +
                    '</div>' +
                    '<div class="tree-detail-expand-all" data-dojo-attach-point="treeTableExpandAP"></div>' +
                    '<div data-dojo-attach-point="treeTableContainerAP"></div>' +
                '</div>' +
                '<div class="tree-detail-view-container inline-block" data-dojo-attach-point="detailViewContainerAP"></div>' +
            '</div>',
        model: null,
        hookAddDialogFields: null, // Function
        detailViewCache: {},
        detailViewClass: {}, // Extension of DetailView.js passed in to tell TreeTableView how to create a DetailView

        /**
         * show DetailView for currently selected TreeTable row and hide all others.
         * gets DetailView to show from either cache or generate it.
         */
        showDetailView : function (resource) {
            this._hideDetailViews();
            var view = this._getDetailView(resource);
            view.show();
        },
        /**
         * Destroy and recreate the treetable to update contents
         */
        refresh : function () {
            this._refreshTreeTable();
        },

        /****************************************************************
         * Private Methods
         ***************************************************************/
        postCreate: function () {
            this._generateAddBehavior();
            this._generateExpandBehavior();
            this._generateTreeTable();
        },

        /**
         * Create UI elements for adding items to the TreeTable
         */
        _generateAddBehavior : function () {
            var _this = this;
            var addButton = new Button({
                name: "add",
                label: this.addLabel
            },this.addButtonAP);
            on(addButton, "click", function () {
                _this._generateModifyDialog('add');
            });
        },
        _generateExpandBehavior : function () {
            var _this = this;
            var expandAll = domConstruct.create("a", {
                innerHTML: i18n("Expand All"),
                value: 0
            }, this.treeTableExpandAP);

            on(expandAll, "click", function(resource) {
                if (expandAll.value === 0){
                    array.forEach(_this.treeTable.rowObjectList, function(rowObject) {
                        _this.treeTable.expand(rowObject.item);
                    });
                    expandAll.value = 1;
                    expandAll.innerHTML = i18n("Collapse All");
                }
                else {
                    array.forEach(_this.treeTable.rowObjectList, function(rowObject) {
                        _this.treeTable.collapse(rowObject.item);
                    });
                    expandAll.value = 0;
                    expandAll.innerHTML = i18n("Expand All");
                }

            });

        },
        /**
         * create and show a Dialog associated with the add button's click event
         * action is either "add" or "edit".
         */
        _generateModifyDialog : function (action, resource) {
            var editing = action === 'edit' ? true : false;
            var _this = this;
            if(!this.dialog) {
                this.dialog = new Dialog();
            } else {
                domConstruct.empty(this.dialog.containerNode);
            }

            var form = new ColumnForm({
              submitUrl: this.model.getResourcePath(),
              submitMethod: editing ? "POST" : "PUT",
              submitFormat: "JSON",
              postSubmit: function(data, ioArgs) {
                  _this.refresh();
                  _this.dialog.hide();
              },
              onCancel: function() {
                  _this.dialog.hide();
              }
            });
            if(this.hookAddDialogFields) {
                this.hookAddDialogFields(form, resource, editing);
            } else {
                //default form
              form.addField({
                name: "name",
                label: i18n("Name"),
                required: true,
                type: "Text",
                value: editing ? resource.get('name') : ""
              });

              form.addField({
                name: "description",
                label: i18n("Description"),
                type: "Text Area",
                value: editing ? resource.get('description') : ""
              });
              form.addField({
                name: "id",
                type: "hidden",
                value: resource.get('id')
              });
            }
            form.placeAt(this.dialog.containerNode);
            this.dialog.show();
        },
        /**
         * Refresh the root resource for the TreeTable (not the DetailView), destroy existing TreeTable and build a new one backed by the now updated model.
         */
        _refreshTreeTable : function () {
            var _this = this;
            this.model.load().then(function () {
                if(_this.treeTable) {
                    _this.treeTable.destroyRecursive();
                }
                _this._generateTreeTable();
            });
        },
        /**
         * Create the All TreeTable behavior, including add button, and event for add button
         */
        _generateTreeTable : function () {
            var _this = this,
            editButton,
            delButton;
            //this._generateAddBehavior();
            //this._generateExpandBehavior();

            //define TreeTable column structure
            this.columns = this.columns || [{
                    name: i18n("Name"),
                    formatter: function(item) {
                        var element = domConstruct.create("a",{
                          innerHTML: item.get("name")
                        });
                        return element;
                    }
                },{
                   name: i18n("Description"),
                   formatter: function(item) {
                       return item.get("description");
                   }
                },{
                    name: "",
                    formatter: function(item) {
                        var element = domConstruct.create("div", {style:{"position":"relative",height:"10px",width:"40px",right:"0"}});
                        editButton = domConstruct.create("div", {
                            "innerHTML": "",
                            "class": "edit-button"
                        }, element);
                        delButton = domConstruct.create("div", {
                            "innerHTML": "",
                            "class": "delete-button"
                        }, element);

                        _this.own(
                            on(editButton, "click", function(e) {
                                event.stop(e);
                                _this._generateModifyDialog('edit',item);
                            }),
                            on(delButton, "click", function(e) {
                                event.stop(e);
                                console.warn(i18n('not implemented'));

                        }));
                        return element;
                    }
                }];


            //create TreeTable
            this.treeTable = new ResourceTreeTable({
                hideFooter: true,
                columns: this.columns,
                model:this.model,
                getData: function () {
                    var result = [];

                    array.forEach(_this.model.getMembers(), function(member) {
                        result.push(member);
                    });
                    return result;
                }
            });
            this.treeTable.own(
                on(this.treeTable, "rowSelect", function(resource) {
                    _this.showDetailView(resource);
            }));
            this.treeTable.placeAt(this.treeTableContainerAP);

        },
        /**
         * Load the DetailView from local cache or generate a new one.
         */
        _getDetailView : function (resource) {
            var resourceId = resource.get('id');
            if(!this.detailViewCache[resourceId]) {
                this.detailViewCache[resourceId] = this._generateDetailView(resource);
            }
            return this.detailViewCache[resourceId];
        },
        /**
         * Create a new DetailView
         */
        _generateDetailView : function (resource) {
            var detailView = new this.detailViewClass({
                    model:resource
            });

            detailView.placeAt(this.detailViewContainerAP);
            return detailView;
        },
        /**
         * Hide all DetailViews that are in the cache
         */
        _hideDetailViews : function () {
            var index;
            for (index in this.detailViewCache) {
                if(this.detailViewCache.hasOwnProperty(index)) {
                    var view = this.detailViewCache[index];
                    view.hide();
                }
            }
        }
    });
});
