/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals define, i18n, util, dijit*/
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/html",
    "dojo/on",
    "dojo/hash",
    "dojo/topic",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/query",
    "app/widgets/UReleaseWidget"
], function (
    declare,
    array,
    html,
    on,
    hash,
    topic,
    domClass,
    domConstruct,
    query,
    UReleaseWidget
) {
        /**
         * a Tab Widget for showing and loading tabbed information on a page.
         * Expects properties:
         *  {
                defaultTab : "string" // <== OPTIONAL
         *      configuration: [
         *          {
         *              hash: "string",
         *              label: "string"
         *              populate: function (tabDomNode) {
         *                  new Widget().placeAt(tabDomNode)
         *              },
         *          },{
         *              hash: "string2",
         *              label: "string2"
         *              populate: function (tabDomNode) {
         *                  new Widget().placeAt(tabDomNode)
         *              },
         *          }]
         *
         *  }
         *
         * Supported functions:
         * none external.  In order to programmatically change tabs, change the hash in the browser.
         *
         */
        return declare("app/widgets/TabManager", [UReleaseWidget], {

            configuration : null,
            defaultTab : null,
            cachedTabContent : {},
            tabs : [],
            templateString :
                '<div>' +
                    '<div data-dojo-attach-point="defaultTabNode"></div>' +
                    '<div data-dojo-attach-point="contentNode"></div>' +
                '</div>',

            /**
             *
             */
            postCreate : function() {
                var _this = this;
                this.inherited(arguments);

                if (!this.tabNode) {
                    this.tabNode = this.defaultTabNode;
                }
                else {
                    domClass.add(this.defaultTabNode, "hidden");
                }

                domClass.add(this.domNode, "tab-manager");

                // setup onclick for tabs
                array.forEach(this.configuration, function (tab) {
                    var tabNode = domConstruct.create('a',{'class':tab.hash + " inlineBlock tab","href":"#",'innerHTML':'<span>' + tab.label + '</span>'}, _this.tabNode);
                    _this.tabs.push(tabNode);
                    tab.tabNode = tabNode;
                    on(tabNode, "click", function (e) {
                        if (tab.onClick){
                            tab.onClick();
                        }
                        e.preventDefault();
                        // change hash location
                        window.location.hash = tab.hash;
                    });

                });
                this.own(topic.subscribe("/dojo/hashchange", function(changedHash){
                    var tab = _this.getTabForHash();
                    if(tab) {
                        _this.showTab(tab);
                    }
                }));
                if (window.location.hash === "") {
                   window.location.hash = this.defaultTab;
                }
                //open tab on page load
                var tab = this.getTabForHash();
                if(tab) {
                    _this.showTab(tab);
                }
                array.forEach(this.configuration, function (tab) {
                    _this._loadTab(tab);
                });
            },


            /**
             * returns: a tab from the configuration list or null if none match.
             */
            getTabForHash : function () {
                var t = null;
                array.every(this.configuration, function (tab) {
                    if(window.location.hash === '#' + tab.hash) {
                        t = tab;
                        return false;
                    }
                    return true;
                });
                return t;
            },

            _loadTab : function (tab) {
                if(this.cachedTabContent[tab.hash] === undefined) {
                    var tabDomNode = new domConstruct.create("div",{"class":"container-content hidden"});
                    tab.populate(tabDomNode);
                    domConstruct.place(tabDomNode,this.contentNode);
                    this.cachedTabContent[tab.hash] = tabDomNode;
                }

            },

            /**
             * builds and shows a tab from the cache or from scratch if not in the cache
             */
            showTab : function(tab) {
                //notify the world that the tabs are changing
                topic.publish('TabManager/showTab');
                //Toggle selected tab css
                array.forEach(this.tabs, function (t) {
                    domClass.remove(t,"selected");
                });

                domClass.add(tab.tabNode, "selected");

                // Build tab if it doesn't already exist
                this._loadTab(tab);

                array.forEach(this.contentNode.children,function(child){
                    domClass.add(child, "hidden");
                });
                domClass.remove(this.cachedTabContent[tab.hash],"hidden");


            },

            /**
             * Reloads a tab from a given hash if it exists.
             */
            refreshTab: function(tabHash){
                var refreshTab = null;
                //Find tab from configuration
                array.forEach(this.configuration, function (tab) {
                    if (tabHash === tab.hash){
                        refreshTab = tab;
                    }
                });
                if (refreshTab){
                    //Destroy tab contents and clear from cache, then rebuild.
                    array.forEach(
                        this.findEveryNestedWidget(
                        this.cachedTabContent[refreshTab.hash]),
                        function(widget) {
                            //Destroy every widget that used to exist in the dom
                            widget.destroy();
                    });
                    domConstruct.destroy(this.cachedTabContent[refreshTab.hash]);
                    this.cachedTabContent[refreshTab.hash] = undefined;
                    this._loadTab(refreshTab);
                }
            },

            // XXX: This is a query to find all the widgets given a dom node.
            findEveryNestedWidget: function(initialNode) {
                return query("[widgetid]", initialNode)
                    .map(dijit.byNode)
                    .filter(function(wid) {
                        return wid;
                    }); //filter invalid widget ids that yielded undefined
            }
        }
    );
});
