/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013, 2015. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
define([
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/_base/array",
    "dojo/_base/fx",
    "dojo/fx",
    "dojo/on",
    "dojo/Deferred",
    "dojo/dom-style",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-attr",
    "app/widgets/UReleaseWidget",
    "dijit/form/Form",
    "js/webext/widgets/FormDelegates",
    "dijit/form/Button",
    "js/webext/widgets/Alert",
    "js/webext/widgets/GenericConfirm",
    "app/widgets/FieldValidation",
    "dijit/Tooltip",
    "dojox/string/BidiComplex"
], function (
    declare,
    lang,
    array,
    fx,
    coreFx,
    on,
    Deferred,
    domStyle,
    domClass,
    domConstruct,
    domAttr,
    UReleaseWidget,
    Form,
    Delegates,
    Button,
    Alert,
    Confirm,
    validation,
    Tooltip,
    bidi_stt
) {
        /**
         *
         */
        return declare("app/widgets/ResourceView",
                [UReleaseWidget], {

            /**
             * This templateString may be overridden by child widgets, but shouldn't need to be in most
             * cases. Simply apply classes to the appropriate attach points to customize the appearance
             * of this widget.
             *
             * All attach points must still be present: resourceForm, viewAttach, editAttach, expandAttach
             *
             * Optional Parameters:
             *
             *      selectOnExpand: Override selecting the first field when expanding a row.
             *
             *      doubleClickExpandNode: Override the node that is double-clicked to expand
             *                             the row
             *
             *  Optional parameters of Parent Widget:
             *
             *      parent.textResourceButton: Set true to use text and an icon for edit, show and delete buttons.
             *
             *      parent.noResourceIcon: Set true to just use text edit, show and delete icons
             *
             *      parent.textResourceButtonClassName: Override the class of the text button..
             */
            templateString:
                "<div class=\"resource resource-collapsed\">" +
                    "<div data-dojo-attach-point=\"resourceForm\">" +
                        "<div data-dojo-attach-point=\"expandAttach\"></div>" +
                        "<div class=\"resource-title\" data-dojo-attach-point=\"headerAttach\"></div>" +
                        "<div class=\"resource-view-container\">" +
                            "<div class=\"resource-view\" data-dojo-attach-point=\"viewAttach\"></div>" +
                            "<div class=\"resource-view-actions\" data-dojo-attach-point=\"actionAttach\"></div>" +
                        "</div>" +
                        "<div class=\"resource-edit\" data-dojo-attach-point=\"editAttach\"></div>" +
                        "<div class=\"resource-button-spacer\" data-dojo-attach-point=\"buttonSpacerAttach\"></div>" +
                        "<div class=\"resource-buttons\" data-dojo-attach-point=\"buttonAttach\">" +
                            "<div class=\"save-button inline-block\" data-dojo-attach-point=\"saveButtonAttach\"></div>" +
                            "<div class=\"cancel-button inline-block\" data-dojo-attach-point=\"cancelButtonAttach\"></div>" +
                        "</div>" +
                        "<div class=\"outlet\" data-dojo-attach-point=\"outlet\"></div>" +
                    "</div>" +
                "</div>",

            delegates: undefined,
            expandSpeed: 50,
            deleteSpeed: 300,
            /**
             *
             */
            doubleClickExpandNode:null,
            expanded: false,

            /**
             *
             */
            postCreate: function() {
                this.inherited(arguments);

                // if there is no overridden node on which a user
                // can double click to expand the row, set it to be
                // the whole row
                if(!this.doubleClickExpandNode) {
                    this.doubleClickExpandNode = this.domNode;
                }

                this.validations = [];
                this.extraFieldRows = {};
                this.newModel = false;

                //prepare for Webext forms
                if (!this.delegates) {
                    this.delegates = new Delegates();
                }

                this.setupExpandButtonBehavior();
                this.prepareForm();

                this.initActionButtons();

            },

            initActionButtons: function () {
                this.addPatchFunction('actionButtons',this.actionAttach, lang.hitch(this, '_updateActionButtons'));
            },

            /**
             * Creates the behavior when expanding a row.
             */
            setupExpandButtonBehavior : function () {
                var _this = this;

                if (this.model) {
                    if (this.model.canExpand()) {

                        // show the expand/edit button
                        _this._showExpand();

                        // this.doubleClickExpandNode is set to this.domNode by default,
                        // but can be overridden to be any node.
                        this.own(on(this.doubleClickExpandNode, "dblclick", function(event) {

                            // prevent double-clicking on a nested ResourceView from
                            // also expanding its containing ResourceView

                            event.stopPropagation();
                            // Prevent selecting text on double click.
                            _this.deselectCursorText();

                            // unless double click is disabled, or this is already
                            // expanded, toggle it.
                            if (!_this.disableDoubleClick && !_this.expanded){
                                _this.toggleExpanded();
                            }
                        }));


                    } else { // if !this.model.canExpand()
                        domClass.add(this.expandAttach, "hidden");
                    }

                    // if we are the root in a hierarchy of resources
                    if (!this.model.parent) {
                        // refresh the row when the model is loaded
                        // and the row is collapsed
                        this.model.addChangeListener(function() {
                            if (!_this.expanded) {
                                _this.refresh();
                            }
                        }, this);
                    }

                    // if there is no resource id on the model, this is a newModel
                    this.newModel = !this.model.resourceId;
                    if (this.newModel) {
                        // a new model starts expanded
                        this.toggleExpanded();
                    }
                    //XXX if it's not a new model, refresh the widget?? Eh?
                    else {
                        this.refresh();
                    }

                    if(this.loadExpandedState()) {
                        this.toggleExpanded();
                    }
                }
            },

            /**
             *
             */
            prepareForm : function () {
                var _this = this;
                if(this.form) {
                    this.form.destroy();
                }

                // if we can make changes to this widget
                // replace the resourceForm div with a form
                this.form = new Form({}, this.resourceForm);
                this.own(this.form);

                if(this.model && this.model.canWrite()) {
                    this.own(on(this.form, "submit", function() {
                        _this.onSave();
                        return false;
                    }));
                } else {
                    this.own(on(this.form, "submit", function() {
                        // no-op
                        return false;
                    }));
                }
            },

            /**
             *
             */
            show: function() {
                // XXX it is a huge, huge gap in the UI that we might not
                // have a reference to this attach point.
                // we destroy and rebuild a form that contains elements with their own attach points.  NEVER EVER ^@#$ing do this.
                if(this.viewAttach) {
                    domConstruct.empty(this.viewAttach);
                }
            },

            /**
             *
             */
            _updateActionButtons: function () {
                domConstruct.empty(this.actionAttach);
                this._showExpand();
                this._showDelete();
            },
            /**
             * Creates the Edit/Show button
             */
            _showExpand: function() {
                var _this = this;
                if (this.model.canExpand() && this.model.resourceId) {
                    var showClass = "resource-button show-button";
                    var titleText = this.titleText || i18n("Show");
                    var closeTitle = i18n("Close");
                    if (this.model.canWrite()|| this.isBlankModel) {
                        showClass = "resource-button edit-button";
                        titleText = i18n("Edit");
                    }
                    if (this.containingList && this.containingList.textResourceButton){
                        if (typeof this.containingList.textResourceButton === "string"){
                            titleText += " " + this.containingList.textResourceButton;
                        }
                        if (this.containingList.noResourceIcon) {
                            showClass = "resource-text-button-no-show-icon";
                        }
                        if (this.containingList.textResourceButtonClassName) {
                            showClass += " " + this.containingList.textResourceButtonClassName;
                        }
                        var showButtonContents = '<a class="resource-text-button ' + showClass + '">' + titleText + '</a>';
                        _this.showDiv = domConstruct.create("div", {
                            title: titleText,
                            innerHTML: showButtonContents,
                            className: "resource-text-button-container"
                        }, this.actionAttach);
                    }
                    else {
                        if (this.containingList && this.containingList.resourceButtonClassName) {
                            showClass += " " + this.containingList.textResourceButtonClassName;
                        }
                        _this.showDiv = domConstruct.create("div", {
                            className: showClass,
                            title: titleText
                        }, this.actionAttach);
                    }
                    this.own(on(_this.showDiv, "click", function() {_this.showFunction();}));
                }
            },

            /**
             * Creates the delete button
             */
            _showDelete: function() {
                var _this = this;
                if (this.model.canDelete() && this.model.resourceId) {
                    var deleteDiv;
                    var deleteTitle = this.deleteTitle || i18n("Delete");
                    if (this.containingList && this.containingList.textResourceButton){
                        if (typeof this.containingList.textResourceButton === "string"){
                            deleteTitle += " " + this.containingList.textResourceButton;
                        }
                        var deleteIcon = this.containingList.noResourceIcon ? "resource-text-button-no-delete-icon" : "delete-button";
                        if (this.containingList.textResourceButtonClassName) {
                            deleteIcon += " " + this.containingList.textResourceButtonClassName;
                        }
                        var deleteButtonContents = '<a class="resource-text-button resource-button ' + deleteIcon + '">' + deleteTitle + '</a>';
                        deleteDiv = domConstruct.create("div", {
                            title: deleteTitle,
                            innerHTML: deleteButtonContents,
                            className: "resource-text-button-container"
                        }, this.actionAttach);
                    }
                    else {
                        var deleteClass = "resource-button delete-button";
                        if (this.containingList && this.containingList.resourceButtonClassName) {
                            deleteClass += " " + this.containingList.textResourceButtonClassName;
                        }
                       deleteDiv = domConstruct.create("div", {
                            className: deleteClass,
                            title: deleteTitle
                        }, this.actionAttach);
                    }
                    // XXX TODO remove task from edited list when canceling or deleteing
                    this.own(on(deleteDiv, "click", function() {_this.deleteFunction();}));
                }
            },

            deleteFunction: function() {
                var _this = this;
                _this.confirmDialog = new Confirm({
                    message: i18n("Are you sure you want to delete?"),
                    showUnderField: true,
                    action: function() {
                        _this.onDelete();
                    }
                });
                _this.own(_this.confirmDialog);
            },

            showFunction: function() {
                var _this = this;
                var titleText = this.titleText || i18n("Show");
                var closeTitle = i18n("Close");
                _this.toggleExpanded();
                if (_this.expanded) {
                    domAttr.set(_this.showDiv, "title", closeTitle);
                }
                else {
                    domAttr.set(_this.showDiv, "title", titleText);
                }
            },
            /**
             *
             */
            refresh: function() {
                this.show();
                this._updateActionButtons();
            },

            //--------------------------------------------------------------------------------------
            // EDIT MODE
            //--------------------------------------------------------------------------------------

            /**
             * Expands a row in a collapsed state or collapses an expanded row.
             */
            toggleExpanded: function() {
                this.expanded = !this.expanded;
                // if it has a resourceId because you can't collapse new rows, only cancel them...
                if (this.expanded) {
                    this.showEdit();
                    this.doSelectOnExpand();
                }
                else if (!this.expanded && this.model.resourceId) {
                    this.hideEdit();
                }
                this.saveCurrentState();
            },
            doSelectOnExpand: function () {
                var _this = this;
                var children = this.domNode.firstChild;
                // If specified by widget, select given field on expand.
                if (this.selectOnExpand){
                    if (typeof this.selectOnExpand === "object" && this.selectOnExpand.focusNode){
                        setTimeout(function(){
                            // checking a second time as a super-fast fix, so that if focusNode no longer exists by the time setTimeout is called at the end of the queue, this doesn't freaking blow up.
                            if (typeof this.selectOnExpand === "object" && this.selectOnExpand.focusNode){
                                _this.selectOnExpand.focusNode.focus();
                            }
                        }, 0);
                    }
                    else if (children[this.selectOnExpand] !== undefined && !children[this.selectOnExpand].disabled) {
                        try {
                            children[this.selectOnExpand].focus();
                        } catch(e) {
                            console.error(i18n("nothing to focus on"),e);
                        }
                    }
                }
                // Else, Iterate through domNode and find first field to select on expand.
                else {
                    var i;
                    for (i = 1; i < children.length; i++){
                        if (children[i] !== undefined && !children[i].disabled) {
                            try {
                                children[i].focus();
                            } catch(err) {
                                console.warn(i18n("ignored error"), err);
                            }
                            break;
                        }
                    }
                }
            },
            /**
             * Save the current expanded state of a widget, based on its model ID.
             */
            saveCurrentState : function () {
                var stateId = this._buildStateId();
                if (stateId) {
                    this._createStateObjectIfNeeded(stateId);
                    uReleaseConfig.userInterfaceState[stateId].segmentExpanded = this.segmentExpanded;
                    uReleaseConfig.userInterfaceState[stateId].expanded = this.expanded;
                }
            },
            /**
             * Private utility function for saveCurrentState and loadSavedState
             * @return string
             */
            _buildStateId: function () {
                // for example "Segment--ddcd9de7-150a-4ead-8733-5c0cd74b52be"
                var id = this.model.get('id');
                return id ? this.model.getShortDeclaredClass() + "--" +  this.model.get('id') : null;
            },
            /**
             * Private utility function for saveCurrentState
             */
            _createStateObjectIfNeeded : function (stateId) {
                if(uReleaseConfig.userInterfaceState[stateId] === undefined) {
                    uReleaseConfig.userInterfaceState[stateId] = {};
                }

            },
            /**
             * Load the special expanded state for Segments, used to show their tasks
             * @return Boolean
             */
            loadSegmentExpandedState: function () {
                var stateId = this._buildStateId();
                var ret;
                this._createStateObjectIfNeeded();
                if(uReleaseConfig.userInterfaceState[stateId]) {
                    ret = uReleaseConfig.userInterfaceState[stateId].segmentExpanded;
                }
                return ret;
            },

            /**
             * Load the cached state of the ResourceView.  Expanded true/false
             * @return Boolean
             */
            loadExpandedState: function () {
                var stateId = this._buildStateId();
                var ret;
                this._createStateObjectIfNeeded();
                if(uReleaseConfig.userInterfaceState[stateId]) {
                    ret = uReleaseConfig.userInterfaceState[stateId].expanded;
                }
                return ret;
            },

            /**
             *
             */
            showEdit: function() {
                var _this = this;
                this.validations = [];

                domClass.remove(this.domNode, "resource-collapsed");
                domClass.add(this.domNode, "resource-expanded");
                domConstruct.empty(this.viewAttach);

                if(_this.model.canWrite() || _this.model.isBlankModel) {
                    var saveButton = new Button({
                        type: "submit",
                        label: i18n("Save")
                    });
                    this.own(saveButton);
                    if (!!this.saveButtonAttach) {
                        saveButton.placeAt(this.saveButtonAttach);
                        domClass.add(saveButton.domNode, "idxButtonSpecial");
                    }

                    var cancelButton = new Button({
                        type: "cancel",
                        label: i18n("Cancel"),
                        onClick: function() {
                            _this.toggleExpanded();
                            if (!_this.model.resourceId && !!_this.containingList) {
                                _this.containingList.removeChild(_this);
                            }
                            _this.postCancel();
                        }
                    });
                    this.own(cancelButton);
                    if (!!this.cancelButtonAttach) {
                        cancelButton.placeAt(this.cancelButtonAttach);
                    }

                }
                domStyle.set(this.editAttach, {display:"none"});
                domStyle.set(this.buttonAttach, {display:"none"});
                _this._showDelete();
                _this._showExpand();
                coreFx.wipeIn({
                    node: this.editAttach,
                    duration: _this.expandSpeed
                }).play();
                coreFx.wipeIn({
                    node: this.buttonAttach,
                    duration: _this.expandSpeed
                }).play();
            },

            /**
             * Use delegates to create a widget for a given fieldData object containing a type
             */
            generateFieldWidget: function(fieldData) {
                // If this row is read-only, make sure this field is too.

                var delegate = this.delegates.getDelegate(fieldData.type);
                if (!delegate) {
                    throw new Error(i18n("No delegate found for type %s", fieldData.type));
                }
                // have formDelegates build the field.
                var field = delegate(fieldData);

                // Array of all the types we can have.
                var types = ['Text', 'Validation Text', 'Number', 'NumberBox', 'Secure', 'Enum'];

                // Some widgets should be resized to fit columns.
                if (types.indexOf(fieldData.type) > -1) {
                    var width = "90%";
                    if (fieldData.width !== undefined) {
                        width = fieldData.width;
                    }
                    field.domNode.style.width = width;
                }

                if (fieldData.required && !fieldData.readOnly) {
                    this.addRequiredField(fieldData.name, fieldData.label);
                    domClass.add(field.domNode, "required-field");
                }
                if (fieldData.selectOnExpand){
                    this.selectOnExpand = field;
                }

                if (fieldData.bidiDynamicSTT !== undefined) {
                     bidi_stt.attachInput(field.focusNode, fieldData.bidiDynamicSTT);
                }
                if (fieldData.bidiStaticSTT !== undefined) {
                     field.set("value", bidi_stt.createDisplayString(field.value, fieldData.bidiStaticSTT));
                }
                if (fieldData.textDir !== undefined) {
                     field.set("textDir", fieldData.textDir);
                }

                return field;
            },

            /**
             * Utility function to set up a div for a field in the extra fields section
             */
            createExtraFieldRow: function(field, name, beforeName) {
                var classes = "";

                if (field.fieldRowClass){
                    classes += " " + field.fieldRowClass;
                }

                var result = domConstruct.create("div", {
                    "class": "field-row" + classes
                });

                // Figure out where to insert the new field - either at the end, or before an existing
                // named entry if a beforeName is given which matches something.
                var before;
                if (beforeName) {
                    before = this.extraFieldRows[beforeName];
                }
                if (before) {
                    this.editAttach.insertBefore(result, before);
                }
                else {
                    this.editAttach.appendChild(result);
                }

                classes = "field-label";
                if (field.className){
                    classes += " " + field.className;
                }

                if (field.label){
                    var label = field.label;
                    if (field.required){
                        label = '<span class="required">*</span>' + label;
                    }
                    domConstruct.create("div", {
                        "class": classes,
                        "innerHTML": label
                    }, result);
                }
                else {
                    domConstruct.create("div", {
                        "class": classes
                    }, result);
                }

                classes = "inlineBlock field-widget";
                if (field.className){
                    classes += " " + field.className;
                }

                var fieldDiv = domConstruct.create("div", {
                    "class": classes
                }, result);

                var fieldObject;
                if (field.widget) {
                    fieldObject = field.widget;
                }
                else if (field.domNode) {
                    fieldObject = field.domNode;
                }
                else {
                    fieldObject = this.generateFieldWidget(field);
                }

                if (fieldObject.declaredClass) {
                    fieldObject.placeAt(fieldDiv);
                }
                else {
                    fieldDiv.appendChild(fieldObject);
                }

                if (field.tooltip) {
                    var helpCell = domConstruct.create("div", {
                        "class": "labelsAndValues-helpCell inlineBlock"
                    }, fieldDiv,"last");

                    // helpTip
                    this.own(new Tooltip({
                        connectId: [helpCell],
                        label: field.tooltip,
                        showDelay: 100,
                        position: ["after", "above", "below", "before"]
                    }));
                }
                else if (field.description) {
                    domConstruct.create("div", {
                        "class": "inlineBlock field-description",
                        "innerHTML": field.description
                    }, result);
                }

                if (field.type === "Invisible") {
                    domClass.add(result, "hidden");
                }

                if (name) {
                    this.extraFieldRows[name] = result;
                }
                return result;
            },

            /**
             *
             */
            removeExtraFieldRow: function(name) {
                var fieldRow = this.extraFieldRows[name];
                if (fieldRow) {
                    domConstruct.destroy(fieldRow);
                }
                this.removeFieldValidation(name);
            },

            /**
            *
            */
           hideExtraFieldRow: function(name) {
               var fieldRow = this.extraFieldRows[name];
               if (fieldRow) {
                   domClass.add(fieldRow,"hidden");
               }
           },

           /**
           *
           */
          showExtraFieldRow: function(name) {
              var fieldRow = this.extraFieldRows[name];
              if (fieldRow) {
                  domClass.remove(fieldRow,"hidden");
              }
          },

            /**
             *
             */
            hideEdit: function() {
                if (this && this.editAttach && this.viewAttach && this.buttonAttach){

                    // XXX something is very wrong if we are checking to see if the widget even has dom
                    if (this.domNode){
                        domClass.remove(this.domNode, "resource-expanded");
                        domClass.add(this.domNode, "resource-collapsed");
                    }
                    domConstruct.empty(this.editAttach);
                    domConstruct.empty(this.viewAttach);
                    if (this.saveButtonAttach){
                        domConstruct.empty(this.saveButtonAttach);
                    }
                    if (this.cancelButtonAttach){
                        domConstruct.empty(this.cancelButtonAttach);
                    }
                    this.refresh();
                }
            },

            //--------------------------------------------------------------------------------------
            // FORM SUBMISSION, RELATED BEHAVIOR
            //--------------------------------------------------------------------------------------

            /**
             * Action to take when the form is submitted. By default, this will cause a save of the
             * resource attached to this row.
             */

            onSave: function() {
                var _this = this, propertyName;
                var submittedValues = _this.form.getValues();
                if (_this.validateFields(submittedValues)) {
                    for (propertyName in submittedValues) {
                        if (submittedValues.hasOwnProperty(propertyName)) {
                            var value = submittedValues[propertyName];

                            // This annoying hack will convert results from checkboxes (an array
                            // of one element) to a simple boolean.
                            if (lang.isArray(value)) {
                                if (value.length === 1) {
                                    if (typeof value[0] === "boolean") {
                                        value = value[0];
                                    }
                                    else if(value[0] === "on" || value[0] === "true") {
                                        value = true;
                                    }
                                    else if(value[0] === "off" || value[0] === "false"){
                                        value = false;
                                    }
                                }
                                else if (value.length === 0) {
                                    value = false;
                                }
                            }
                            _this.model.set(propertyName, value);
                        }
                    }
                    _this.preSave();
                    _this.model.save().then(function(response) {
                        _this.postSave(response);
                    },
                    function(error) {
                        _this.postError(error);
                    });
                }
            },

            addRequiredField: function(fieldName, fieldLabel) {
                this.addFieldValidation(fieldName, fieldLabel, validation.required());
            },

            addFieldValidation: function(fieldName, fieldLabel, method) {
                var validation = {"name":fieldName, "label": fieldLabel, "method": method};
                this.validations.push(validation);
            },

            removeFieldValidation: function(fieldName) {
                var _this = this, check;
                var nullFunction = function() {return null;};
                for (check in _this.validations) {
                    if (_this.validations.hasOwnProperty(check)) {
                        if (_this.validations[check].name === fieldName) {
                            _this.validations[check].method = nullFunction;
                        }
                    }
                }
            },

            validateFields: function(submittedValues) {
                var _this = this;
                var validationMessages = [];

                array.forEach(_this.validations, function(validation) {
                    var message = validation.method(submittedValues, validation);
                    if (message) {
                        validationMessages.push(message);
                    }
                });

                var result = true;
                if (validationMessages.length > 0) {
                    this.own(new Alert({
                        messages: validationMessages
                    }));
                    result = false;
                }
                return result;
            },

            /**
             * hook for actions to be taken after the async Resource.save() method is called, and returns successfully.
             * @param response: the payload from the xhr request sent to save the resource
             */
            postSave: function(response) {
                // If we're creating a new object, add it to the parent model if one is given
                if (this.newModel && this.listModel) {
                    // If the response has a relative position defined (index can be zero and even -1)
                    if (!!response && (response.relativePosition !== null) && (response.relativePosition !== undefined)) {
                        this.listModel.addMember(this.model, response.relativePosition.valueOf()-1);
                    }
                    else {
                        this.listModel.addMember(this.model);
                    }
                }


                this.toggleExpanded();
                this.newModel = false;
            },

            /**
             * hook for actions to be taken after the async Resource.save() method is called, and returns unsuccessfully.
             * @param  {Object} error Error passed across from Resource.save()
             * @return {undefined}
             */
            postError: function(error) {
                // If we're creating a new object, add it to the parent model if one is given
                console.error(error);
            },

            /**
             * Optional function to do some dynamic modification of the data to submit when vanilla form
             * submission is not adequate. This should operate on the model in-place.
             */
            preSave: function() {
                // no-op by default
            },


            /**
             * Method invoked after the confirm dialog closes with positive confirmation.
             */
            onDelete: function() {
                var _this = this;
                _this._wipeIt();
            },

            _wipeIt: function(){
                var _this = this;
                domStyle.set(this.domNode, "position", "relative");

                fx.animateProperty({
                    node: this.domNode,
                    duration: this.deleteSpeed,
                    properties: {
                        marginLeft: 600,
                        height: 0,
                        opacity: 0.5
                    }
                }).play();

                setTimeout(function(){
                    _this.model.remove().then(function() {
                    if(_this && _this.domNode) {
                        domStyle.set(_this.domNode, "display", "none");
                    }

                        _this.postDelete();
                    },function(error) {

                        domStyle.set(_this.domNode, "height", "");
                        fx.animateProperty({
                            node: _this.domNode,
                            duration: _this.deleteSpeed,
                            properties: {
                                marginLeft: 0,
                                opacity: 1
                            }
                        }).play();

                        _this.postDeleteError(error);
                    });
                }, _this.deleteSpeed);
            },

            /**
             * Method invoked synchronously after the onDelete async method completes.
             * Can be further chained with other Promises
             */
            postDelete: function() {
                // Remove this model from the original list
                var promise;
                var dfd = new Deferred();
                if (this.listModel) {
                    promise = this.listModel.removeMember(this.model);
                } else {
                    dfd.resolve();
                }
                this.containingList.removeChild(this);

                return promise || dfd.promise;
            },

            /**
             * Method invoked synchronously after the onDelete async method completes with an error.
             * @param error the payload returned from a save request, when it fails (an error)
             */
            postDeleteError: function(/*error*/) {
               //Handle the error in the model row
            },

            /**
             * Method invoked after the user click on the cancel button
             */
            postCancel: function() {

            },

            destroy: function() {
                if(this.refs) {
                    delete this.refs;
                }
                if (this.model) {
                    this.model.removeChangeListeners(this);
                }

                this.inherited(arguments);
            }
        }
    );
});
