/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2011, 2013, 2015. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
define([
    "dojo/_base/declare",
    "dojo/_base/kernel",
    "dojo/_base/array",
    'dojo/_base/lang',
    "dojo/on",
    "dojo/keys",
    "dojo/dom-class",
    "dojo/dom-construct",
    "app/widgets/ResourceList",
    "dijit/form/Form",
    "dijit/form/Button",
    "app/widgets/utilities/PageSelector",
    "js/webext/widgets/Dialog",
    "app/widgets/DateRangePickerDialog"
], function(
    declare,
    kernel,
    array,
    lang,
    on,
    keys,
    domClass,
    domConstruct,
    ResourceList,
    Form,
    Button,
    PageSelector,
    Dialog,
    DateRangePickerDialog
) {
    /**
     * Table variant of ResourceList which shows the members as rows in a table
     * Takes an object following:
     *  {
     *      model: Resource instance containing a collection
     *      widgetClass: The dojo "class" object to instantiate for each member of the collection
     *      showHeaders: boolean, indicates whether to show column headers in the table
     *  }
     */
    return declare("app/widgets/ResourceTable", [ResourceList], {
        templateString: "<div class=\"resource-list resource-table\">" +
            "<div data-dojo-attach-point=\"titleAttach\" class=\"resource-title\"></div>" +
            "<div data-dojo-attach-point=\"paginationAttach\" class=\"resource-pagination\"></div>" +
            "<div data-dojo-attach-point=\"headerAttach\" class=\"resource-header\"></div>" +
            "<div data-dojo-attach-point=\"formAttach\">" +
            "<div data-dojo-attach-point=\"filterAttach\" class=\"resource-filters hidden\"></div>" +
            "</div>" +
            "<div data-dojo-attach-point=\"listAttach\" class=\"resource-rows\"></div>" +
            "<div data-dojo-attach-point=\"paginationAttachBelow\" class=\"resource-pagination\"></div>" +
            "<div data-dojo-attach-point=\"noneAttach\" class=\"resource-list-none hidden\"></div>" +
            "<div class=\"table-actions\" data-dojo-attach-point=\"buttonAttach\"></div>" +
            "</div>",

        // Display a header text in expanded form.
        showHeaders: true,
        // Force show hearders even if there is are no rows.
        forceShowHeaders: null,
        // Force show Filters even if there is are no rows.
        forceShowFilters: null,
        // If you want to enable sort for a specific column, set this value to the column value in the model serviceData.
        sortBy: null,
        // If sortBy value is an object in the serviceData, you can specify what value you want to sort by inside of this object.
        sortByObject: null,
        // If sortByObject value is an array, specify what value to sort by. Will sort using the first member of the array.
        sortByCollection: null,
        // if we should sort on first load based on if columns match (see below)
        _sortOnFirstLoad: false,
        _columnIndexToSortOnFirstLoad: null,
        //
        //pagination properties
        realTimeFilters: false, // disabled for now.
        orderField: null,

        postCreate: function() {
            var _this = this;
            // this filters object is not the same as the resource uses.  This object supports legacy local filtering.
            this.filters = [];
            if (this.model && this.model.get('isPaginated')) {
                this._displayPagination();
                this.useStandby();
            }
            this.inherited(arguments);

            if (this.showHeaders||(!!this.forceShowHeaders)) {
                this.loadHeaders();
            }
            else {
                domClass.add(this.headerAttach, "hidden");
            }
            if (this.model) {
                if (this.model.get('isPaginated')) {
                    this.model.addChangeListener(function() {
                        //TODO.  This should be part of the other PageSelector genre of widgets. Or be its own widget,
                        // not a random function on ResourceTable
                        _this._updatePaginationCount();
                    }, this);
                    if (this._sortOnFirstLoad && this.childWidgets.length !== 0) {
                        // I don't like that we have to get all the columns here, we should look into a better way of doing this in the future
                        this.applySort(this.filterWidgetPrototype.getColumns()[this._columnIndexToSortOnFirstLoad]);
                    }
                }
            }
        },

        loadHeaders: function() {
            var _this = this;
            // TODO update this to use existing widget as template rather than instantiating a widget without a model
            // (I'm hacking a model in here to keep the Resource changes from breaking on this.)
            if (this.model.getMembers()[0]||(!!this.forceShowFilters)) {
                this.createFilterWidget();
            }
            if (this.showHeaders && this.filterWidgetPrototype && !this.headersLoaded) {
                // this filters object is not the same as the resource uses.  This object supports legacy local filtering.
                this.filters = [];
                var i = 0;
                array.forEach(this.filterWidgetPrototype.getColumns(), function(column) {
                    _this.filterWidgetPrototype._buildColumnHeader(column);
                    _this._buildFilters(column);
                    if (column.sortBy === _this.sortBy) {
                        _this._sortOnFirstLoad = true;
                        _this._columnIndexToSortOnFirstLoad = i;
                    }
                    i++;
                });
                this.headersLoaded = true;
            }
        },

        /**
         * build and place the filters in a row under the table header.  Store filter data for queries.
         * See helper methods below this for what they do
         * @private
         */
        _buildFilters: function(column) {
            var _this = this;
            if (column.filterData) {
                var filterData = column.filterData;

                // build the actual filter UI
                var filterContainer = domConstruct.create("div", {
                "class": "resource-filter",
                    "style": {
                        "width": column.width,
                        "minWidth": column.minWidth
                    }
                }, this.filterAttach);
                var filterWrapper = domConstruct.create("div", {
                    "class": "resource-filter-wrapper"
                }, filterContainer);
                domClass.remove(this.filterAttach, "hidden");

                // Filter for type data (text, enum, etc.)
                this._forFilterType(filterData, column, filterWrapper);

                // Used in legacy areas of UCR
                this._forLegacy(column);

            }
            else {
                domConstruct.create("div", {
                    "class": "inlineBlock",
                    "style": {
                        "width": column.width
                    }
                }, this.filterAttach);
            }
        },

        /**
         * Code for different types of fileters to build
         **/

         /**
          * Takes a filter type to decide who to filter the data
          *
          * @param filterDataType - should be passed as filterData.type
          * is the type to decide how the data is filtered.
          * @private
          **/
         _forFilterType: function(filterData, column, placePoint) {
            var _this = this;
            var type = filterData.type;

            var filterMap = {
                //We hitch the functions so we keep context when they are called
                'Select': lang.hitch(this, this._filterOnStatus),
                'Text': lang.hitch(this, this._filterOnText),
                "DateRange": lang.hitch(this, this._filterOnDateRange)
            };

             function createGetInputCallBack(inputDom) {
                 return function() {
                     return inputDom.value;
                 };
             }

             function generateDateRangeFilterCallBack(filterForm, filterWidget, startDateRangePicker,endDateRangePicker, startDateRangeDisplay, endDateRangeDisplay) {
                  function result() {
                    var dateRangePickerDialog = new DateRangePickerDialog({
                        startDateRangePicker:startDateRangePicker,
                        endDateRangePicker: endDateRangePicker,
                        filterWidget: filterWidget,
                        filterForm: filterForm
                    });
                    on(dateRangePickerDialog,"hide", function() {
                        startDateRangeDisplay.innerHTML="";
                        endDateRangeDisplay.innerHTML="";
                        if(!!startDateRangePicker.value) {
                            startDateRangeDisplay.innerHTML= new Date(Number(startDateRangePicker.value)).toLocaleDateString();
                        }
                        if(!!endDateRangePicker.value) {
                            endDateRangeDisplay.innerHTML= new Date(Number(endDateRangePicker.value)).toLocaleDateString();
                        }
                    });
                    dateRangePickerDialog.show();
                  }
                return result;
            }

            if (type || column.type) {
                filterData.name = column.name;

                if (this.filterWidgetPrototype) {
                    var filterWidget;
                    if(type.toUpperCase() === "DATERANGE") {
                        filterData.type = "Text";
                        filterWidget = this.filterWidgetPrototype.generateFieldWidget(filterData);
                        filterData.type = type;
                        domConstruct.empty(filterWidget.domNode);

                        var dateRangePicker = domConstruct.create("div",{
                            "class":"data-range-picker-filter"
                        },filterWidget.domNode);
                        var startDateRangePicker = domConstruct.create("input",{
                            "type":"hidden",
                            "PlaceHolder":"Start"
                        }, dateRangePicker);
                        var endDateRangePicker = domConstruct.create("input",{
                            "type":"hidden",
                            "placeholder":"End"
                        }, dateRangePicker);

                        var startDateRangeDisplayContainer = domConstruct.create("div",{
                            "class":"date-start-filter-display-data-container"
                        }, dateRangePicker);

                        var startDateRangeDisplay = domConstruct.create("span",{
                            "class":"date-start-filter-display"
                        }, startDateRangeDisplayContainer);

                        var toText = domConstruct.create("div",{
                            "class":"date-start-filter-display-data-container",
                            "innerHTML":"&mdash;"
                        }, dateRangePicker);

                        var endDateRangeDisplayConainer = domConstruct.create("div",{
                            "class":"date-end-filter-display-data-container"
                        }, dateRangePicker);

                        var endDateRangeDisplay = domConstruct.create("span",{
                            "class":"date-end-filter-display"
                        }, endDateRangeDisplayConainer);
                        if(!!filterData.getDefaults) {
                            var defaultStringValues = filterData.getDefaults();
                            startDateRangePicker.value = defaultStringValues[0];
                            endDateRangePicker.value = defaultStringValues[1];
                            startDateRangeDisplay.innerHTML = new Date(Number(startDateRangePicker.value)).toLocaleDateString();
                            endDateRangeDisplay.innerHTML = new Date(Number(endDateRangePicker.value)).toLocaleDateString();
                        }

                        filterWidget.set("startValue", createGetInputCallBack(startDateRangePicker));
                        filterWidget.set("endValue", createGetInputCallBack(endDateRangePicker));
                        on(dateRangePicker, "click", generateDateRangeFilterCallBack(this.filterForm, filterWidget, startDateRangePicker, endDateRangePicker, startDateRangeDisplay, endDateRangeDisplay));
                    }
                    else {
                        filterWidget = this.filterWidgetPrototype.generateFieldWidget(filterData);
                    }

                    this.filters.push(filterWidget);

                    // If this is a checkbox filter then select all of the rows in this table
                    if(type === "CheckBox") {
                        filterWidget.onChange = function() {
                                _this.selectAll(filterWidget.getValue());
                        };
                    }

                    // if we have a model to filter, and the model is
                    // paginated. We store the old onChange function if
                    // there is one and run it so as to not override.
                    if (this.model && this.model.get('isPaginated')) {
                        var oldOnChange = filterWidget.onChange;
                        filterWidget.onChange = function() {
                            var value = filterWidget.get('value');
                            if (filterWidget.hasOwnProperty('onChange')) {
                                oldOnChange(value);
                            }

                            /**
                             * Filter for specific types by methods
                             * EMPTY, NOTSTARTED, INPROGRESS, SOFTFAILED, FAILED, COMPLETE, ABORTED;
                             * These will be statically set till the endpoints are ready
                             *
                             * Field is the status
                             * type is eq
                             * class Enum
                             * value is filterWidget.Get('value')
                             **/
                            if (filterMap.hasOwnProperty(type)) {
                                filterMap[type](filterData, filterWidget);
                            }

                            if (this.doFiltersTimeout) {
                                clearTimeout(this.doFiltersTimeout);
                            }
                            if (_this.realTimeFilters) {
                                this.doFiltersTimeout = setTimeout(function() {
                                    _this.model.load();
                                }, 500);
                            }
                        };
                        filterWidget.own(on(filterWidget, "keyup", filterWidget.onChange));
                    }

                    filterWidget.placeAt(placePoint);
                }
            }
         },

         /**
          * Allows checking in places were we don't use pagination and just
          * have very long lists of items
          *
          * @param column - the item being passed in to be checked
          * @private
          **/
         _forLegacy: function(column) {
            var _this = this;
            if (!column.filterData.onChange) {
                if (column.filterData.type && column.filterData.type.toUpperCase() === "CHECKBOX") {
                    column.filterData.onChange = function(value) {
                        _this.selectAll(value);
                    };
                }
                else {
                    if (!this.usingPagination) {
                        _this.applyFilters();
                    }
                }
            }
         },


        /**
         * Filter on a enumerated status drop down. If we ask for ANY we should
         * get back all the list items
         *
         * @param filterData The filter data being passed in
         * @param filterWidget The widget that contains the value being passed in
         * @private
         **/
        _filterOnStatus: function(filterData, filterWidget) {
            var _this = this;
            var selectedValue = filterWidget.get('value');

            if (selectedValue === 'ANY') {
                //Return every item
                _this.model.filter({
                    'field': filterData.filterField,
                    'type': 'eq',
                    'class': filterData['class']
                });
            }
            else {
                _this.model.filter({
                    'field': filterData.filterField,
                    'type': 'eq',
                    'class': filterData['class'],
                    'value': selectedValue
                });
            }

            // This is hacky. It forces the form to call the
            // on(this.filterForm, "submit", function() {
            // in createFilterWidget. The dropdown doesn't allow
            // for the method and we don't want to modify the prototype
            this.filterForm.submit();
         },

        /**
         * Filter on text
         *
         * @param filterData The filter data being passed in
         * @param filterWidget The widget that contains the value being passed in
         * @private
         **/
        _filterOnText: function(filterData, filterWidget) {
            var _this = this;
            _this.model.filter({
                'field': filterData.filterField,
                'type': 'like',
                'class': filterData['class'],
                'value': filterWidget.get('value')
            });
         },
        _filterOnDateRange: function(filterData, filterWidget) {
            var _this = this;
            var startValue = filterWidget.get('startValue')();
            var endValue = filterWidget.get('endValue')();
            if(startValue !== endValue) {
                if(!!endValue && !!startValue) {
                    _this.model.filter({
                        'field': filterData.filterField,
                        'type': "range",
                        'class': "Long",
                        'value': filterWidget.get('startValue')(),
                        'isMultiple': true,
                        'getMultiValues': function() {
                            var result = [];
                            result.push(endValue);
                            result.push(startValue);
                            return result;
                        }
                    });
                }
                else {
                    if(!!endValue) {
                        _this.model.filter({
                            'field': filterData.filterField,
                            'type': 'le',
                            'class': "Long",
                            'value': endValue
                        });
                    } else {
                        if(!!startValue) {
                            _this.model.filter({
                                'field': filterData.filterField,
                                'type': 'ge',
                                'class': "Long",
                                'value': startValue
                            });
                        }
                    }
                }
            } else {
                var oneDayInMs = 86400000;
                endValue = Number(startValue)+oneDayInMs;
                _this.model.filter({
                    'field': filterData.filterField,
                    'type': "range",
                    'class': "Long",
                    'value': filterWidget.get('startValue')(),
                    'isMultiple': true,
                    'getMultiValues': function() {
                        var result = [];
                        result.push(endValue);
                        result.push(startValue);
                        return result;
                    }
                });
            }
         },

        /**
         * @see ResourceRow
         */
        show: function() {
            this.inherited(arguments);
            if (this.showHeaders && !this.headersLoaded) {
                this.loadHeaders();
            }

            if (this.model && !this.model.get('isPaginated')) {
                this.applyFilters();
            }
            this.afterLoad();
        },

        /**
         * method is overriden by a function that is executed after the widget loads
         */
        afterLoad: function() {
        },

        /**
         * calculate and show pagination information to the user.
         */
        _displayPagination: function() {
            var _this = this;
            _this.pagination = domConstruct.create("div", {
                className: "pagination-summary pagination-label inline-block"
            }, this.paginationAttach);

            var rowsPerPageContainer = domConstruct.create("div", {
                "className": "pagination-rows-per-page inline-block"
            }, this.paginationAttach);

            var rowsPerPageLabel = domConstruct.create("div", {
                "innerHTML": i18n("Rows Per Page:"),
                "className": "pagination-label inline-block"
            }, rowsPerPageContainer);

            // this is terrible, terrible reuse of a widget.  These two widgets look
            // similar but don't really do the same thing.
            var itemsPerPageSelector = new PageSelector({
                defaultValue: 10,
                className: "inline-block",
                options: [{
                    label: "10",
                    value: 10
                }, {
                    label: "25",
                    value: 25
                }, {
                    label: "50",
                    value: 50
                }, {
                    label: "100",
                    value: 100
                }],
                onClick: function(value) {
                    // Add spinner while we expand the table
                    _this.standby.show();
                    _this.model.set('itemsPerPage', value);
                    _this.model.set('pageNumber', 1);
                    _this.model.load().then( function() {
                        _this.standby.hide();
                    });
                }
            });
            itemsPerPageSelector.placeAt(rowsPerPageContainer);

            var paginationPageSelectorContainer = domConstruct.create("div", {
                "className": "pagination-page-selector"
            }, this.paginationAttach);

            this.pageSelector = new PageSelector({
                numberOfPages: 1,
                model: this.model,
                onClick: function(value) {
                    _this.standby.show();
                    _this.model.set('pageNumber', value);
                    _this.model.load().then( function() {
                        _this.standby.hide();
                    });
                }
            });
            this.pageSelector.placeAt(paginationPageSelectorContainer);
            _this._updatePaginationCount();
        },


        /**
         * update the UI to reflect the count and current page returned from the query
         * TODO: refactor this to be a widget that goes along side PageSelector.  Have it take
         * the collection model and get it's updated info directly.
         * @private
         */
        _updatePaginationCount: function() {
            this.pageSelector.refresh(this.model.get('pageNumber'), +this.model.get('totalPages'));
            var pageResult = i18n("Page: %1 / %2", this.model.get('pageNumber'), this.model.get('totalPages'));
            var itemResult = i18n("%1 to %2 of %3 items",
                this.model.get('rangeStart'),
                this.model.get('rangeEnd'),
                this.model.get('totalRecords'));
            this.pagination.innerHTML = itemResult + ", " + pageResult;
        },

        /**
         * select all child widget's checkBoxes, if any.
         */
        selectAll: function(value) {
            var _this = this;
            kernel.deprecated("app/widgets/ResourceTable.selectAll()", "Use the selectAll and related methods in CheckboxResourceTable");
            array.forEach(this.childWidgets, function(childWidget) {
                if (!domClass.contains(childWidget.domNode, "does-not-match-filter")) {
                    childWidget.setChecked(value);
                    if(_this.dnd) {
                        childWidget.model.selectedCheckBox = value;
                    }
                }
            });
        },

        createFilterWidget: function() {
            var _this = this;
            this.filterWidgetPrototype = new this.widgetClass({
                model: this.model.getMembers()[0],
                containingList: this
            });
            this.filterForm = new Form({}, this.formAttach);
            this.own(
                on(this.filterForm, "submit", function() {
                    if (_this.model && _this.model.get('isPaginated')) {
                        //Server side Filter
                        _this.standby.show();
                        _this.pageNumber = 1;
                        _this.model.load().then( function() {
                            _this.standby.hide();
                        });
                    }
                    else {
                        //Client side Filter
                        //if client side filtering, no spinner is needed
                        var filterData = {};
                        array.forEach(_this.filters, function(filterObject) {
                            filterData[filterObject.get("name")] = filterObject.get("value");
                        });
                        array.forEach(_this.childWidgets, function(childWidget) {
                            if (childWidget.matchesFilters && childWidget.matchesFilters(filterData)) {
                                domClass.remove(childWidget.domNode, "does-not-match-filter");
                            }
                            else {
                                domClass.add(childWidget.domNode, "does-not-match-filter");
                            }
                        });
                    }
                    return false;
                }),

                on(this.filterForm, "keypress", function(evt) {
                    var keyCode = evt.charCode || evt.keyCode;
                    if (keyCode === keys.ENTER) {
                        _this.applyFilters();
                        return false;
                    }
                })
            );

            // the below check should not have to be made.  filterAttach
            // should always be available as an attach point, but for some
            // reason, something non-obvious is blowing it away.  Until
            // them, we will prevent to UI from trying to attach buttons
            // to a filter that doesn't exist, so that it doesn't break
            // other refresh behaviors of the UI by bringing js to a halt
            if (this.filterAttach) {
                var filterButton = new Button({
                    type: "submit",
                    label: i18n("Apply")
                });
                filterButton.placeAt(this.filterAttach);
                domClass.add(filterButton.domNode, "filter-submit");
            }
        },

        applyFilters: function() {
            var filterData = {};

            if (this.model && !this.model.get('isPaginated')) {
                // only do local filtering if we are not paginating results.
                array.forEach(this.childWidgets, function(childWidget) {
                    if (childWidget.matchesFilters && childWidget.matchesFilters(filterData)) {
                        domClass.remove(childWidget.domNode, "does-not-match-filter");
                    }
                    else {
                        domClass.add(childWidget.domNode, "does-not-match-filter");
                    }
                });
            }
        },

        applySort: function(column, sortStatus) {
            var _this = this;
            this.sortColumn = column;
            this.sortStatus = sortStatus;
            var customSort = column.useCustomSort;
            var sortColumn = column.sortBy;
            var sortObject = column.sortByObject;
            var sortCollection = column.sortByCollection;
            if (this.model && this.model.get('isPaginated')) {
                _this.standby.show();
                this.model.sort(column.sortBy)
                    .load().then(function() {
                        _this.standby.hide();
                    });
            }
            else {
                if (customSort) {
                    _this.customSort(column, sortStatus);
                }
                else {
                    this.childWidgets.sort(function(object1, object2) {
                        var result = 0;
                        if (object1 !== undefined && object2 !== undefined) {
                            var sortColumn1 = object1.model.get(sortColumn);
                            var sortColumn2 = object2.model.get(sortColumn);
                            if (sortObject) {
                                if (sortColumn1 === undefined) {
                                    result = 1;
                                }
                                else if (sortColumn2 === undefined) {
                                    result = -1;
                                }
                                else if (sortColumn1.get(sortObject) < sortColumn2.get(sortObject)) {
                                    result = -1;
                                }
                                else if (sortColumn1.get(sortObject) > sortColumn2.get(sortObject)) {
                                    result = 1;
                                }
                            }
                            else if (sortCollection) {
                                if (sortColumn1.getMembers()[0] === undefined) {
                                    result = 1;
                                }
                                else if (sortColumn2.getMembers()[0] === undefined) {
                                    result = -1;
                                }
                                else if (sortColumn1.getMembers()[0].get(sortCollection) < sortColumn2.getMembers()[0].get(sortCollection)) {
                                    result = -1;
                                }
                                else if (sortColumn1.getMembers()[0].get(sortCollection) > sortColumn2.getMembers()[0].get(sortCollection)) {
                                    result = 1;
                                }
                            }
                            else {
                                if (sortColumn1 < sortColumn2) {
                                    result = -1;
                                }
                                else if (sortColumn1 > sortColumn2) {
                                    result = 1;
                                }
                            }
                        }
                        if (sortStatus) {
                            result *= -1;
                        }
                        return result;
                    });
                    array.forEach(this.childWidgets, function(member) {
                        member.placeAt(_this.listAttach);
                    });
                }
            }
        },

        resort: function() {
            if (this.model.get('isPaginated') || this.sortColumn) {
                this.applySort(this.sortColumn, this.sortStatus);
            }
        },

        removeChild: function( /*child*/ ) {
            if (this.model && this.model.get('isPaginated')) {
                this.model.load();
            }
            else {
                this.inherited(arguments);
            }
        },
        destroy: function() {
            if (this.filterWidgetPrototype && this.filterWidgetPrototype.destroy) {
                this.filterWidgetPrototype.destroy();
            }
            this.inherited(arguments);
        }
    });
});
