/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
/*globals define, document*/
define([
        "dojo/_base/declare",
        "dojo/on",
        "dojo/_base/array",
        "dojo/dom-class",
        "dojo/dom-style",
        "dojo/dom-construct",
        "app/widgets/ResourceView",
        "app/widgets/mixins/_MemoryCacheMixin"
    ],
    function(
        declare,
        on,
        array,
        domClass,
        domStyle,
        domConstruct,
        ResourceView,
        _MemoryCacheMixin
    ) {
        /**
         * A base class for resource objects which will be represented as a row in a list of divs.
         * Provides a basic framework with various functions to be overridden by subclasses.
         *
         * Expects properties:
         *  {
         *      model: Resource             a Resource instance for the object represented by this row
         *      listModel: Resource         the Resource instance representing the collection this resource belongs to.
         *      containingList :            the containing widget (ResourceList or ResourceTable)
         *  }
         *
         * Supported functions:
         *  getColumns: Return an array of column data objects. Each column object supports the following:
         *  {
         *      name: String            Row heading for this column (1)
         *      width: String           Value to use for CSS 'width' property for this column
         *      formatter: function     Return a string, DOM node, or widget to insert into the cell for
         *                              this column.
         *
         *      filterData: {           An object used to represent the filter fields over headers,
         *                              and give  information to ResourceTable to help it filter against the DB
         *
         *          name: String        derived from the name property above (1)
         *
         *          type: String        The field type (Text, Checkbox).  What kind of widget the user uses to
         *                              filter
         *
         *          filterField: String the relationship of the column being filtered to the database table in HQL.  For example:
         *                              filtering Release name on a ChangeRow, there is no name for Release in the change DAO
         *                              filterField would be "release.name", allowing the query to traverse the release relationship
         *                              to get the release id from its own object.
         *      }
         *      getValue: function      Return a text value for this cell - used as display if no formatter
         *                              is given. Also used as the default value for edit mode.
         *      fieldData: {            Data for generating the edit-mode of this column
         *          getField / Function A function to return the widget or DOM node to show for edit mode
         *              - or -
         *          type / String       Type of the field. See FormDelegates.js for built-in types.
         *          ...and any properties documented by FormDelegates.js
         *      }
         *  }
         *
         *
         * REFERENCE IMPLEMENTATION: widgets/integration/IntegrationProvider.js
         */
        return declare("app/widgets/ResourceRow", [ResourceView, _MemoryCacheMixin], {

            /**
             *
             */
            postCreate: function() {
                this.inherited(arguments);
                domClass.add(this.domNode, "resource-row");
            },

            /**
             *
             */
            show: function() {
                this.inherited(arguments);
                var _this = this;
                array.forEach(this.getColumns(), function(column) {
                    if(column.useWidget) {
                        domConstruct.place(_this.showAsWidget(column), _this.viewAttach);
                    } else {
                        domConstruct.place(_this.buildColumn(column), _this.viewAttach);
                    }

                });
            },

            /**
            * Show in the row as a widget instead of just a cell with a value.
            */
            showAsWidget: function(column) {
                var _this = this;
                var fieldData = column.fieldData;

                var additionalClasses = column["class"] || column.className || "";
                var cellDom = domConstruct.create("div", {
                    "class": "inline-block resource-row-cell " + additionalClasses
                }, _this.viewAttach);
                if (fieldData) {

                    domStyle.set(cellDom, {
                        "width": column.width,
                        "minWidth": column.minWidth
                    });
                    if ((_this.newModel && fieldData.label) || fieldData.alwaysShowLabel) {
                        var label = fieldData.label;
                        if (fieldData.required) {
                            label = '<span class="required">*</span>' + label;
                        }
                        domConstruct.create("div", {
                            "class": "resource-label",
                            "innerHTML": label
                        }, cellDom);
                    }

                    var initialValue = "";

                    // if there's a model to query, and a method in place, use it.
                    if (_this.model && column.getValue) {
                        initialValue = column.getValue();
                    }

                    var field;
                    if (fieldData.getField !== undefined) {
                        field = fieldData.getField();
                    } else {
                        fieldData.value = fieldData.value || initialValue;
                        field = _this.generateFieldWidget(fieldData);
                    }

                    if (field.declaredClass) {
                        field.placeAt(cellDom);
                    } else {
                        domConstruct.place(field, cellDom);
                    }
                }
                return cellDom;
            },
            /**
             *
             */
            buildColumn: function(column) {
                var _this = this;
                //
                // add classes
                var additionalClasses = column["class"] || column.className || "";
                var cellDom = domConstruct.create("div", {
                    "class": "inline-block resource-row-cell " + additionalClasses,
                    "style": {
                        "width": column.width,
                        "minWidth": column.minWidth
                    }
                });

                var result = null;
                if (_this.model) {
                    if (column.formatter) {
                        // Not Escaped
                        result = column.formatter(cellDom);
                    }
                    else if (column.getValue) {
                        var value = column.getValue();

                        // we want to keep 0 or ""
                        if (value === undefined || value === null) {
                            value = "";
                        }
                        // Always escaped
                        result = String(value).escape();
                    }
                }

                // Handle the type of the result of the formatter, and use the appropriate
                // method to add the result to the cell.
                if (result !== null) {
                    if (!!result && result.placeAt) { // if the thing is a Widget
                        result.placeAt(cellDom);
                    }
                    else if ((typeof result === "object") && (result.nodeType === 1) &&
                        (typeof result.style === "object") && (typeof result.ownerDocument === "object")) {
                        cellDom.appendChild(result);
                    }
                    else if (result !== undefined && typeof result !== "object") {
                        cellDom.innerHTML = util.applyBTD(result);
                    }
                }

                return cellDom;
            },

            /**
             *
             */
            _buildColumnHeader: function(column) {
                var _this = this;
                this.headersLoaded = true;
                var columnHeader = domConstruct.create("div", {
                    "innerHTML": column.name,
                    "style": {
                        "width": column.width,
                        "minWidth": column.minWidth
                    },
                    "class": "resource-header-cell"
                }, this.containingList.headerAttach);
                // When having sorting columns, you can apply styling to columns that do not sort.
                if (column.styleAsSort) {
                    columnHeader.className += " sorting-header sort-unhighlighted no-sort-link";
                }
                if (column.sortBy) {
                    columnHeader.className += " sorting-header sort-unhighlighted";
                    var columnHeaderSorter = domConstruct.create("div", {
                        "class": "inlineBlock sort-button sort-display",
                        "value": 1
                    }, columnHeader);
                    // TODO put sort behavior trigger in ResourceTable
                    this.own(on(columnHeader, "click", function() {
                        // document.getElementsByClassName is undefined in IE7 & IE8, specify function.
                        // TODO stop getting all the elements.  Use a more targeted means.
                        if (!document.getElementsByClassName) {
                            document.getElementsByClassName = function(className) {
                                var allTags = document.getElementsByTagName('*'),
                                    allClassNames = [];
                                array.forEach(allTags, function(tag) {
                                    if (tag.className === className) {
                                        allClassNames[allClassNames.length] = tag;
                                    }
                                });
                                return allClassNames;
                            };
                        }
                        array.forEach(document.getElementsByClassName("inlineBlock sort-button sort-down"), function(button) {
                            button.className = "inlineBlock sort-button sort-display";
                        });
                        array.forEach(document.getElementsByClassName("inlineBlock sort-button sort-up"), function(button) {
                            button.className = "inlineBlock sort-button sort-display";
                        });
                        array.forEach(document.getElementsByClassName("inlineBlock resource-header-cell sorting-header sort-highlighted"), function(header) {
                            header.className = "resource-header-cell sorting-header sort-unhighlighted";
                        });
                        if (columnHeaderSorter.value === 0) {
                            columnHeaderSorter.className = "inlineBlock sort-button sort-down";
                            columnHeaderSorter.value = 1;
                            _this.containingList.applySort(column, true);
                        } else if (columnHeaderSorter.value === 1) {
                            columnHeaderSorter.className = "inlineBlock sort-button sort-up";
                            columnHeaderSorter.value = 0;
                            _this.containingList.applySort(column, false);
                        }
                        columnHeader.className = "resource-header-cell sorting-header sort-highlighted";
                    }));
                }

            },

            /**
             *
             */
            setInnerHtml: function(result) {
                return String(result).escape().replace(/\n/g, "<br />");
            },

            /**
             * Must be overridden to return an array of objects representing the column headings, size, etc.
             *
             * Each column object can include:
             *  {
             *      name: Name to show in header
             *      width: How big to make this column
             *      getValue: Optional function to determine the text value of this column
             *      formatter: Optional function to return a widget/dom node to put in the cell for display
             *                 purposes.  note, formatter will be called IN PLACE OF getValue, if both are defined.
             *      fieldData: { (include this object if the column is to be editable)
             *          name: Name of the property for this field
             *          type: (currently ignored) Input type for this field
             *      }
             *  }
             */
            getColumns: function() {
                return [];
            },

            /**
             * title: Title to display when creating new resource rows.
             * separator: Display a separator between the newly row and extra field rows.
             */
            title: null,
            separator: false,

            //--------------------------------------------------------------------------------------
            // EDIT MODE
            //--------------------------------------------------------------------------------------
            /**
             *
             */
            showEdit: function() {
                this.inherited(arguments);
                var _this = this;
                if (this.separator) {
                    domClass.add(this.domNode, "resource-row separator");
                }

                if (this.newModel && this.title) {
                    domConstruct.create("h2", {
                        "class": "new-resource-header",
                        "innerHTML": this.title
                    }, _this.viewAttach);
                }

                array.forEach(this.getColumns(), function(column) {
                    var fieldData = column.fieldData;
                    if (fieldData) {
                        if (!(fieldData instanceof Array)) {
                            fieldData = [fieldData];
                        }

                        var additionalClasses = column["class"] || column.className || "";
                        var cellDom = domConstruct.create("div", {
                            "class": "inline-block resource-row-cell " + additionalClasses
                        }, _this.viewAttach);

                        domStyle.set(cellDom, {
                            "width": column.width,
                            "minWidth": column.minWidth
                        });

                        array.forEach(fieldData, function(data) {
                            if ((_this.newModel && data.label) || data.alwaysShowLabel) {
                                var label = data.label;
                                if (data.required) {
                                    label = '<span class="required">*</span>' + label;
                                }
                                domConstruct.create("div", {
                                    "class": "resource-label",
                                    "innerHTML": label
                                }, cellDom);
                            }

                            var initialValue = "";

                            // if there's a model to query, and a method in place, use it.
                            if (_this.model && column.getValue) {
                                initialValue = column.getValue();
                            }

                            var field;
                            if (data.getField !== undefined) {
                                field = data.getField();
                            } else {
                                data.value = data.value || initialValue;
                                field = _this.generateFieldWidget(data);
                            }

                            if (field.declaredClass) {
                                field.placeAt(cellDom);
                            } else {
                                domConstruct.place(field, cellDom);
                            }
                        });
                    } else {
                        domConstruct.place(_this.buildColumn(column), _this.viewAttach);
                    }
                });
                this.showExtraForm();
            },

            /**
             * Optional function to be overridden to populate the form with additional elements when not
             * all editable values are shown as "cells".
             *
             * This method should:   add elements to this.editAttach.
             *                       show a read only and normal view based on the state of this.model.canWrite()
             */
            showExtraForm: function() {
                // no-op by default
            },

            /**
             * handle drawin the header when an item is added to a list
             * @see app/widgets/ResourceView
             */
            postSave: function() {
                //Load table headers if list was previously empty.
                this.inherited(arguments);

                //Update table if using pagination.
                if (this.containingList.model.get('isPaginated')) {
                    this.model.load();
                }
            },

            setChecked: function(value) {
                var _this = this;
                _this.checkBox.set("checked", value);
            },

            /**
             *
             */
            matchesFilters: function(filterValues) {
                var result = true;
                array.forEach(this.getColumns(), function(column) {
                    if (column.filterData && column.getValue) {
                        var value = column.getValue();
                        var filterValue = filterValues[column.name];
                        var filterType = column.filterData.type;
                        if (value &&
                            filterValue &&
                            filterType === "Text" &&
                            value.toLowerCase().indexOf(filterValue.toLowerCase()) === -1) {
                            result = false;
                        }
                    }
                });
                return result;
            },

            //This function returns true if the model has an integration provider that disables
            //the given field
            //It expects that _this.model.get("integrationProvider") has been set already
            //Reference ChangeRow.js or InitiativeRow.js for an example.
            isIntegratedFieldReadOnly: function(fieldName) {
                var _this = this;
                //all fields are editable if the change was created manually
                if (!_this.model.get("integrationProvider")) {
                    return false;
                }
                var iPArray = array.filter(_this.getCache("integrationProviders"), function(item) {
                    return item.id === _this.model.get("integrationProvider").get("id");
                });
                if (iPArray.length === 0) {
                    return false;
                }
                //there should never be more than 1 integration provider for a change
                return iPArray[0]["propSheet/disabled:" +
                        _this.model.getShortDeclaredClass().toLowerCase() +
                        ":" + fieldName.toLowerCase()] === "disabled";
            }
        });
    });
