/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
define([
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/topic",
    "dojo/on",
    "dojo/fx",
    "dojo/dom-construct",
    "dojo/dom-class",
    "dojo/_base/array",
    "dijit/form/Button",
    "app/widgets/UReleaseWidget"
], function (
    declare,
    lang,
    topic,
    on,
    coreFx,
    domConstruct,
    domClass,
    array,
    Button,
    UReleaseWidget
) {

    /**
     * Creates a list of widgets corresponding to the members of a collection-based Resource passed in.
     * Takes an object following:
     *  {
     *      model: Resource instance containing a collection
     *      widgetClass: The dojo "class" object to instantiate for each member of the collection
     *      factory: The dojo "class" object used to create the rows based on the model passed to it.
     *  }
     *
     *  Optional Parameters:
     *
     *      scrollOnClick: Whether or not to scroll page when clicking on the add button.
     *
     *      scrollAmount: Amount of pixels to scroll page. Leave undefined to scroll to bottom.
     *
     *      selectField: Which field to select when creating a new item. Default is field 1 (Usually Name Field).
     *
     *      onClick: Override what happens when the add button is clicked.
     *
     *      addLabel: Set the label of the add button, Set to null to hide. Default is "Add New"
     *
     *      buttonClass: You can specify a class name for the button.
     *
     *      textButton: Set true to use a text button instead of a regular button
     *
     *      textButtonClass: You can specify a class name for the text button.
     *
     *      textButtonIcon: You can specify a class name that defines an icon to use with the text button.
     *
     *      labelNode: Override which attach point to place the add button.
     *
     *      newItemPlacement: Where to display new items (first or last): Default: last
     *
     */


    return declare("app/widgets/ResourceList",
            [UReleaseWidget], {
        templateString:
            "<div class=\"resource-list\">"+
                "<div data-dojo-attach-point=\"listAttach\"></div>"+
                "<div data-dojo-attach-point=\"noneAttach\"></div>"+
                "<div data-dojo-attach-point=\"buttonAttach\"></div>"+
            "</div>",

        scrollOnClick: false, // whether or not to scroll page when clicking on the add button.
        scrollAmount: null, // Amount of pixels to scroll page. Leave undefined to scroll to bottom.
        selectField: 1, // Which field to select when creating a new item. Default is field 1 (Usually Name Field).
        addLabel: "",
        labelNode: null,
        buttonClass: "",
        newButton: null, // reference to the button to create new items in the list
        noneMessage: "",
        animationSpeed: 80,
        asyncCreation: false,
        asyncCreationDelay: 0,
        firstLoad: true,//in async creation we will draw rows differently for the first load only,
        //For other updates it will be handled the regular way
        childMiscData:null,//holds data to be passed to each child widget
        postCreate: function() {
            this.inherited(arguments);
            this.useStandby();
            this._initializeTable();
        },
        _initI18n: function() {
            this.inherited(arguments);
            if(this.addLabel  !== null) {
                this.addLabel = !!this.addLabel ? this.addLabel : i18n("Add New");
            }
            this.noneMessage = i18n("No items to show");
        },
        _initializeTable: function () {
            var _this = this;
            this.childWidgets = [];
            this.createAndPlaceAddLabel();

            // since we haven't yet set the change listener (below),
            // trigger a load if the resource backing this isn't
            // already loaded.
            if (!this.model.isLoaded()) {
                if (!! _this.standby) {
                    _this.standby.show();
                }
                this.model.load().then(function() {
                    _this.show();
                });
            }
            else {
                this.show();
            }

             this.model.addChangeListener(function() {
                _this.show();
             }, this);
        },

        getChildWidgets : function () {
            return  this.childWidgets;
        },

        show: function() {
            if(this.firstLoad || !this.skipRedraw) {
                this._emptyAndDestroyChildWidgets();
                this.showList();
            }
            this.displayNoneMessage();
            //this._setEvenOddClasses();

            //It is not the first load of the Resource list anymore
            this.firstLoad = false;
            this.afterLoad();
            if (!!this.standby) {
                this.standby.hide();
            }
        },
        /**
         * method is overriden by a function that is executed after the widget loads
         */
        afterLoad: function() {
        },

        /**
         * Destroy all row widgets in this list, both the widget and the dom
         */
        _emptyAndDestroyChildWidgets : function () {
            array.forEach(this.childWidgets, function(childWidget) {
                childWidget.destroyRecursive();
            });
            this.childWidgets = [];
            domConstruct.empty(this.listAttach);

        },

        displayNoneMessage: function() {
            if (this.getDisplayList().length === 0) {
                this.noneAttach.innerHTML = this.noneMessage;
                domClass.remove(this.noneAttach, "hidden");
                domClass.add(this.listAttach, "empty-list");
            }
            else {
                domClass.add(this.noneAttach, "hidden");
            }
        },

        //Layout of div with an unique id for each row that need to be drawn
        createLayout: function (prefix) {
            var _this = this;

            var i = 0;
            //The div will have the following id: prefix+order ex: [prefix]1, [prefix]2
            for (i = 0; i < this.getDisplayList().length; i++) {
                var anchor = domConstruct.create("div", {
                    id: prefix+i
                }, _this.listAttach);
            }
        },

        showList: function() {
            var _this = this;

            var delay = 0;
            //Delay between drawing 2 widgets
            //IE does not behave as well if the delay is < 200 ms
            //To the user's eyes 200 is still pretty fast though
            var delayIncrease = _this.asyncCreationDelay;
            var index = 0;
            var prefix;

            //If we are in async load mode and this is the first load of the list
            if (_this.asyncCreation && _this.firstLoad) {
                //We get the short name of the class to use it as a prefix
                prefix = _this.model.getShortDeclaredClass();
                //We lay out all divs that will be used to draw each item in a specific order since
                //creating widgets using the timeout function won't guaranty the order
                _this.createLayout (prefix);
            }

            array.forEach(this.getDisplayList(), function(member) {
                // TODO remove listModel, and make references to listModel in ResourceViews and inheritors refer to model through containingList.model
                //If we are in async load mode and this is the first load of the list
                //We don't want to reload the entire list every time a change is made after the full list has been loaded once.
                //So after the first load we will handle refresh of the list the classic way
                if (_this.asyncCreation && _this.firstLoad) {
                    _this.member = member;

                    var timeout = lang.hitch(_this, function (order) {

                        var widget = _this._createChildWidget(member);
                        //We add a spinner for each row created
                        if (!widget.standby) {
                            widget.useStandby();
                        }

                        widget.standby.show();
                        widget.placeAt(dojo.byId(prefix+order));

                        //Expand a row if an anchor exists for it
                        _this.autoExpand(widget, member);
                        widget.standby.hide();
                    });

                    //Here we add a delay between each draw
                    delay += delayIncrease;

                    //We create a time out to run the display function and we pass it an index
                    //that will ensure that it will use the div created earlier to place that node
                    setTimeout(timeout.bind(this, index), delay);

                    index++;
                }
                else {
                    var widget = _this._createChildWidget(member);
                    _this._placeChildWidget(widget);
                    //Expand a row if an anchor exists for it
                    _this.autoExpand(widget, member);
                }
            });
        },

        autoExpand: function(item, member){
            //We expand the row if we find a custom anchor
            if (uReleaseConfig.customAnchors) {
               array.forEach(uReleaseConfig.customAnchors, function (anchor) {
                   if (member) {
                       if (member.getShortDeclaredClass() === anchor.type) {
                           if (anchor.value === member.get("id")) {
                               item.toggleExpanded();
                            }
                       }
                   }
               });
            }
        },

        /**
         * @private
         */
        _createChildWidget : function (member) {
            //XXX: Refactor to attempt to have an optional variable of a "Factory"
            //class, which generates widgets that will be used for the rows.
            var childWidget;
            if(!!this.factory) {
                var rowFactory = new this.factory();
                childWidget = rowFactory.createWidgetForModel(this, member, this.model);
            }
            else {
                var childMiscData = this.childMiscData;
                childWidget = new this.widgetClass({
                    containingList : this,
                    model: member,
                    listModel: this.model,
                    childMiscData: childMiscData
                });
            }
            this.childWidgets.push(childWidget);
            return childWidget;
        },
        /**
         * @private
         */
        _placeChildWidget : function (widget) {
            widget.placeAt(this.listAttach);
        },

        /**
         * Creates the "Create Segment" button
         * Sets disabled or enabled based on if user canAddMember()
         *
         * TODO: put elsewhere
         */
        _createAndPlaceCreateSegment: function(isText) {
            var _this = this;
            if (!isText && this.labelNode !== null) {
                if (!_this.buttonClass){
                    this.buttonClass = "";
                }
                //We set the onClick event when the page is originally locked incase it unlocks
                if (!this.onClick) {
                    this._setOnClick();
                }

                // HACK: segments use canAddMember() for adding tasks, use canWrite() for segments intead
                var isDisabled = this.model.getShortDeclaredClass().indexOf("Segment") === -1 ? !_this.model.canAddMember() : !_this.model.canWrite();

                _this.newButton = new Button({
                    label: _this.addLabel,
                    onClick: _this.onClick,
                    disabled: isDisabled,
                    baseClass: "dijit dijitReset dijitInline dijitButton idxButtonSpecial" + _this.buttonClass
                });
                _this.newButton.placeAt(_this.labelNode);
            }
        },

        /**
         * sets onClick event for things like "Create Segment" and "Add Manual Task"
         */
        _setOnClick: function() {
            var _this = this;
            //We dont want to override the onClick if it was specifically set.
            //We check to see if the onClick has been set already
            if (!this.onClick) {
                this.onClick = function(anchor, before) {

                    _this.showBlankChild(anchor, before);
                    if (_this.scrollOnClick) {
                        if (_this.scrollAmount){
                            window.scrollBy(0, _this.scrollAmount);
                        }
                        else {
                            window.scrollTo(0, document.body.scrollHeight);
                        }
                    }
                    // Select field on expand.
                    try {
                        if (_this.selectFieldParent && _this.blankChild.domNode[_this.selectFieldParent][_this.selectField]){
                            setTimeout(function(){
                                _this.blankChild.domNode[_this.selectFieldParent][_this.selectField].focus();
                            }, _this.animationSpeed+20);
                        }
                        else if (_this.selectFieldArray){
                            var blankChild = _this.blankChild.domNode;
                            array.forEach(_this.selectFieldArray, function(index){
                                if (blankChild.children[index]){
                                    blankChild = blankChild.children[index];
                                }
                            });
                            setTimeout(function(){
                                blankChild.focus();
                            }, _this.animationSpeed+20);
                        }
                        else if (_this.blankChild.domNode.firstChild[_this.selectField]) {
                            setTimeout(function(){
                                if (_this.blankChild && _this.blankChild.domNode) {
                                    _this.blankChild.domNode.firstChild[_this.selectField].focus();
                                }
                            }, _this.animationSpeed+20);
                        }
                    } catch(e) {
                        console.error(i18n('failed to find element on which to focus'));
                    }
                };
            }
        },

        /**
         * Creates a label and places it in the resource list
         * Depends on canAddMember and this.Label
         */
        createAndPlaceAddLabel : function () {
            var _this = this;
            var isText = _this.textButton || _this.textButtonIcon;

            // HACK: segments use canAddMember() for adding a task
            var canAddMember = this.model.getShortDeclaredClass().indexOf("Segment") === -1 ? this.model.canAddMember() : this.model.canWrite();

            if(canAddMember && this.addLabel) {
                this._setOnClick();

                if (this.labelNode === null) {
                    if (this.domNode && this.domNode.parentNode && this.domNode.parentNode.parentNode){
                        this.labelNode = this.domNode.parentNode.parentNode.children[0].children[1];
                        this.newItemPlacement = "first";
                    }
                    else {
                        this.labelNode = this.buttonAttach;
                        if (!this.newItemPlacement){
                            this.newItemPlacement = "last";
                        }
                    }
                }
                else if (!this.newItemPlacement){
                    this.newItemPlacement = "first";
                }
                if(!this.newButton) {
                    if (isText){
                        var buttonClassName = this.textButtonIcon ? "resource-button inline-block " + this.textButtonIcon : "inline-block";
                        var buttonContents = this.textButtonIcon ? "<div class=\"" + buttonClassName + "\"></div>" : "";
                        this.newButton = domConstruct.create("a", {
                            className: "text-button " + this.textButtonClass,
                            innerHTML: buttonContents + this.addLabel
                        }, this.labelNode);
                        on(this.newButton, "click", this.onClick);
                    }
                    // TODO - This should not be called here, and should be refactored to be called separately.
                    this._createAndPlaceCreateSegment(isText);
                }
            }
        },

        getDisplayList: function() {
            var _this = this;
            var result = [];

            array.forEach(this.model.getMembers(), function(member) {
                if (_this.filterMember(member)) {
                    result.push(member);
                }
            });

            return result;
        },

        /**
         * A function which can optionally be overridden to filter the contents of the list. This is
         * useful to provide different slices of the same list of resources.
         */
        filterMember: function(resource) {
            return true;
        },

        removeChild: function(child) {
            util.removeFromArray(this.childWidgets, child);
            child.destroyRecursive();
            //this._setEvenOddClasses();
            this.displayNoneMessage();
        },
        /**
         * Create a new row for adding a new object
         * @private
         */
        _createBlankChild: function () {
            if (!this.blankModel || !!this.blankModel.resourceId ||
                    !this.blankChild || !this.blankChild.domNode) {
                var modelClass = this.model.declaredClass;
                var modelClassObject = lang.getObject(modelClass);

                this.blankModel = new modelClassObject({}, this.model);
                this.blankModel.isBlankModel = true;
                var childMiscData = this.childMiscData;
                this.blankChild = new this.widgetClass({
                    "containingList": this,
                    "model": this.blankModel,
                    "listModel": this.model,
                    "className": "new-item resource",
                     "childMiscData" : childMiscData//passes data meant for child widget data
                });
            }
        },

        /**
         * Show a new row for adding a new object
         * If index is defined, it will place the new row at that index
         */
        showBlankChild: function(index) {
            // If there is already a blank child being shown remove it
            if (!!this.blankChild && !!this.blankChild.domNode) {
                this.listAttach.removeChild(this.blankChild.domNode);
                util.removeFromArray(this.childWidgets, this.blankChild);
            }
            this._createBlankChild();
            // Only show the row if we don't have one, if the one we had is saved (has an ID), or if the one
            // we had is deleted (no domNode)
            if (this.newItemPlacement === "first") {
                domClass.remove(this.blankChild.domNode,"new-item");
                domClass.add(this.blankChild.domNode,"new-item-top");
            }
            // Check if we have an index (which could be zero and even -1)
            if ((index === null) || (index === undefined) || (isNaN(index))) {
                this.blankChild.placeAt(this.listAttach, this.newItemPlacement);
            }
            else {
                this.blankChild.placeAt(this.listAttach, index + 1);
            }
            this.childWidgets.push(this.blankChild);
            coreFx.wipeIn({
                node: this.blankChild.domNode,
                duration: this.animationSpeed
            }).play();
            //this._setEvenOddClasses();
            domClass.add(this.noneAttach, "hidden");
        },

        /**
         * Apply even and odd classes to each row of the table.
         */
        _setEvenOddClasses : function () {
            array.forEach(this.childWidgets, function (row,i) {
                if(row.domNode) {
                    domClass.remove(row.domNode,"odd");
                    domClass.remove(row.domNode,"even");
                    if (i%2 === 0) {
                        domClass.add(row.domNode,"even");
                    } else {
                        domClass.add(row.domNode,"odd");
                    }
                }
            });
        },

        destroy: function() {
            if (this.childWidgets) {
                array.forEach(this.childWidgets, function(childWidget) {
                    childWidget.destroyRecursive();
                });
            }

            if (this.model) {
                this.model.removeChangeListeners(this);
            }

            this.inherited(arguments);
        }
    });
});
