/*globals define, i18n, uReleaseConfig */
define([
        'dojo/aspect',
        'dojo/dom-construct',
        'dojo/_base/array',
        'dojo/_base/declare',
        'dojo/_base/lang',
        'dojo/_base/xhr',
        'dijit/form/CheckBox',
        'dojo/store/JsonRest',
        'dojo/store/Memory',
        'dojo/store/Observable',
        'dijit/tree/ObjectStoreModel',
        'dijit/Tree'
        ],
function(
        aspect,
        domConstruct,
        array,
        declare,
        lang,
        xhr,
        CheckBox,
        JsonRest,
        Memory,
        Observable,
        OSM,
        Tree
) {
    /**
     * REFACTORING REQUIRED.  This does not use resources and has a custom overriding of Memory to
     * determine the checked values.
     *
     * A class that extends dijit's Tree implementation.  This class is backed by
     * a data store, and will update the information in the table based on the data
     * in the data store.
     *
     * To create a new NestedList, you need to create a store and a model to back the list.
     * To do this, you create one of the following:
     * 1. Memory using dojo/store.Memory
     * 2. Pass this a url, and we will create a JSONRest store.
     * 3. Create both a Memory and a JsonRest store, and then a Cache store, which will handle keeping
     * data locally. This can be done like so:
     *     var cacheStore = new Cache(memoryStore, restStore);
     * Or something similar to that. Once you have your store, you can make it Observable (incase something changes)
     * and then pass it to the ObjectServiceModel. The way the data is handles is like so:
     *     Store -> OSM -> Tree
     *
     * The tree just displays the data, the OSM acts as a conduit, and the store gets the most recent data.
     *
     * After creating this object, you must run .placeAt(DOM AREA).startup() for it to work,
     * followed by parser.parse()
     *
     * ****************************
     * Class Details
     * ****************************
     *
     * PROPERTIES
     * model: a ObjectServiceModel that will backing the table,
     * showRoot: this should always be false because we need a dummy root node in this table,
     *
     * CONSTRUCTOR - Takes an object with the following
     *     - url: String                    The rest endpoint to use so the Tree will be backed by a JsonRest store
     *     - data: JSON                     Data formatted in a hierarchy so the Tree will be backed by Memory. See dojo docs for example
     *     - checkedData: JSON              Data of already checked items. This should be set to empty if it doesn't exist.
     *     - parent: String/UUID            UUID of the root node containing the list of items (eg dependancy contains segments)
     *     - childrenAreAnnotated           Boolean that states if the children should have addition info tacted onto the innerText
     *     - annotatedName                  Name of the object id that you want to tack on.
     *     - checkboxesOnItems: boolean     Turns the tree into a checkbox Tree.
     *
     * The tree is structured in a way that it expects to have to ask the database for new data when you expand
     * a node. So first you get a list of nodes that contain the base of what you want. Each of those nodes should contain
     * an id and a boolean if they have children or not. If they have children, the getChildren() function will go and hit
     * the rest endpoint by tacking on the id of the object you just clicked on to it.
     * XXX - This isn't perfect, and maybe should be more generic in the future
     */
    return declare([Tree], {
        model: null,

        showRoot: false,

        /**
         * We use the constructor to create a JsonRest store if we are passed in a url
         * Otherwise, we do nothing.
         */
        constructor: function (params) {
            var _this = this;

            if (!!params.url) {
                var restStore = new JsonRest({
                    target:params.url + params.parent,

                    getChildren: function (object) {
                        // object may just be a stub object, so get the full object first and then return it's
                        // list of children

                        var result;

                        if (object.length > 0) {
                            // on first load we get all the segments. Would be better if we got a dummy root and
                            // hit an endpoint for the segments. Only because of how the store is structured
                            result = object;
                        }
                        else {
                            // TODO - Stop doing this and extract this so the endpoint is more generic.
                            if (object.type === "SegmentPlan") {
                                this.target = uReleaseConfig.urls.base + "segmentPlans/" + object.id;
                            }
                            if (object.type === "SegmentExecution") {
                                this.target = uReleaseConfig.urls.base + "segmentExecutions/" + object.id;
                            }
                            result = this.get("/task/" + _this.model.task + "/dependencies").then(function (fullObject) {
                                return fullObject;
                            });
                        }

                        return result;
                    }
                });

                _this.model = new OSM({
                    store: restStore,
                    root: params.parent,
                    task: params.task,

                    /**
                     * Tells if an item has or may have children.  Implementing logic here
                     * avoids showing +/- expando icon for nodes that we know don't have children.
                     * (For efficiency reasons we may not want to check if an element actually
                     * has children until user clicks the expando node). This method overrides mayHaveChildren()
                     * from OSM.
                     *
                     * @param item      The item from the dojo store that is being checked.
                     * @return Boolean  False if leaf node (no children), true otherwise.
                     */
                    mayHaveChildren: function (item){
                        return !!item.children && item.children > 0;
                    },

                    // TODO - Change the response from the endpoint so we don't have to override this method.
                    getRoot: function (onItem) {
                        var _this = this;
                        return this.store.get('').then(lang.hitch(_this, function(items){
                            var result = [items];
                            onItem(result[0]);
                        }));
                    }
                });
            }
            else if (!!params.data) {
                var memoryStore  = new Memory({
                    data: params.data,

                    getChildren: function (object) {
                        var result;
                        if (object.length > 0) {
                            // on first load we get all the segments.
                            result = object;
                        }
                        else if (object.tasks){
                            // Then we get the tasks for that segment
                            result = object.tasks;
                        }
                        return result;
                    }
                });

                // We watch the store for any changes that may happen so the tree updates
                var store = new Observable(memoryStore);
                _this.model = new OSM({
                    store: store,
                    root: params.data.segments,
                    task: params.task
                });
            }
            else {
                throw new ReferenceError("Constructor params for NestedList must contain either a url or data.");
            }

            if (!!params.checkboxesOnItems) {
                if (!params.checkedData || !lang.isArray(params.checkedData)) {
                    params.checkedData = [];
                }
                _this.checkedMemoryStore = new Memory({
                    data: params.checkedData,
                    /**
                     * Returns all the object in the memory store
                     * @return Map Map of objects and the ids they are mapped to in the data store.
                     */
                    getAll: function () {
                        return this.data;
                    },

                    get: function(entry) {
                        return this.data[entry];
                    },

                    put: function(entry) {
                        this.data[entry] = entry;
                    },

                    remove: function(entry) {
                        this.data[entry] = null;
                    },

                    /**
                     * Get all the checked objects in the store
                     * @return Array Array of strings with the ids in memory
                     */
                    getAllCheckedIds: function () {
                        var _this = this;
                        var checkedIds = [];
                        var property;

                        for(property in _this.data) {
                            // This statement is needed for JSLint
                            if(_this.data.hasOwnProperty(property)) {
                                if(!!_this.data[property]) {
                                    checkedIds.push(_this.data[property]);
                                }
                            }
                        }

                        return checkedIds;
                    }
                });

                // We meed to add a checkbox to each node after it is created.
                aspect.after(this, "_createTreeNode", function (node) {
                    // We don't want to add it the item we are dealing with is a segment
                    if (!!node.item.type && node.item.type !== "SegmentPlan" && node.item.type !== "SegmentExecution"){
                        var checkbox = new CheckBox({
                            name: "checkbox",
                            checked: !!_this.checkedMemoryStore.data[node.item.id],
                            onChange: lang.hitch(_this, function() {
                                var node = _this.selectedItem;
                                if (_this.checkedMemoryStore.get(node.id)) {
                                    _this.checkedMemoryStore.remove(node.id);
                                }
                                else {
                                    _this.checkedMemoryStore.put(node.id);
                                }
                            })
                        });

                        // Set the checkbox to checked if we already have the data in memory
                        checkbox.placeAt(node.domNode.childNodes[0].lastChild, "first");
                        node.isExpandable = false;
                        node._setExpando();
                    }
                    return node;
                });
            }
            // Used when apprending addition text to the original child.
            // eg: Task -- AppID
            if (!!params.childrenAreAnnotated) {
                aspect.after(this, "_createTreeNode", function (node){
                    if (!!node.item.type && node.item.type !== "Segment") {
                        // We shouldn't be hard coding these values if things change in the future, this is bad
                        // TODO - Change this to be more abstract and not make assumptions
                        var text = node.domNode.childNodes[0].lastChild.childNodes[3];
                        // If the task has an application add it here
                        var application = null;
                        if(node.item[params.annotatedName]) {
                            application = node.item[params.annotatedName];
                        }
                        else {
                            application = "No Application";
                        }
                        text.innerHTML = text.innerHTML + "--" +  application;
                    }
                    return node;
                });
            }
        },

        postCreate: function () {
            var _this = this;
            this.inherited(arguments);
        },

        /**
         * Method to determine what icon the list item gets
         * This class overrides Tree's default getIconClass
         *
         * @param item  The item from the tree being passed in
         * @param opened Says if the item is expanded or collapsed
         * @return CSS Class a class to be added to the icon in the table. Default is no icon.
         */
        getIconClass: function (/*dojo.store.item*/ item, /*Boolean*/ opened){
            return "";
        }
    });
});