/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Release
 * (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/dom-construct",
    "app/widgets/ResourceRow",
    "dijit/form/CheckBox"
], function(
    declare,
    array,
    domConstruct,
    ResourceRow,
    CheckBox
) {
    /**
     * A base class for resource objects which will be represented as a row in a list of divs.
     * Provides a basic framework with various functions to be overridden by subclasses.
     *
     * Expects properties:
     *  {
     *      model: Resource             a Resource instance for the object represented by this row
     *      listModel: Resource         the Resource instance representing the collection this resource belongs to.
     *      containingList :            the containing widget (ResourceList or ResourceTable)
     *  }
     *
     * Supported functions:
     *  getColumns: Return an array of column data objects. Each column object supports the following:
     *  {
     *      name: String            Row heading for this column (1)
     *      width: String           Value to use for CSS 'width' property for this column
     *      formatter: function     Return a string, DOM node, or widget to insert into the cell for
     *                              this column.
     *
     *      filterData: {           An object used to represent the filter fields over headers,
     *                              and give  information to ResourceTable to help it filter against the DB
     *
     *          name: String        derived from the name property above (1)
     *
     *          type: String        The field type (Text, Checkbox).  What kind of widget the user uses to
     *                              filter
     *
     *          filterField: String the relationship of the column being filtered to the database table in HQL.  For example:
     *                              filtering Release name on a ChangeRow, there is no name for Release in the change DAO
     *                              filterField would be "release.name", allowing the query to traverse the release relationship
     *                              to get the release id from its own object.
     *      }
     *      getValue: function      Return a text value for this cell - used as display if no formatter
     *                              is given. Also used as the default value for edit mode.
     *      fieldData: {            Data for generating the edit-mode of this column
     *          getField / Function A function to return the widget or DOM node to show for edit mode
     *              - or -
     *          type / String       Type of the field. See FormDelegates.js for built-in types.
     *          ...and any properties documented by FormDelegates.js
     *      }
     *  }
     *
     *
     * REFERENCE IMPLEMENTATION: widgets/integration/IntegrationProvider.js
     */
    return declare([ResourceRow], {
        /**
         *
         */
        checked: false,
        show: function() {
            var _this = this;
            array.forEach(this._getAlteredColumns(), function(column) {
                domConstruct.place(_this.buildColumn(column), _this.viewAttach);
            });
        },

        /**
         * prepend a Checkbox to the first widget in the row
         */
        _getAlteredColumns: function() {
            var _this = this;

            var row;

            if (this._isSelectable()) {
                row = {
                    "name": "&nbsp;",
                    "filterData": {
                        "type": "Checkbox",
                        "name": "enabled",
                        "label": i18n("Enabled"),
                        "onChange": function(value) {
                            _this.containingList.selectAll(value);
                        }
                    },
                    "width": "30px",
                    "formatter": function() {
                        var cb = new CheckBox({
                            "checked": false,
                            "onChange": function(value) {
                                _this.set('checked', value);
                            }
                        });
                        _this.checkbox = cb;
                        return cb;
                    }
                };
            }
            else {
                row = {
                    "name": "&nbsp;",
                    "filterData": {
                        "type": "Checkbox",
                        "name": "enabled",
                        "label": i18n("Enabled"),
                        "onChange": function(value) {
                            _this.containingList.selectAll(value);
                        }
                    },
                    "width": "30px"
                };
            }

            return [row].concat(this.getColumns());
        },

        _setCheckedAttr: function(value) {
            this.checked = value;

            this.containingList.checkedRows[this.model.get('id')] = value;
            this.checkbox.set('checked', value);

            if (value) {
                this.checkbox.set('value', this.model.get('id'));
            } else {
                this.checkbox.set('value', null);
            }
        },

        /**
         * We assume by default that we are selectable
         **/
        _isSelectable: function() {
            return true;
        }

    });
});
