/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Release
* (c) Copyright IBM Corporation 2011, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
/*globals bootstrap */
define([
    "dojo/_base/declare",
    "dojo/_base/array",
    "dojo/_base/lang",
    "dojo/dom-geometry",
    "dojo/Deferred",
    "dojo/store/JsonRest",
    "dojox/html/entities",
    "app/model/ResourceRegistry",
    "js/webext/widgets/Alert",
    "js/webext/widgets/Dialog",
    "app/util/xhrResponseHandlers",
    "app/util/String",
    "js/webext/widgets/Util"
], function (
    declare,
    array,
    lang,
    domGeom,
    Deferred,
    JsonRest,
    entities,
    Registry,
    Alert,
    Dialog,
    xhrResponseHandlers
) {
    "use strict";
        /**
         * a shared value between resources used to prevent multiple attempts to set the scroll position of the window when multiple tabs are open
         */
        var alreadyScrolled = false;

        /**
         * Resource serves as the model for some object we're showing somewhere. It is designed to be
         * "subclassed", one subclass per back-end class.
         *
         * At a bare minimum, the subclass MUST implement getResourcePath() to return a URL ending in a
         * slash. Subclasses may also extend the constructor to set the associations object.
         *
         * REFERENCE IMPLEMENTATION: model/integration/IntegrationProvider.js
         */
        return declare("app/model/Resource", [], {

            /**
             * The format parameter used in the new Controllers.  Each view/widget should specify
             * which format it wants to receive from the controller
             * TODO move this into a mixin of some kind.
             */
            format: null,

            /**
             * contains model data associated with this resource
             * @private
             */
            serviceData: null,

            /**
             * contains other Resources that were extracted from the JSON used to build this resource
             * @private
             */
            cachedResources: {},

            /**
             * the id of this resource, taken either from get('id') or assigned directly if the resource is instantiated with an id only.
             * @private
             */
            resourceId: null,

            /**
             * a list of getters that override this.get('prop')
             * @type Object
             */
            getters:{},

            /**
             * a list of setters that override this.set('prop', 'value')
             * @type Object
             */
            setters:{},

            /**
             * a module that contains filtering information.
             * TODO: This should be refactored into a mixin.
             */
            _filtering: null,

            /**
             * a module that contains pagination information.
             * TODO: This should be refactored into a mixin.
             */
            _pagination: null,

            /**
             * a module that contains sorting information.
             * TODO: This should be refactored into a mixin.
             */
            _sorting: null,

            /**
             * the last time this Resource synchronized its data with the server
             * @private
             */
            lastRefresh: null,

            /**
             * the dojo widget that mediates communication with the server.  Resource wraps this widget
             * @private
             */
            store: null,

            /**
             * whether or not the object contains changes not saved to the server
             * @private
             */
            isDirty: false,

            /**
             * whether or not this Resource has been populated with JSON from the server
             * @private
             */
            loaded: false,

            /**
             * whether or not this Resource represents a collection of Resources of its own type, or not
             * @private
             */
            isCollection: false,

            /**
             * if the resource is a collection, an array in which to store the members of this collection
             * @private
             */
            members: [],

            /**
             * a list of functions that will be executed when a change event is fired.  You can add to this list with the addChangeListener method.
             * @see addChangeListener
             * @private
             */
            changeListeners: [],

            /**
             * reference to the parent of this resource
             */
            parent: null,

            /**
             * Construct with the ID of an object to create an empty Resource pointing to that ID. It will
             * load its data using the appropriate REST service when needed.
             *
             * Pass an object to create a Resource with pre-loaded data. It can still be refreshed and
             * saved but will not need to use a REST service to load. If an "id" property is part of this
             * object, it represents an existing object. To create a Resource for an unsaved (new) object,
             * pass an object with no "id" property.
             */
            constructor: function(argument, parent) {
                var _this = this;

                // instance properties
                this._filtering = {
                    /**
                     * list of filters that are applied on every load.
                     * Filters should never be added directly.  Instead use this.filter({})
                     * structure:
                     *                  "myEntity.prop": {
                     *                      "field":"myEntity.prop"
                     *                      "type":"like",
                     *                      "class":"String",
                     *                      "value":"Foobar",
                     *                      "locked":"false" <-- optional, prevent the filter from being
                     *                  }                    deleted when clearFilters is called
                     *                                       you can still remove filters by name.
                     */
                    filters: {},

                    /**
                     * whether or not the current filters represent the data that this resource contains.
                     */
                    filtersAreDirty: false,

                    /**
                     * whether or not the filtering is done by the server.
                     */
                    serverSide: false

                };

                this._pagination = {
                    /**
                     * whether or not we are using pagination.
                     * we want this to be default true someday soon.
                     */
                    isPaginated:false,

                    /**
                     * the content-range values that the server returns
                     */
                    range: {
                        start: 0, // XXX we increment the value of start,
                                  // when we get it back from the server
                                  // by 1 since the server is returning odd data
                        end: 0
                    },

                    /**
                     * The total number of records the server knows of that
                     * could be loaded into this resource.
                     */
                    totalRecords:0,

                    /**
                     * The number of items that should be loaded in a
                     */
                    itemsPerPage:10,

                    /**
                     *
                     */
                    pageNumber:1,

                    /**
                     *
                     */
                    totalPages:1
                };

                this._sorting = {
                    orderField:null, // we cannot guess the order-by field, since the server will die if it doesn't exist.
                    sortType: null, // default is no sort.
                    isSortable: false
                };

                this.registry = new Registry();
                this.parent = parent;

                if (uReleaseConfig.urls.base) {
                    this.baseUrl = uReleaseConfig.urls.base;
                } else {
                    throw new Error(i18n("baseUrl is not defined in uReleaseConfig.urls.base. Without this information from the jsp, Resource and its children will not work."));
                }

                // Arrays must always be initialized to [] for new objects to avoid sharing members with all other Resources
                this.members = [];
                this.changeListeners = [];
                this.cachedResources = {};

                this.store = new JsonRest({
                    target: this.getResourcePath(),
                    headers:{"Accept":"application/json"}
                    //accepts: "application/json"
                });

                if (lang.isObject(argument) && !lang.isArray(argument)) {
                    // A prepopulated object has been passed in.
                    this.serviceData = argument;
                    this.lastRefresh = new Date();
                    this.loaded = true;

                    // Get the ID of the object, if specified.
                    this.resourceId = this.serviceData.id;
                    if (this.resourceId) {
                        this.isDirty = false;
                    } else {
                        this.isDirty = true;
                    }

                    this._setChildren();

                    if (this.isDirty) {
                        this.initializeData();
                    }
                } else if (lang.isArray(argument)) {
                    this.isCollection = true;
                    this.lastRefresh = new Date();
                    this.loaded = true;

                    array.forEach(argument, function(child) {
                        _this.members.push(_this._createNestedObject(_this.declaredClass, child));
                    });
                } else if (lang.isString(argument) ||
                        (typeof argument === "number") ||
                         (argument instanceof Number)) {
                    // An ID has been passed as the argument.
                    this.resourceId = argument;
                } else {
                    this.isCollection = true;
                }
            },

            /**
             * This can be overridden to initialize data for a new instance of this resource
             * @abstract
             */
            initializeData: function() {
                // No-op by default
            },

            /***************************************************************************
            * PERMISSIONS METHODS
            ****************************************************************************/

            /**
             * determine whether the current user has permission to add an associated Resource from a list
             * @return Boolean
             * @protected
             */
            canAddMember: function () {
                if(!this.isCollection) {
                    throw new Error(i18n("'canAddMember' may only be called on a collection"));
                }
                return this.testPermission('canAddMember');
            },

            /**
             * determine whether the current user has permission to remove an associated Resource from a list
             * @return Boolean
             * @protected
             */
            canRemoveMember: function () {
                if(!this.isCollection) {
                    throw new Error(i18n("'canRemoveMember' may only be called on a collection"));
                }
                return this.testPermission('canRemoveMember');
            },

            /**
             * determine whether the current user has permission to create or edit a Resource
             * @return Boolean
             * @protected
             */
            canWrite: function () {
                if(this.isCollection) {
                    throw new Error(i18n("Cannot call 'canWrite' on a collection"));
                }
                return this.testPermission('canWrite');
            },

            /**
             * determine whether the current user has permission to delete a Resource
             * @return Boolean
             * @protected
             */
            canDelete: function () {
                if(this.isCollection) {
                    throw new Error(i18n("Cannot call 'canDelete' on a collection"));
                }
                return this.testPermission('canDelete');
            },

            /**
             * determine whether the current user has permission to expand a collapsed view
             * @return Boolean
             * @protected
             */
            canExpand: function () {
                return this.testPermission('canExpand') && this.canWrite();
            },

            /**
             * Set this property if a class should testPermission using a name other than
             * getShortDeclaredClass().  Set the alternate name here.
             */
            permissionType: false,

            /**
             * determine whether or not special permissions have been set for the current user in uReleaseConfig.user
             * @return Boolean
             * @protected
             */
            testPermission : function (prop) {
                var propIsObject = typeof uReleaseConfig.user[prop] === 'object';
                var shortDeclaredClass = (this.permissionType !== false) ? this.permissionType : this.getShortDeclaredClass();
                var declaredClassHasPermission;
                if (propIsObject) {
                    declaredClassHasPermission = uReleaseConfig.user[prop][shortDeclaredClass];
                }
                // if there is a permission set (declaredClassHasPermissionl), use that,
                // otherwise check if the default permission is set to true or false.
                return declaredClassHasPermission !== undefined ? declaredClassHasPermission : uReleaseConfig.user.permitByDefault;
            },

            /**
             * MUST BE OVERRIDDEN.
             * Get the path to the controller class for this type of resource. MUST be overridden.
             * Because of how JsonRest works, the URL this returns must end in a slash - it just appends
             * the ID of the target object to the given URL.
             * @abstract
             * @return {String} path to Resource's base REST endpoint
             */
            getResourcePath: function() {
                throw new Error(i18n('Required method "getResourcePath" has not been implemented for %s subclass of Resource.', this.declaredClass));
            },

            /**
             * to be overridden, returns a string to be appended to the query string -- after an id only, no base endpoint
             */
            getAuxPathComponent: function () {
                if(this.format) {
                    return "?format=" + this.format;
                }

                return "";
            },

            /**
             * get the REST url for this resource
             */
            getResourceUrl: function() {
                var resPath = null;
                if (this.resourceId) {
                    resPath = this.getResourcePath() + this.resourceId;
                }
                return resPath;
            },

            /**
             * This function must be overridden by any subclasses with children of any classes to return
             * an object representing those associations, as in:
             *  return {
             *      "propSheet": "app/model/PropSheet"
             *  };
             * When a property is encountered with an object or array as a value, the associations object
             * will be consulted to determine which Resource subclass to use for that. The necessary
             * subclasses must be required by the class setting the associations to work reliably.
             */
            getAssociations: function() {
                return {};
            },

            /**
             * Determine whether the data has been loaded for this resource yet
             */
            isLoaded: function() {
                return !!this.lastRefresh;
            },

            /**
             * Get a property of the data for this Resource, by property name. This may return:
             *  - A Resource instance, if that property represents another object
             *  - A raw value, otherwise
             * default behavior may be overridden by defining a setter or getter.  These should be defined in pairs
             */

            get: function(propertyName) {
                // using WidgetBase pattern to mimic dojo getter
                var getter = "_get" + propertyName.capitalize() + "Attr";
                if ( this[getter] !== undefined && typeof this[getter] === 'function' ) {
                    return this[getter]();
                }

                if (!this.isLoaded()) {
                    throw new Error(i18n("Resource data has not been loaded yet."));
                }

                // for a collection, return an array of the values of the
                // specified propertyName for each member of the collection.
                if (this.isCollection) {
                    return array.map(this.getMembers(), function (member) {
                        return member.get(propertyName);
                    });
                }
                // TODO:ath, for Resources that are not loaded, rather than throwing an error, load and then complete get (sync)

                // Favor resource objects by name, or fall back to raw value
                return this.cachedResources[propertyName] || this.serviceData[propertyName];
            },

            /**
             * return the total number of records of this type that the
             * endpoint thinks exist.
             */
            _getTotalRecordsAttr: function() {
                return this._pagination.totalRecords;
            },

            /**
             *
             */
            _getItemsPerPageAttr: function () {
                return this._pagination.itemsPerPage;
            },

            /**
             * set the number of items displayed on the page
             */
            _setItemsPerPageAttr: function (itemsPerPage) {
                this._pagination.itemsPerPage = Number(itemsPerPage);
                // since we're setting the number of items to display, the resource should reflect the new number.
                return this;
            },

            /**
             * get the currently selected page number.
             */
            _getPageNumberAttr: function () {
                return this._pagination.pageNumber;
            },

            /**
             * set the current page number (TODO should this update the filter?  Probably)
             */
            _setPageNumberAttr: function (pageNumber) {
                this._pagination.pageNumber = Number(pageNumber);
                // since we're setting the page we're on, the resource should reflect the new page.
                return this;
            },

            /**
             * get the total number of pages available as reported by the server.
             * Note that there is no setter, since this doesn't make sense.
             */
            _getTotalPagesAttr: function () {
                return this._pagination.totalPages;
            },

            /**
             * get the starting offset of the items currently contained in the resource
             */
            _getRangeStartAttr: function () {
                return this._pagination.range.start;
            },

            /**
             * get the ending offset of the items currently contained in the resource
             */
            _getRangeEndAttr: function () {
                return this._pagination.range.end;
            },

            /**
             *
             */
            _getIsPaginatedAttr: function () {
                return this._pagination.isPaginated;
            },

            /**
             * set whether or not this resource is paginated.
             * @return reference to this resource.
             */
            _setIsPaginatedAttr: function (isPaginated) {
                this._pagination.isPaginated = isPaginated;
                this._filtering.serverSide = isPaginated;
                this._sorting.isSortable = isPaginated;
                return this;
            },

            /**
             *
             */
            _getIsSortableAttr: function () {
                return this._sorting.isSortable;
            },

            /**
             * set whether or not this resource is sortable.
             * @return reference to this resource.
             */
            _setIsSortableAttr: function (isSortable) {
                // for the time being,
                this.set('isPaginated', isSortable);
                //this._sorting.isSortable = isSortable;
                return this;
            },

            /**
             *
             */
            _getHasSortingPropertiesAttr: function () {
                // we don't need a sortType.  If we don't specify,
                // we just toggle the next type.
                return this._sorting.orderField;
            },

            /**
             * return html encoded Resource serviceData, using get.
             * This method prevents rendering of undefined or null by
             * converting them to "".
             * @see this.get
             * @param {String} property.
             * @return {String} html encoded property string
             *
             */
            getHTML : function (property) {
                var value = this.get(property);

                // we assume that for the purposes of UI representation of HTML
                // encoded Resource properties that null and undefined are
                // equivalent to ""
                // ( Oracle DB will return null for empty strings )
                if(value === undefined || value === null) {
                    return "";
                }
                // entities.encode only works with strings.
                return entities.encode(String(value));
            },

            /**
             * Set a property of the data for this Resource, by property name. The given value must be
             * of equivalent type to the existing value (if the property is another Resource, the argument
             * must be a Resource, etc.)
             * default behavior may be overridden by defining a setter or getter.  These should be defined in pairs
             * @returns this set() is a chainable method on resource.
             */
            set: function(propertyName, propertyValue) {
                // using WidgetBase pattern to mimic dojo setter
                var setter = "_set" + propertyName.capitalize() + "Attr";
                if ( this[setter] !== undefined && typeof this[setter] === 'function') {
                    this[setter](propertyValue);
                }
                else {
                    if (!this.isLoaded()) {
                        throw new Error(i18n("Resource data has not been loaded yet."));
                    }

                    if (this.isCollection) {
                        throw new Error(i18n("This resource is a collection: ")+this);
                    }

                    // Only dirty when the propertyValue is actually changing.  In the case of objects,
                    // we can't really tell without deep checking, so skip it and assume it's dirty.
                    if ((typeof propertyValue !== "string" && typeof propertyValue !== "number") || this.get(propertyName) !== propertyValue) {
                        this.isDirty = true;

                        // likewise, only set an instance of clean data when we are actually changing serviceData
                        if(!this.cleanServiceData) {
                            this.cleanServiceData = lang.clone(this.serviceData);
                        }
                    }

                    if (propertyValue === undefined || propertyValue === null) {
                        // When clearing a propertyValue (setting to null/undefined), make sure it's removed from both
                        // serviceData and cachedResources.
                        this.serviceData[propertyName] = null;
                        this.cachedResources[propertyName] = null;
                    }
                    else {
                        if (propertyValue instanceof lang.getObject("app/model/Resource")) {
                            // Setting a resource as a propertyValue
                            this.cachedResources[propertyName] = propertyValue;
                            this.serviceData[propertyName] = propertyValue.resourceId;
                        }
                        else {
                            // Setting a raw propertyValue
                            this.cachedResources[propertyName] = null;
                            this.serviceData[propertyName] = propertyValue;
                        }

                        this._doPatch();
                    }
                }
                return this;
            },

            /**
             * sort the members of a Collection according to their name (can be overridden)
             * @return reference to this resource.
             */
            sortMembers: function () {
                if(!this.isCollection) {
                    throw new Error(i18n('cannot sort instance of Resource that is not a collection'));
                }
                this.members.sort(function (a,b) {
                    return b.serviceData.name.toUpperCase() < a.serviceData.name.toUpperCase() ? 1 : -1;
                });
                return this;
            },

            /**
             * @param {String} name : the name of the property to be created
             * @param {String} value : the value to be assigned to the property
             */
            addProperty: function(name, value){
                if (!name) {
                    throw new Error(i18n("Property Name is required"));
                }
                if (value === undefined) {
                    throw new Error(i18n("Property %s's Value is undefined", name));
                }
                if (this.serviceData[name]) {
                    throw new Error(i18n("Property %s already exists", name));
                }

                this.serviceData[name] = value;
            },

            /**
             * @param {String} name : the name of the property to be removed
             * @return {String} the name of the property removed
             */
            removeProperty: function(name){
                if (!name) {
                    throw new Error(i18n("Property Name is required"));
                }
                if (!this.serviceData[name]) {
                    throw new Error(i18n("Property %s doesn't exists", name));
                }
                delete this.serviceData[name];

                return name;
            },

            /**
             * Get the resource object at the top of this Resource's parental hierarchy
             */
            getRootParent: function() {
                var ret;
                if (this.parent) {
                    ret = this.parent.getRootParent();
                }
                else {
                    ret = this;
                }
                return ret;
            },

            /**
             * given a shortened class name, will search a this Resource's parentage for the first shortened
             * class name matching the param given (ex: app/model/application/Application could be shortened to Application)
             * @param {String} className the shortened classname representing a class that may be in the Resource's parental hierarchy
             * @return {Resource} the first instance of Resource in the parentage of this resource corresponding to the string param
             */
            getNearestParentByShortClass: function(classname) {
                var ret;
                if (this.parent) {
                    if (this.parent.getShortDeclaredClass() === classname) {
                        ret = this.parent;
                    }
                    else {
                        ret = this.parent.getNearestParentByShortClass(classname);
                    }
                    return ret;
                }
            },
            /**
             * given a shortened class name, will search a this Resource's parentage for the first
             * class name matching the param given (ex: 'app/model/application/Application')
             * @param {String} className the classname representing a class that may be in the Resource's parental hierarchy
             * @return {Resource} the first instance of Resource in the parentage of this resource corresponding to the string param
             */
            getNearestParentByClass: function(classname) {
                var ret;
                if (this.parent) {
                    if (this.parent.declaredClass === classname) {
                        ret = this.parent;
                    }
                    else {
                        ret = this.parent.getNearestParentByClass(classname);
                    }
                    return ret;
                }
            },

            /**
             * Get all members of the collection this Resource represents. (Collections only)
             */
            getMembers: function() {
                if (!this.isCollection) {
                    throw new Error(i18n("This resource is not a collection: %s",this));
                }
                return this.members;
            },

            /**
             * get a member resource by Id. (Collections only)
             * @return the member found or `null`
             */
            getMemberById: function(id) {
                var filtered = array.filter(this.getMembers(), function(member) {
                    return member.resourceId === id;
                });

                var result;
                if (filtered.length > 0) {
                    result = filtered[0];
                }
                else {
                    result = null;
                }
                return result;
            },

            /**
             * Add a new child to this resource collection.
             */
            addMember: function(child, index) {
                if (!this.isCollection) {
                    throw new Error(i18n("This resource is not a collection: %s",this));
                }
                // Check if index is defined (it can be zero or even -1)
                if (index === undefined || index === null) {
                    this.members.push(child);
                }
                else {
                    //splice the array including index and then insert the new child
                    var startMembers = this.members;
                    var beforeElements = startMembers.splice(0, index + 1);
                    beforeElements.push(child);
                    this.members = beforeElements.concat(startMembers);
                }

                if (!child.parent) {
                    child.parent = this;
                }

                //this._incrementTotalRecords(); <-- this is not a sufficient solution.
                //                                   We have to load the list again completely
                //                                   because the item we just created may not meet
                //                                   the requirements.
                //                                   Is there any reason not to show the newly created item?

                this._doPatch();
                this._fireChangeEvent();
            },

            /**
             * Add a new children to this resource collection.
             */
            addMembers: function(children) {
                if (!this.isCollection) {
                    throw new Error(i18n("This resource is not a collection: %s",this));
                }
                var _this = this;
                array.forEach(children, function(child){
                    _this.members.push(child);
                    if (!child.parent) {
                        child.parent = _this;
                    }
                    //_this._incrementTotalRecords();
                });
                this._doPatch();
                this._fireChangeEvent();
            },

            /**
             * remove a child from this collection.
             * @param {Resource|String} child: a child resource or a string that is
             *                          the childs id.
             */
            removeMember: function(child) {
                if(!child) {
                    throw new Error("Must indicate the child to remove.");
                }

                if (!this.isCollection) {
                    throw new Error(i18n("This resource is not a collection: %s",this));
                }

                //TODO use a more explicit comparison implementation once it is merged into Webext.
                //util.removeFromArray(this.members, child, function (arrayIndex, value) {
                    //if(typeof value === 'string') {
                        //return value === arrayIndex.get('id');
                    //}
                    //return value.get('id') === arrayIndex.get('id');
                //});
                util.removeFromArray(this.members, child);
                if (child.parent === this) {
                    child.parent = undefined;
                }
                //this._decrementTotalRecords();
                this._doPatch();
                this._fireChangeEvent();
            },

            /**
             * Remove children from this resource collection.
             */
            removeMembers: function(children) {
                if (!this.isCollection) {
                    throw new Error(i18n("This resource is not a collection: %s",this));
                }
                var _this = this;
                array.forEach(children, function(child){
                    util.removeFromArray(_this.members, child);
                    if (child.parent === _this) {
                        child.parent = undefined;
                    }
                    //_this._decrementTotalRecords();
                });
                this._doPatch();
                this._fireChangeEvent();
            },

            /**
             * Load the data for this Resource and return a Promise to handle load completion
             */
            load: function(queryObject, localLoad) {
                var _this = this;

                // capture where the page is scrolled, so we can return here after load.
                this.setCurrentScrollPosition();

                // if there is a request in-flight cancel it.
                // TODO: allow ResourceLists and ResourceViews to accept promises of resources
                // so that we can use this behavior.
                //if(this.promise) {
                    //this.promise.cancel();
                //}


                // There are two types load requests we can make.
                // 1. to a collection
                // 2. to an individual resource.
                //
                // the response from both of these requests is handled in exactly the same way
                // so we differentiate between isCollection and an individual resource
                // to make our request, but we chain the same response handler to whichever request
                // path-in-code was chosen.

                var requestString;

                if(this.isCollection) {
                    // if we are a collection, build and run a query...
                    // ... and include any filters that have been set
                    requestString = this._buildStoreQuery(queryObject);

                    this.promise = this.store.query(requestString);
                }
                else {
                    // if we are a single resource, get that resource and mix in individual path components
                    requestString = this.resourceId || "";
                    requestString += this.getAuxPathComponent();

                    // as a single resource, we must make a get request to the server for that specific record
                    // instead of querying (which happens above).
                    this.promise = this.store.get(requestString);



                    var deferred = new Deferred();

                    // since the total promise is not returned from a store.get()
                    // request (only a query), we create a stand-in, and resolve
                    // total to Number(1) -- representing a single record returned.
                    this.promise.total = deferred.promise;
                    deferred.resolve(1);
                }

                // now that we've made a request, and guaranteed that both get and query promises contain
                // an additional promise called total attached to them (we shimmed one in for the get request),
                // we can begin handling responses.
                return this.promise.then(function (data) {

                    // FIRST, we start handling meta-data that we have chosen or are forced to store on the main
                    // reponse
                    if(_this.get('isPaginated')) {
                        // ...in the case where our resource is using server-side pagination, we gather data about
                        // the range of content returned from the server.
                        _this._updatePaginationWithRangeHeader(_this.promise.ioArgs.xhr.getResponseHeader('Content-Range'));

                        // because filters can be set on the resource without executing a query
                        // we have to differentiate between a resource whose filters represent
                        // what is actually contained in the resource, and a resource whose
                        // filters have been changed without executing a query.
                        // In practice, setting a filter will usually be immediately followed
                        // by a query() call.
                        //
                        // TODO : Continue R&D on executing a query automatically
                        // in the next execution cycle,  after all filters have been set.
                        _this._filtering.filtersAreDirty = false;
                    }

                    // NEXT we extract additional meta-data from the
                    // attached `total` promise. We do this before returning control
                    // to the process that called resource.load() in order to guarantee
                    // we have all the meta-data available before we start acting on
                    // the data in our app.
                    // Note: The `total` promise is returned at the same time as the
                    // response in the current implementation of JsonRest, but this
                    // pattern future-proofs us against a change in that behavior.
                    return _this.promise.total.then(function (total) {
                        _this._pagination.totalRecords = Number(total);
                        _this._pagination.totalPages =
                            Math.ceil(
                                // Math converts implicitly to a number.
                                total / _this.get('itemsPerPage')
                            );


                        return data;
                    }).then(function (data) {
                        // FINALLY now that we have all of our
                        // meta-data gathered from the main
                        // reponse and the `total` secondary
                        // promise, we can use the data to update
                        // our local reosurce's cache.
                        _this.update(data);

                        // and scroll to the position the user was at
                        // if this load() call redrew the UI.
                        // XXX: move this out of Resource.
                        _this.scrollToLastDocumentPosition();
                        return data;
                    }, function(error) {
                        xhrResponseHandlers.handleError(error);
                    });
                }, function (error) {
                    // In the event that our load() request fails,
                    // handle the error.  This is an entirely
                    // different chain from what happens above with
                    // a reponse and a secondary `total` promise.

                    //  Now that we can cancel requests in flight
                    //  we need to make sure that the request was
                    //  rejected by the server, not canceled by the client
                    //  before we throw an error.  We only check for not
                    //  canceled, since this is the closest to our
                    //  original behavior
                    if(error.name !== "CancelError") {
                        xhrResponseHandlers.handleError(error);
                    }
                });
            },

            /**
             *
             */
            query: function (queryObject, localLoad) {
                var promise;
                if(!queryObject) {
                    console.warn(i18n('Querying without any parameters.  This behaves exactly like load(). Should you be using Load() ?'));
                }
                promise = this.load(queryObject, localLoad);
                return promise;
            },

            /**
             * method to set or unset a filter
             * @param {Object} args: contains arguments for this filter in the form:
             *              {
             *                  "field":"myEntity.prop"
             *                  "type":"like",
             *                  "class":"String",
             *                  "value":"Foobar",
             *                  "locked":"false" <-- optional, prevent the filter from being
             *                                       deleted when clearFilters is called
             *                                       you can still remove filters by name.
             *              }
             * @return reference to this resource.
             */
            filter: function (kwargs) {

                // if we aren't specifying any field to filter,
                // do nothing.
                if(!kwargs.field) {
                    return this;
                }

                this._filtering.filtersAreDirty = true;

                var queryType = kwargs.type || "like", //query type 'like'
                    field = kwargs.field,
                    queryClass = kwargs['class'] || kwargs.className || "String", // String,
                    value = kwargs.value || "";

                if(!value) {
                    this.removeFilter(kwargs.field);
                    return this;
                }

                //========================
                // try to set some intelligent defaults
                //========================
                switch(queryClass.toUpperCase()) {
                    case "UUID":
                        // if we are looking for a UUID, but haven't
                        // specified a type, assume equals
                        if(!queryType) {
                            queryType = "eq";
                        }
                        break;
                    case "BOOLEAN":
                        // if we are looking for a BOOLEAN, but haven't
                        // specified a type, assume equals
                        if(!queryType) {
                            queryType = "eq";
                        }
                        break;
                    case "LONG": //( Date )
                        // if we are looking for a LONG (date), but haven't
                        // specified a type, assume.... ?
                        break;
                    case "STRING":
                        if(!queryType) {
                            queryType = "like";
                        }
                        break;
                    case "ENUM":
                        // if the value we are searching for is an array,
                        // assume queryType is "in"
                        if (lang.isArray(value)) {
                            queryType = "in";
                        } else  {
                            // otherwise, assume equals
                            queryType = "eq";
                        }
                        break;
                    default:
                        throw new Error("Unrecognized queryClass:", queryClass);
                }

                //========================
                // save the filter
                //========================
                //this._createFilter(field, 'filterFields', field);

                this._filtering.filters[field] = {};
                this._filtering.filters[field].filterFields = field;
                this._createFilter(field, 'filterType', queryType);
                this._createFilter(field, 'filterClass', queryClass);
                if(!!kwargs.isMultiple) {
                    var values = kwargs.getMultiValues();
                    var multipleValIndex;
                    for(multipleValIndex=0; multipleValIndex<values.length; multipleValIndex++) {
                        this._createFilter(field, 'filterValue', values[multipleValIndex]);
                    }
                }
                else {
                    this._createFilter(field, 'filterValue', value);
                }

                return this;
            },

            /**
             * helper function for this.filter()
             * @private
             */
            _createFilter: function (field, filter, content) {
                var filterName = filter + "_" + field;


                if(!this._filtering.filters[field][filterName]) {
                    this._filtering.filters[field][filterName] = [];
                }

                this._filtering.filters[field][filterName].push(content);
            },

            /**
             * destroy all filters that are not designated as "locked"
             * locked filters can be removed by name
             * using removeFilter("filterName");
             * @return reference to this resource.
             */
            clearFilters: function () {
                var i;
                // loop through our filters removing
                // any that are not locked.
                //
                for(i in this._filtering.filters) {
                    if(this._filtering.filters.hasOwnProperty(i)) {
                        var filter = this._filtering.filter[i];
                        if(!filter.locked) {
                            delete this._filtering.filter[i];
                        }
                    }
                }

                return this;
            },

            /**
             * remove/unset a filter by name
             * @param {String} filterName dot-delimited string referencing a filter.
             * @return reference to this resource.
             */
            removeFilter: function (filterName) {

                // if the filter exists, delete it.
                if (this._filtering.filters[filterName]) {
                    delete this._filtering.filters[filterName];
                }
                return this;
            },

            /**
             * sort by a given field in the direction specified.
             * currently we only allow sorting by a single column.
             * @param {String} prop: the name of the property to sort by.
             * @param {String} direction: the direction to sort
             * @return reference to this resource.
             */
            sort: function (prop, dir) {

                // if there is no prop, do nothing..
                if(!prop) {
                    return this;
                }
                var direction;
                //capture a variety of inputs.
                if(dir) {
                    switch(dir.toLowerCase()) {
                        case "ascending":
                        case "ascend":
                        case "asc":
                        case "+": // <-- dojo store pattern
                            direction = "asc";
                            break;

                        case "descending":
                        case "descend":
                        case "desc":
                        case "des":
                        case "-": // <-- dojo store pattern
                            direction = "desc";
                            break;
                        default:
                            // if the string is defined, but not something we recognize, throw an error
                            throw new Error("Cound not parse sort direction");
                    }

                    //console.log("specifying sort of", direction);
                } else {
                    //console.log("changing sort to next type");

                    // if we don't explicitly set direction, toggle the next sort.
                    // note that once the user has sorted, it toggles between
                    // ascending and descending with no way to return to un-sorted.

                    if(this._sorting.sortType === "asc" || !this._sorting.sortType) {
                        direction = "desc";
                    } else {
                        direction = "asc";
                    }
                }
                //console.log("direction is", direction);

                this._sorting.orderField = prop;
                this._sorting.sortType = direction;

                return this;
            },



            /**
             * this method gathers filters that have been set on this resource, and
             * mixes in the query parameter object passed.
             * @param {String} queryObject: a query object, (dojo query object).
             * @private
             */
            _buildStoreQuery: function (queryObject) {
                var filterIndex,
                    f,
                    baseQuery = {};

                // If this resource has a specified format, then it must be included in the query
                // params
                if(!!this.format && this.format !== '') {
                    baseQuery = {'format': this.format};
                }

                // set filters
                // NOTE: unlike the builders below where we can just use lang.mixin()
                // we must do this "by hand" here, so that we correctly handle
                // multiple values stored in the same queryParam (same key);
                if(this._filtering.serverSide) {

                    for(filterIndex in this._filtering.filters) {
                        if (this._filtering.filters.hasOwnProperty(filterIndex)) {
                            var filterToAdd = this._filtering.filters[filterIndex];

                            // add each property of the filter object to our query object.
                            for(f in filterToAdd) {
                                if (filterToAdd.hasOwnProperty(f)) {

                                    var existingFilter = baseQuery[f];
                                    // mix in each individual filter's properties
                                    // we cannot use lang.mixin here because it will overwrite identical properties

                                    // if we need to add a new value for the a property that already exists (filterFields)
                                    if(existingFilter) {
                                        // io-query converts arrays of properties into duplicate property names
                                        // with different values, so we need to store each value for filterFields
                                        // greater than 1 in an array

                                        // if the existing filter is not already an array
                                        // make it one.
                                        if(!lang.isArray(existingFilter)) {
                                            baseQuery[f] = [ baseQuery[f] ];
                                        }

                                        // add our new filter to the existing set of filters.
                                        baseQuery[f].push(filterToAdd[f]);
                                    } else {
                                        baseQuery[f] = filterToAdd[f];
                                    }

                                }
                            }
                        }
                    }
                }

                // currently a request for a page number that is greater than
                // the available pages of data will return a 500 error.  We
                // continue to punt on this by resetting the page to 1 when
                // filtering.
                //
                // The above is old, this should work as typed below. WATCH THIS JUST IN CASE THE ABOVE CASE HAPPENS
                if(this._filtering.filtersAreDirty) {
                    // we reach out directly to the property
                    // because we don't want to trigger a load by using the setter.
                    this._pagination.pageNumber = 1;
                }

                // set pagination
                if(this.get('isPaginated')) {
                    lang.mixin(baseQuery, {
                        rowsPerPage: this.get('itemsPerPage'),
                        pageNumber: this.get('pageNumber')
                    });
                }

                if(this.get('isSortable') && this.get('hasSortingProperties')) {
                    lang.mixin(baseQuery, {
                        sortType: this._sorting.sortType,
                        orderField: this._sorting.orderField
                    });
                }

                lang.mixin(baseQuery, queryObject);

                return baseQuery;
            },

            /**
             * determine items per page and current page from the range that is returned by the store
             * @param {String} rangeString the header "Content-Range" (XXX:should it be "X-Content-Range"??)
             */
            _updatePaginationWithRangeHeader: function (rangeString) {
                var start, end;
                if(rangeString) {
                    start = rangeString.match(/(\d*)-/)[1];
                    end = rangeString.match(/-(\d*)\//)[1];

                    // hack to get around the fact that we display 0-10 and show 10 not 11 items.
                    // We'll need to look into why this result is returned in QueryResponse in commons-util
                    this._pagination.range.start = +start+1;
                    this._pagination.range.end = +end;
                } else {
                    this._pagination.range.start = null;
                    this._pagination.range.end = null;
                }

                // total is taken care of in the `total` promise.

            },

            /***************************************************************************
            * RETURNING TO SCROLL POSITION METHOD
            * XXX: should be moved out of resource.  Including the supporting methods at the top.
            ****************************************************************************/

            /**
             *
             */
            setCurrentScrollPosition : function () {
                uReleaseConfig.userInterfaceState.scrollPosition = domGeom.docScroll();
                alreadyScrolled = false;
            },

            /**
             *
             */
            scrollToLastDocumentPosition: function () {
                if(!alreadyScrolled) {
                    window.scrollTo(uReleaseConfig.userInterfaceState.scrollPosition.x, uReleaseConfig.userInterfaceState.scrollPosition.y);
                    alreadyScrolled = true;
                }
            },

            /**
             * Ensure that all required fields have been populated and return any fields which have not.
             */
            validateRequired: function() {
                var _this = this;
                var offendingFields = [];

                array.forEach(this.fieldsArray, function(field) {
                    if (field.required) {
                        var value = _this.getValueByField(field);

                        if (field.type === "Checkbox") {
                            if (value !== "true") {
                                offendingFields.push(field);
                            }
                        }
                        else {
                            if (value === null || value === undefined || value.length === 0) {
                                if (field.label) {
                                    offendingFields.push(i18n("%s is a required field.", field.label));
                                }
                                else {
                                    offendingFields.push(i18n("%s is a required field.", field.name));
                                }
                            }
                        }
                    }
                });

                return offendingFields;
            },

            /**
             * Save the object represented by this Resource. If no id property is set, this will cause a
             * POST to create a new object. Otherwise, it will PUT to the object's URL to update its
             * values.
             */
            save: function() {
                var _this = this;

                this._setStoreHeader();
                var promise = this.store.put(this.serviceData, {
                    headers: this.header
                });

                promise.then(function (response) {
                    _this.isDirty = false;
                    _this.lastRefresh = new Date();

                    // If we have saved a previously unsaved object (no ID), get the ID back from the
                    // Location header of the response.
                    if (!_this.resourceId) {
                        var location = promise.ioArgs.xhr.getResponseHeader("Location");
                        if (location) {
                            _this.resourceId = location.substring(location.lastIndexOf("/")+1);
                            _this.serviceData.id = _this.resourceId;
                        }
                    }
                    _this._doPatch();
                    _this._fireChangeEvent();
                    return response;
                }).then(function(response) {
                    return response;
                }, function(error) {
                    var alert = new Alert({
                        title: i18n("Error saving %s", _this.getResourceName()),
                        forceRawMessages: true,
                        message: error.responseText});
                    console.error(i18n("Error processing result from save:"), error.responseText);
                    return error;
                });

                return promise;
            },


            /**
             * inject XSS protection headers with requests.
             */
            _setStoreHeader : function () {

               if (!this.header) {
                   this.header = {};
               }
               if (!!bootstrap) {
                    var apiTokenName = bootstrap.apiTokenName || bootstrap.expectedSessionCookieName;
                    if (apiTokenName) {
                        var apiToken = util.getCookie(apiTokenName);
                        if (!!apiToken) {
                            this.header[apiTokenName] = apiToken;
                        }
                    }
                }
            },

            /**
             * Rearrange all members of this collection to match the given array. This should be overridden
             * in any child implementations which support ordered collections so that they save to some
             * REST service.
             */
            setMembers: function(orderedMembers) {
                this.members = orderedMembers;
                this._doPatch();
                this._fireChangeEvent();
            },

            /**
             * Delete the object backing this Resource.
             */
            remove: function() {
                var _this = this;

                if (!this.resourceId) {
                    throw new Error(i18n("Cannot delete an unsaved Resource"));
                }
                this._setStoreHeader();
                var promise = this.store.remove(this.resourceId, {
                    headers:this.header
                });
                promise.then(function(){
                    if (_this.parent && _this.parent.isCollection) {
                        _this.parent.removeMember(_this);
                    }
                }, function(error) {
                    console.error(i18n("Error processing result from remove:"), error);
                });
                return promise;
            },

            /**
             * Update the data for this resource with the given argument.
             */
            update: function(data) {
                var _this = this,
                    handledMembers = [],
                    membersToRemove =[];

                if (this.isCollection) {
                    if (this.get('isPaginated')) {
                        // WARNING: emptying and rebuilding the list of members in a collection is a RADICAL departure from what
                        // we did before.  Keep an eye on this.
                        this.members = []; // <-- TODO could be a memeory leak here?  Probably not since it's a Resource not a widget.

                        array.forEach(data, function(d) {
                            var newMember = _this._createNestedObject(_this.declaredClass, d);
                            _this.members.push(newMember);

                        });
                    } else {
                        // original behavior, mapping server data to existing client data.
                        // TODO, remove this logic entirely, so long as everything still works.
                        array.forEach(data, function(item) {
                            var foundMatch = false;
                            array.forEach(_this.members, function(member) {
                                if (member.resourceId !== undefined && item.id !== undefined && member.resourceId === item.id) {

                                    // perform this same process on the member, passing in
                                    // the object (from json) that it will use.
                                    member.update(item);

                                    // the data item is a member resource of this collection.
                                    foundMatch = true;

                                    // add this item to the list of things we have taken care of.
                                    handledMembers.push(member);
                                }
                            });

                            if (!foundMatch) {
                                var newMember = _this._createNestedObject(_this.declaredClass, item);
                                handledMembers.push(newMember);
                                _this.members.push(newMember);
                            }
                        });


                        // Remove any members which are no longer present as of this refresh.
                        array.forEach(_this.members, function(member) {
                            // if the member is not in the list of members that
                            // were sent from the server, flag it for removal
                            if (handledMembers.indexOf(member) === -1) {
                                membersToRemove.push(member);
                            }
                        });
                        array.forEach(membersToRemove, function(member) {
                            util.removeFromArray(_this.members, member);
                        });
                    }
                }
                else {
                    this.serviceData = data;
                    this.resourceId = data.id;
                }

                this.isDirty = false;
                this.lastRefresh = new Date();
                this._setChildren();

                this._doPatch();
                this._fireChangeEvent();
            },

           /**
            *
            */
           reset: function () {
               // reset
               // description: reset all changes to serviceData, using a clean backup copy
               this.serviceData = this.cleanServiceData;
               delete this.cleanServiceData;
               this.isDirty = false;
           },

            /**
             * functionCall: a function to execute when change is detected
             * source: the object (widget) this change listener is coming from
             */
            addChangeListener: function(functionCall, source) {
                var changeListener = {
                    source: source,
                    functionCall: functionCall
                };
                this.changeListeners.push(changeListener);
                return changeListener;
            },

            /**
             *
             */
            removeChangeListeners: function(source) {
                var _this = this;

                var listenersToRemove = [];
                array.forEach(this.changeListeners, function(changeListener) {
                    if (changeListener.source === source) {
                        listenersToRemove.push(changeListener);
                    }
                });

                array.forEach(listenersToRemove, function(listener) {
                    util.removeFromArray(_this.changeListeners, listener);
                });
            },

            //--------------------------------------------------------------------------------------
            // INTERNAL FUNCTIONS
            //--------------------------------------------------------------------------------------

            /**
             * Go through all object data and pick out any objects to create child resources for
             */
            _setChildren: function() {
                var value,
                    propertyName,
                    className,
                    existingResource,
                    childResource,
                    handledProperties = [];


                for (propertyName in this.serviceData) {
                    if (this.serviceData.hasOwnProperty(propertyName)) {
                        value = this.serviceData[propertyName];

                        if (lang.isObject(value)) {
                            className = this.getAssociations()[propertyName];
                            if (className) {
                                delete this.serviceData[propertyName];

                                existingResource = this.cachedResources[propertyName];

                                // Update resources which we already have, or set new ones.
                                if (existingResource &&
                                        (existingResource.isCollection ||
                                                existingResource.resourceId === value.id)) {
                                    this.cachedResources[propertyName].update(value);
                                }
                                else {
                                    childResource = this._createNestedObject(className, value);
                                    this.cachedResources[propertyName] = childResource;
                                }

                                handledProperties.push(propertyName);
                            }
                        }
                    }
                }

                // Remove any cached resources which were not set by this update.
                for (propertyName in this.cachedResources) {
                    if (this.cachedResources.hasOwnProperty(propertyName)) {
                        if (handledProperties.indexOf(propertyName) === -1) {
                            delete this.cachedResources[propertyName];
                        }
                    }
                }
            },

            /**
             * Create a resource object for a given Dojo class name and data object or ID
             */
            _createNestedObject: function(className, child) {
                var resourceClass = lang.getObject(className);
                if (resourceClass === undefined) {
                    throw new Error(i18n("Unable to find %s. Perhaps it wasn't required.", className));
                }
                return this.registry.get(resourceClass, child, this);
            },

            /**
             *
             */
            _fireChangeEvent: function() {
                array.forEach(this.changeListeners, function(listener) {
                    listener.functionCall();
                });
            },

            /***************************************************************************
            * HIERARCHY INSPECTION METHODS
            ****************************************************************************/

            /**
             * @param {String} parentShortClass
             * @return {String} the short name of the root parent
             */
            rootParentIs : function (parentShortClass) {
                return this.getRootParent() && this.getRootParent().getShortDeclaredClass() === parentShortClass;
            },

            /**
             * get the last ('/' delimited) section of the resources declaredClass
             * e.g. 'app/model/release/ReleaseDeprecated' will return 'Release'
             */
            getShortDeclaredClass: function () {
                var ret;
                var declaredClass = this.declaredClass;
                if(Array.prototype.lastIndexOf) {
                    ret = declaredClass.slice(declaredClass.lastIndexOf('/')+1, declaredClass.length);
                }
                else {
                    // ECMASCRIPT 5 shim.
                    ret = declaredClass.slice(declaredClass.search(/\/[\w]*$/)+1,declaredClass.length);
                }
                return ret;
            },

            /***************************************************************************
            * DATA-PATCH METHODS
            ****************************************************************************/

            /**
             * exisitng patch request
             */
            _patchRequest:null,

            /**
             *
             */
            _listeningWidgets:null,

            /**
             *
             */
            addWidgetPatchReference:function (widget) {
                if(!this._listeningWidgets) {
                    this._listeningWidgets = {};
                }
                this._listeningWidgets[widget.id] = widget;
            },

            /**
             *
             */
            _doPatch: function () {
                var _this = this;
                if(!util.isEmpty(_this._listeningWidgets)) {
                    if(this._patchRequest) {
                        clearTimeout(this._patchRequest);
                    }
                    this._patchRequest = setTimeout(function () {
                        var widget, listeningWidget;
                        //console.log('executing patch for ', _this.resourceId, _this.declaredClass, '.  Should only happen once');
                        for (widget in _this._listeningWidgets) {
                            if(_this._listeningWidgets.hasOwnProperty(widget)) {
                                listeningWidget = _this._listeningWidgets[widget];
                                listeningWidget.patch();
                            }
                        }
                        clearTimeout(_this._patchRequest);
                    }, 10);
                    // set timeout to execute on next cycle, so as to prevent execution of the same
                    // patch for multiple consecutive, synchronous model.set() requests
                }
            }
        });
    }
);
