package com.urbancode.anthill3

import groovy.util.slurpersupport.GPathResult

public class AHPTool {

    //**************************************************************************
    // CLASS
    //**************************************************************************

    static protected enum PropertyType {
        AGENT('Agent'),
        BUILD_LIFE('BuildLife'),
        JOB('Job'),
        REQUEST('Request'),
        STEP('Step'),
        SYSTEM('System'),
        ORIGINATING_REQUEST('OriginatingRequest'),
        ORIGINATING_WORKFLOW('OriginatingWorkflow')

        final def name
        PropertyType(def name) {
            this.name = name
        }
    }

    //**************************************************************************
    // INSTANCE
    //**************************************************************************

    final def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
    final def ahptool = isWindows ? 'ahptool.cmd' : 'ahptool'
    def out = System.out;
    def err = System.err;

    public AHPTool(){
    }

    public void addChangeSets(changeSetXML) {
        def ahptoolProc = [ahptool, 'addSourceChanges', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withWriter{ it << changeSetXML }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add Change Sets using ahptool: " + changeSetXML)
        }
    }

    /**
     * Get a map of bug id to changeset comment.
     *
     * @param regex the regex to match comments against
     * @return a map of bugid to changeset comment
     */
    public Map<String,String> getChangeSets(String regex) {
        def ahptoolProc = [ahptool, "getSourceChanges"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err) // forward process stderr
        def changeSetXML = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get source changes from ahptool: $changeSetXML")
        }
        def pattern = java.util.regex.Pattern.compile(regex)
        def Map<String, String> id2changecomment = new HashMap<String,String>()

        // will throw a parse exception at random times - any ideas?
        new XmlSlurper().parseText(changeSetXML)."change-set".each { element ->
            def matcher = pattern.matcher(element.comment.text())
            while (matcher.find()) {
                def patternid
                def comment
                if (matcher.groupCount() > 0) {
                    // they specified a '(...)' group within the pattern, use that as the bug-id
                    patternid = matcher.group(1)
                }
                else {
                    // use the whole matching substring as the bug-id
                    patternid = matcher.group()
                }
                comment = (element."comment".text())
                if (patternid != null && comment != null) {
                    id2changecomment.put(patternid,comment)
                }
            } // end finding matches
        } // end for each change-set

        return id2changecomment
    }

    /**
     * Get the raw changeset xml data
     * @return
     */
    public String getChangeSetsXml() {
        def ahptoolProc = [ahptool, "getSourceChanges"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err) // forward process stderr
        def changeSetXML = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get source changes from ahptool: $changeSetXML")
        }

        return changeSetXML
    }

    /**
     * Get the dependency plan xml for a given build life
     * @return
     */
    public String getDependencyPlanXml(String buildLifeId) {
        def ahptoolProc = [ahptool, "getDependencyPlan", "-buildlife", buildLifeId].execute();
        def errThread = ahptoolProc.consumeProcessErrorStream(System.out);
        def depXml = ahptoolProc.text;
        ahptoolProc.waitFor();
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get dependency plan from ahptool: " + depXml)
        }
        return depXml;
    }

    /**
     * Get a map of bugid to changeset
     * @param regex the regex to match comments against
     * @return a map of bugid to changeset element
     */
    public Map<String,GPathResult> getMatchingChangeSets(String regex) {
        def ahptoolProc = [ahptool, "getSourceChanges"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err) // forward process stderr
        def changeSetXML = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get source changes from ahptool: $changeSetXML")
        }
        def pattern = java.util.regex.Pattern.compile(regex)
        def Map<String, GPathResult> id2changeSet = new HashMap<String,GPathResult>()

        // will throw a parse exception at random times - any ideas?
        new XmlSlurper().parseText(changeSetXML)."change-set".each { element ->
            def matcher = pattern.matcher(element.comment.text())
            while (matcher.find()) {
                def patternid
                def comment
                if (matcher.groupCount() > 0) {
                    // they specified a '(...)' group within the pattern, use that as the bug-id
                    patternid = matcher.group(1)
                }
                else {
                    // use the whole matching substring as the bug-id
                    patternid = matcher.group()
                }
                if (patternid != null) {
                    id2changeSet.put(patternid,element)
                }
            } // end finding matches
        } // end for each change-set

        return id2changeSet
    }

    public void addReports(reportName, data) {
        def ahptoolProc = [ahptool, '-report', reportName, 'addReports', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withOutputStream{ it << data }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add report using ahptool: " + reportName)
        }
    }

    public void addSourceAnalytics(reportName, sourceAnalyticsXml) {
        def ahptoolProc = [ahptool, '-report', reportName, 'addSourceAnalytics', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withWriter{ it << sourceAnalyticsXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add Source Analytics using ahptool: " + sourceAnalyticsXml)
        }
    }

    public void addTests(reportName, testsXml) {
        def ahptoolProc = [ahptool, '-report', reportName, 'addTests', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withWriter{ it << testsXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add Test Results using ahptool: " + testsXml)
        }
    }

    /**
     *
     * @param issuesXml a Writable UTF-8 matching the issue schema
     */
    public void addIssues(issuesXml) {
        def ahptoolProc = [ahptool, 'addIssues', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withOutputStream{ procStdIn ->
            new OutputStreamWriter(procStdIn, 'utf-8').withWriter{ it << issuesXml}
        }
        //ahptoolProc.withWriter{ it << issuesXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add Issues using ahptool: " + issuesXml)
        }
    }

    public String getIssuesXml() {

        // get the changes of the build life from ahptool
        def ahptoolProc = [ahptool, "getIssues"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def issuesXml = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get build life issues from ahptool: " + issuesXml)
        }
        return issuesXml;
    }

    public Map<String,String> getAgentProperties() {
        return getProperties(AHPTool.PropertyType.AGENT)
    }


    /**
     * @since 3.8 (2011-04-11)
     * @param linksXml a Writable or String UTF-8 matching the buildlife links schema
     */
    public void addBuildLifeLinks(def linksXml) {
        executeWithXml('addBuildLifeLinks', linksXml)
    }

    /**
     * @since 3.8 (2011-04-11)
     * @return xml document of the build life links
     */
    public String getBuildLifeLinks() {
        return execute("getBuildLifeLinks");
    }

    public Map<String,String> getBuildLifeProperties() {
        return getProperties(AHPTool.PropertyType.BUILD_LIFE)
    }

    public Map<String,String> getBuildLifeProperties(def id) {
        def ahptoolProc = [ahptool, "getBuildLifeProperties", '-buildlife', id].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def propertiesXml = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get BuildLife ${id} properties from ahptool: " + propertiesXml)
        }
        def props = [:]
        new XmlSlurper().parseText(propertiesXml).property.each{ propElem ->
            props[propElem.@name.text()] = propElem.text()
        }
        return props
    }

    public Map<String,String> getOriginatingRequestProperties() {
        return getProperties(AHPTool.PropertyType.ORIGINATING_REQUEST)
    }

    public Map<String,String> getOriginatingWorkflowProperties() {
        return getProperties(AHPTool.PropertyType.ORIGINATING_WORKFLOW)
    }

    public Map<String, String> getJobProperties() {
        return getProperties(AHPTool.PropertyType.JOB)
    }

    public Map<String, String> getRequestProperties() {
        return getProperties(AHPTool.PropertyType.REQUEST)
    }

    public Map<String,String> getStepProperties() {
        return getProperties(AHPTool.PropertyType.STEP)
    }

    public Map<String,String> getSystemProperties() {
        return getProperties(AHPTool.PropertyType.SYSTEM)
    }

    public Map<String, String> getProperties(def propType) {
        def ahptoolProc = [ahptool, "get${propType.name}Properties"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def propertiesXml = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get ${propType.name} properties from ahptool: " + propertiesXml)
        }
        def props = [:]
        new XmlSlurper().parseText(propertiesXml).property.each{ propElem ->
            props[propElem.@name.text()] = propElem.text()
        }
        return props
    }

    public StepInfo getStepInfo() {
        StepInfo result = null;
        def ahptoolProc = [ahptool, "getStepInfo"].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def infoXml = ahptoolProc.text;
        ahptoolProc.waitFor();
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to get step info from ahptool: " + infoXml);
        }
        try {
            result = new StepInfo(new XmlSlurper().parseText(infoXml));
        }
        catch (Exception e) {
            throw new Exception("Failed to parse step info from ahptool : " + infoXml);
        }
        return result;
    }

    public String callGetCommand(def commandName, def args) {
        def command = [ahptool, commandName]
        command.addAll(args)
        def ahptoolProc = command.execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def output = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed executing $commandName with ahptool [$command]: " + output)
        }
        return output
    }

    public void callSetCommand(def commandName, def args, def input) {
        def command = [ahptool, commandName]
        command.addAll(args)
        command.add("-")
        def ahptoolProc = command.execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        ahptoolProc.withWriter{ it << input }
        def output = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed executing $commandName with ahptool [$command]: " + output)
        }
    }

    public void setAgentProperties(def props) {
        setProperties(props, AHPTool.PropertyType.AGENT)
    }

    public void setBuildLifeProperties(def props) {
        setProperties(props, AHPTool.PropertyType.BUILD_LIFE)
    }

    public void setJobProperties(def props) {
        setProperties(props, AHPTool.PropertyType.JOB)
    }

    public void setRequestProperties(def props) {
        setProperties(props, AHPTool.PropertyType.REQUEST)
    }

    public void setStepProperties(def props) {
        setProperties(props, AHPTool.PropertyType.STEP)
    }

    public void setSystemProperties(def props) {
        setProperties(props, AHPTool.PropertyType.SYSTEM)
    }

    protected void setProperties(def props, def propType) {
        def uploadChangeDateCmd = [ahptool, "set${propType.name}Properties", '-']

        def propXml = groovy.xml.DOMBuilder.newInstance().'properties'{
            props.each{key, value ->
                property('name':key, value)
            }
        }

        def ahptoolProc = uploadChangeDateCmd.execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withWriter{ it << propXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to set $propType.name properties using ahptool: " + propXml)
        }
        println "";
    }

    public void setQuietPeriodLatestTime(long unixTimeMillis) {
        setStepProperties(['latest-change':unixTimeMillis]);
    }

    public void setQuietPeriodLatestTime(Date date) {
        setQuietPeriodLatestTime(date.time);
    }

    public void addTestCoverage(reportName, testCoverageXml) {
        def ahptoolProc = [ahptool, '-report', reportName, 'addTestCoverage', '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withWriter{ it << testCoverageXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to add Code Coverage using ahptool: " + testCoverageXml)
        }
    }

    protected String execute(String commandName) {

        // get the changes of the build life from ahptool
        def ahptoolProc = [ahptool, commandName].execute()
        def errThread = ahptoolProc.consumeProcessErrorStream(err);
        def resultText = ahptoolProc.text
        ahptoolProc.waitFor()
        errThread.join();
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to $commandName from ahptool: " + issuesXml)
        }
        return resultText;
    }

    protected executeWithXml(String commandName, def xml) {
        def ahptoolProc = [ahptool, commandName, '-'].execute()
        ahptoolProc.consumeProcessOutput(out, err) // forward process stdout and stderr
        ahptoolProc.withOutputStream{ procStdIn ->
            new OutputStreamWriter(procStdIn, 'utf-8').withWriter{ it << xml}
        }
        //ahptoolProc.withWriter{ it << issuesXml }
        ahptoolProc.waitFor()
        if (ahptoolProc.exitValue()) {
            throw new Exception("Failed to $commandName using ahptool: " + xml)
        }
    }

    public String getWorkDir() {
        return execute('getWorkDir')
    }

    public setWorkDir(String workDir, boolean lockForWorkflow, boolean releasePrior) {
        def lockScope = lockForWorkflow ? 'worklow' : 'job'
        def workDirXml = groovy.xml.DOMBuilder.newInstance().'work-dir'('scope':lockScope, 'releasePrior':releasePrior, targetWorkDir)
        executeWithXml('setWorkDir', workDirXml)
    }

}
