       /*
        *  Licensed Materials - Property of IBM
        *  5725-G92 (C) Copyright IBM Corp. 2011, 2012. All Rights Reserved.
        *  US Government Users Restricted Rights - Use, duplication or
        *  disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
        */


package com.worklight;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.logging.Logger;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;


import com.worklight.server.auth.api.MissingConfigurationOptionException;
import com.worklight.server.auth.api.UserIdentity;
import com.worklight.server.auth.api.UserNamePasswordLoginModule;

public class LDAPLoginModule extends UserNamePasswordLoginModule {
	
	private static final Logger logger = Logger.getLogger(UsernamePasswordJSONAuthenticator.class.getName());
	private String ldapHost;
	private String ldapDomain;
	private String searchBase;
	private String memberOfMustContain;
	
	private String username;
	
	@Override
	public void init(Map<String, String> options) throws MissingConfigurationOptionException {
		logger.info("LDAPLoginModule :: init");
		
		// Active directory's configuration is retrieved from authenticationConfig.xml
		ldapHost = getConfigurationOption("ldapHost", options);
		ldapDomain = getConfigurationOption("ldapDomain", options);
		searchBase = getConfigurationOption("searchBase", options);
		memberOfMustContain = getConfigurationOption("memberOfMustContain", options);
	}
	
	private String getConfigurationOption(String key, Map<String, String> options) throws MissingConfigurationOptionException{
		String value = options.remove(key);
		logger.info("LDAPLoginModule :: getConfigurationOption :: " + key + " = " + value);
		if (null == value) throw new MissingConfigurationOptionException(key);
		return value.trim();
	}

	@Override
	public boolean login(Map<String, Object> authenticationData) {
		logger.info("LDAPLoginModule :: login");
		
		// Getting username and password from authenticator
		String user = getUserName(authenticationData);
		String pass = getPassword(authenticationData);
		
		// Creating an environment
		Hashtable<String, String> env = new Hashtable<String, String>();
		env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
		env.put(Context.PROVIDER_URL, ldapHost);
		env.put(Context.SECURITY_AUTHENTICATION, "simple");
		env.put(Context.SECURITY_PRINCIPAL, user + "@" + ldapDomain);
		env.put(Context.SECURITY_CREDENTIALS, pass);

		LdapContext ldapCtx = null;
		logger.info("LDAPLoginModule :: login :: Trying to connect to AD/LDAP server");
		try {
			// New InitialLdapContext creation success means that user was successfully authenticated
			// In case of authentication failure an exception will be thrown
			ldapCtx = new InitialLdapContext(env, null);
			logger.info("LDAPLoginModule :: login :: User found in AD/LDAP directory");
				
			// Otherwise - retrieving user attributes from AD/LDAP directory  
			ldapCtx.setRequestControls(null);
			SearchControls searchControls = new SearchControls();
		    searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);
		    searchControls.setTimeLimit(30000);
		    
		    NamingEnumeration<?> searchResults = ldapCtx.search(searchBase, "(&(objectClass=user)(sAMAccountName=" + user + "))", searchControls);

		    HashMap<String, Object> retrievedAttributes = new HashMap<String, Object>();
		    while (searchResults.hasMoreElements()){
		    	SearchResult searchResult = (SearchResult) searchResults.next();
		    	Attributes searchResultAttributes = searchResult.getAttributes();
		    	if (null != searchResultAttributes) {
					NamingEnumeration<?> attributesEnumeration = searchResultAttributes.getAll();
					while (attributesEnumeration.hasMore()) {
						Attribute attr = (Attribute) attributesEnumeration.next();
						retrievedAttributes.put(attr.getID(), attr.get());
						logger.info("LDAPLoginModule :: login :: retrievedAttribute :: " + attr.getID() + "=" + attr.get());
					}
					attributesEnumeration.close();
		    	}
		    }
		    
		    // Do memberOfMustContain check
		    String memberOfAttributeValue = (String) retrievedAttributes.get("memberOf");
			if (null == memberOfAttributeValue || !memberOfAttributeValue.contains(memberOfMustContain)){
				throw new RuntimeException("Authentication failure :: '" + memberOfAttributeValue +"' does not contain '" + memberOfMustContain + "'");
			}
			
			// All checks are complete, authentication success
			username = user;
			logger.info("LDAPLoginModule :: login :: Authentication success!");
			return true;
		}
		catch (Exception e){
			logger.info("LDAPLoginModule :: login :: Authentication failure. Reason :: " + e.getMessage());
			throw new RuntimeException("Authentication failure :: invalid username/password");
		}
	}

	@Override
	public UserIdentity createIdenity(String realm) {
		logger.info("LDAPLoginModule :: createIdenity");
		return new UserIdentity(realm, username, username, null, null, null);
	}

	@Override
	public void logout() {
		logger.info("LDAPLoginModule :: logout");
		username = null;
	}

	@Override
	public void abort() {
		logger.info("LDAPLoginModule :: abort");
		username = null;
	}

	@Override
	public LDAPLoginModule clone() throws CloneNotSupportedException {
		logger.info("LDAPLoginModule :: clone");
		return (LDAPLoginModule) super.clone();
	}

}

