//===========================================================================
//
// Module:		LNSTRING.HPP
//
// Description:	Class library definition for LNString and related classes.
//
//===========================================================================
//
// Copyright (c) 1996 Lotus Development Corporation. All rights reserved.
// This software is subject to the Lotus Software Agreement, Restricted
// Rights for U.S. government users, and applicable export regulations.
//
//===========================================================================

#ifndef LNSTRING_HPP
#define LNSTRING_HPP

#ifndef LNGLOBAL_H
#include "lnglobal.h"
#endif

extern const char * LNEMPTYSTRING;

//
//	Class definition for LNChar.
//
class LNChar : public LNNotesClass
{
friend LNSINT LNStringCompare
	( const LNChar &c1, char c2, LNSTRINGCOMPAREFLAGS flags );
friend LNSINT LNStringCompare
	( const LNChar &c1, const LNChar &c2, LNSTRINGCOMPAREFLAGS flags );

public:
	LNChar( const LNChar &other );
	LNChar( LNStringBody *string, LNINT pos );

	~LNChar();
	
	LNChar& operator=( const LNChar &other );
	LNChar& operator=( char c );
#ifndef MAC
	LNChar& operator=( wchar_t c );
#endif
	operator char() const;

	LNCLASSID GetClassID() const {return LNCLASSID_LNCHAR;}
	
	LNBOOL IsAlpha();

	LNBOOL IsAlnum();

	LNBOOL IsArith();

	LNBOOL IsCntrl();

	LNBOOL IsDigit();

	LNBOOL IsLower();

	LNBOOL IsNull();

	LNBOOL IsPunct();

	LNBOOL IsSpace();

	LNBOOL IsUpper();

	LNChar& RemoveAccent();

	LNChar& ToLower();

	LNChar& ToUpper();
	
private:
	LNChar();	// Disabled.

	char * GetTextPtr() const;

	LNStringBody *Body;
	LNINT Pos;
};

//
//	Class definition for LNString.
//
class LNString : public LNSmartPtr
{
friend class LNChar;

public:
	LNString();
	LNString( const LNString &other );
	LNString( const char *string );
	LNString( const char *string, LNINT len );
	LNString( const LNCHAR *string );
	LNString( const LNCHAR *string, LNINT len );
	LNString( LNStringBody *stringb );

	~LNString();

	LNString& operator=( const LNString &other );
	LNString& operator=( const char *string );
	LNString& operator=( const LNCHAR *string );
	LNString& operator=( const LNStringBody *stringb );

	operator const char *() const;

	LNChar operator[]( int n ) const;
	LNChar operator[]( LNINT n ) const;  

	LNString& operator+=( const LNString &string );
	LNString& operator<<( const LNString &string );

	LNSTATUS Append( const LNString &string );

	LNSTATUS Delete( LNINT pos, LNINT count=1 );
  
	LNSTATUS Find( const LNString &substr, LNINT pos, LNINT *loc ) const;

	LNSTATUS FindChar( char c, LNINT pos, LNINT *loc ) const;
#ifndef MAC
	LNSTATUS FindChar( wchar_t c, LNINT pos, LNINT *loc ) const;
#endif
	LNSTATUS FindChar( const LNString &set, LNINT pos, LNINT *loc ) const;

	LNSTATUS FindNextWordStart( LNINT pos, LNINT *loc ) const;

	LNSTATUS FindNextWordEnd( LNINT pos, LNINT *loc ) const;

	LNSTATUS FindPreviousWordStart( LNINT pos, LNINT *loc ) const;

	LNSTATUS FindPreviousWordEnd( LNINT pos, LNINT *loc ) const;

	char * GetBuf() const;

	LNINT GetBufLength() const;

	LNINT GetByteLength() const;

	LNINT GetByteCount() const {return GetBufLength();}

	LNINT GetCharacterCount() const {return GetLength();}

	LNCLASSID GetClassID() const {return LNCLASSID_LNSTRING;}

	LNINT GetLength() const;

	LNINT GetPlatformByteCount() const {return GetByteLength();}

	LNSTATUS GetSubString( LNINT pos, LNINT count, LNString *dest ) const;

	char * GetTextPtr() const {return GetBuf();}

	LNSTATUS Insert( LNINT pos, const LNString &string );

	LNBOOL IsNull() const;

	void RemoveAccent();

	LNSTATUS Replace( LNINT pos, const LNString &string );
	LNSTATUS Replace( LNINT pos, LNINT count, const LNString &string );

	void ToLower();

	void ToUpper();

private:
	void PrepareToModify();
};


//---------------------------------------------------------------------------
//	Relational operators to compare two strings or two characters. The
//	comparison is case and accent insensitive. Note that to avoid ambiguous
//	conversions between the overloaded operators and the built-in versions,
//	each combination of LNString/LNChar and char */char needs to be explicitly
//	defined.
//---------------------------------------------------------------------------

inline LNBOOL operator==( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) == 0;
}

inline LNBOOL operator==( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) == 0;
}

inline LNBOOL operator==( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) == 0;
}

inline LNBOOL operator==( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) == 0;
}

inline LNBOOL operator==( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == 0;
}

inline LNBOOL operator==( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == 0;
}

//---------------------------------------------------------------------------

inline LNBOOL operator!=( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) != 0;
}

inline LNBOOL operator!=( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) != 0;
}

inline LNBOOL operator!=( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) != 0;
}

inline LNBOOL operator!=( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) != 0;
}

inline LNBOOL operator!=( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) != 0;
}

inline LNBOOL operator!=( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) != 0;
}
//---------------------------------------------------------------------------

inline LNBOOL operator<( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) == -1;
}

inline LNBOOL operator<( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) == -1;
}

inline LNBOOL operator<( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) == -1;
}

inline LNBOOL operator<( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) == -1;
}

inline LNBOOL operator<( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == -1;
}

inline LNBOOL operator<( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == -1;
}

//---------------------------------------------------------------------------

inline LNBOOL operator<=( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) <= 0;
}

inline LNBOOL operator<=( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) <= 0;
}

inline LNBOOL operator<=( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) <= 0;
}

inline LNBOOL operator<=( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) <= 0;
}

inline LNBOOL operator<=( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) <= 0;
}

inline LNBOOL operator<=( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) <= 0;
}

//---------------------------------------------------------------------------

inline LNBOOL operator>( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) == 1;
}

inline LNBOOL operator>( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) == 1;
}

inline LNBOOL operator>( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) == 1;
}

inline LNBOOL operator>( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) == 1;
}

inline LNBOOL operator>( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == 1;
}

inline LNBOOL operator>( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) == 1;
}

//---------------------------------------------------------------------------

inline LNBOOL operator>=( const LNString &string1, const char *string2 )
{
	return LNStringCompare(string1, LNString(string2)) >= 0;
}

inline LNBOOL operator>=( const char *string1, const LNString &string2 )
{
	return LNStringCompare(LNString(string1), string2) >= 0;
}

inline LNBOOL operator>=( const LNString &string1, const LNString &string2 )
{
	return LNStringCompare(string1, string2) >= 0;
}

inline LNBOOL operator>=( const LNChar &c1, char c2 )
{
	return LNStringCompare(c1, c2) >= 0;
}

inline LNBOOL operator>=( char c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) >= 0;
}

inline LNBOOL operator>=( const LNChar &c1, const LNChar &c2 )
{
	return LNStringCompare(c1, c2) >= 0;
}

#endif

