//===========================================================================
//
// Module:  LNRTOBJ.HPP
//
// Description:
//
//  Class library definition for LNRTObject, LNRTObjectBody and related classes.
//	(The core rich text editing engine).
//
//  LNRTContainer is defined in this file, but the rest of container objects
//	are defined in lnrtcont.hpp.
//
//===========================================================================
//
// Copyright (c) 1996 Lotus Development Corporation. All rights reserved.
// This software is subject to the Lotus Software Agreement, Restricted
// Rights for U.S. government users, and applicable export regulations.
//
//===========================================================================

#ifndef LNRTOBJ_HPP
#define LNRTOBJ_HPP


// Exposed Rich Text Types

typedef long LNRTTYPE;

const LNRTTYPE LNRTTYPE_OBJECT			= 1;
const LNRTTYPE LNRTTYPE_ELEMENT			= (1L << 1) + LNRTTYPE_OBJECT;
const LNRTTYPE LNRTTYPE_COMPOSITE_DATA	= (1L << 2) + LNRTTYPE_ELEMENT;
const LNRTTYPE LNRTTYPE_FORM_FIELD		= (1L << 3) + LNRTTYPE_ELEMENT;
const LNRTTYPE LNRTTYPE_CHAR			= (1L << 4) + LNRTTYPE_ELEMENT;
const LNRTTYPE LNRTTYPE_CONTAINER		= (1L << 5) + LNRTTYPE_OBJECT;
const LNRTTYPE LNRTTYPE_STYLIZED_TEXT	= (1L << 6) + LNRTTYPE_CONTAINER;
const LNRTTYPE LNRTTYPE_GRAPHIC			= (1L << 7) + LNRTTYPE_CONTAINER;
const LNRTTYPE LNRTTYPE_PARAGRAPH		= (1L << 8) + LNRTTYPE_CONTAINER;
const LNRTTYPE LNRTTYPE_ITEM			= (1L << 9) + LNRTTYPE_CONTAINER;


// Macro to determine if type1.IsType(type2)
#define ISRTTYPE(type1,type2) ((type1 & type2) == type2)

LNBOOL IsRTType(LNRTTYPE type1, LNRTTYPE type2);


typedef LNINT LNRTOBJECTBODYFLAGS;
typedef LNINT LNRTSEARCHFLAGS;

#define LNRTSEARCHFLAGS_DEFAULT 0
#define LNRTSEARCHFLAGS_MATCH_ACCENT INTL_ACCENT_SENSITIVE
#define LNRTSEARCHFLAGS_MATCH_CASE INTL_CASE_SENSITIVE



class LNRTObject : public LNSmartPtr
{
	friend class LNBitmapBody;
	friend class LNCompositeData;
	friend class LNGraphicBody;
	friend class LNParagraphBody;
	friend class LNRichText;
	friend class LNRichTextBody;
	friend class LNRTContainer;
	friend class LNRTContainerBody;
	friend class LNRTCursorBody;
	friend class LNRTElement;
	friend class LNRTInputParser;
	friend class LNRTObjectBody;
	friend class LNRTOutputStream;
	friend class LNStylizedText;
	friend class LNStylizedTextBody;
	friend class LNTableBody;


public:
	
	LNRTObject();
	LNRTObject(const LNRTObject &object);

	~LNRTObject();

	LNRTObject& operator=(const LNRTObject &object);
	
	LNBOOL operator==(const LNRTObject &object) const;

	LNSTATUS Copy(const LNRTObject &object);

	virtual LNCLASSID GetClassID() const { return LNCLASS_LNRTOBJECT; }
	
	virtual LNSTATUS GetText(LNString *buf) const;

	LNRTTYPE GetType() const;
	
	virtual LNBOOL IsType(LNRTTYPE type) const;

protected:

	LNRTObject(WORD cdsig, HANDLE pool, BLOCK block, DWORD cdsize, const LNFontTable &fontTable);
	LNRTObject(LNRTObjectBody *body);
	LNRTObject(LNStylizedTextBody *stylizedTextBody, LNINT offset);

	LNRTObject& operator=(LNRTObjectBody *body);
	LNRTObject& operator=(LNRTContainerBody *body);
	LNRTObject& operator=(LNStylizedTextBody *body);

	virtual LNSTATUS AppendChild(LNRTObject *child);

	virtual LNSTATUS CopyRTObject(const LNRTObject &object, LNBOOL copychildren);

	/* SDK END */
	// TBD - the LNRTObject find methods may not need to be virtual, just the LNRTObjectBody methods.

	virtual LNSTATUS FindContainer(LNRTTYPE type, LNRTContainer *container) const;

	virtual LNSTATUS FindFirst(LNRTTYPE type, LNRTObject *object) const;
	
	virtual LNSTATUS FindFirstInChildren(LNRTTYPE type,
					    				LNBOOL *bFound,
					    				LNRTObject *nextobj) const;

	LNSTATUS FindFirstInEachNextSibling(LNRTTYPE type,
					    				LNBOOL *bFound,
					    				LNRTObject *nextobj) const;

	virtual LNSTATUS FindLast(LNRTTYPE type, LNRTObject *object) const;
	
	virtual LNSTATUS FindLastInChildren(LNRTTYPE type,
					    				LNBOOL *bFound,
					    				LNRTObject *prevobj) const;

	LNSTATUS FindLastInEachPrevSibling(LNRTTYPE type,
					    				LNBOOL *bFound,
					    				LNRTObject *prevobj) const;

	LNSTATUS FindNext(const LNRTObject &bound, 
							  			LNRTTYPE type, 
							  			LNRTObject *nextobj) const;
	
	virtual LNSTATUS FindNext(const LNString &text, 
										LNRTSEARCHFLAGS options, 
										LNRTObject *object,
										LNINT *offset) const;

	LNSTATUS FindPrevious(const LNRTObject &bound,
										LNRTTYPE type, 
										LNRTObject *prevobj) const;

	virtual LNSTATUS FindPrevious(const LNString &text, 
		              					LNRTSEARCHFLAGS options, 
				      					LNRTObject *object,
				      					LNINT *offset) const;



	LNSTATUS GetClosestCommonAncestor(const LNRTObject &other, LNRTObject *ancestor) const;

	virtual LNINT GetEndIndexValue() const;

	virtual LNINT GetIndexValue() const;

	LNSTATUS GetNextSibling(LNRTObject *object) const;

	virtual LNINT GetOffset() const;

	LNSTATUS GetParentContainer(LNRTContainer *container) const;
	
	LNSTATUS GetPreviousSibling(LNRTObject *object) const;
	
	virtual LNSTATUS GetRichTextItem(LNRichText *richtext) const;

	LNRTObjectBody* GetRTObjectBody() const { return (LNRTObjectBody *) Body; }

   	virtual LNINT GetSize() const;

	LNSTATUS GetTopParentContainer(LNRTContainer *container) const;

	LNBOOL HasCommonAncestor(const LNRTObject &other) const;

	virtual LNBOOL HasOffset() const;

	LNBOOL HasNextSibling() const;	

	LNBOOL HasParentContainer() const;	
	
	LNBOOL HasPreviousSibling() const;	

	LNSTATUS IndexAllChildren();

	LNSTATUS InsertNextSibling(LNRTObject *object);

	LNSTATUS InsertPreviousSibling(LNRTObject *object);

	
	LNBOOL IsBoundedBy(const LNRTObject &object) const;

   	LNBOOL IsContainedBy(const LNRTContainer &ancestor) const;
	LNBOOL IsContainedBy(const LNRTContainerBody *ancestor) const;
	
	LNBOOL IsDirty() const;

	LNBOOL IsLeftOf(const LNRTObject &object) const;

	virtual LNBOOL IsValid() const;

	LNSTATUS RemoveThis();

	virtual LNSTATUS SetDirty(LNBOOL dirty = TRUE);

	void SetEndIndexValue(LNINT value);

	void SetIndexValue(LNINT value);

	// These only work if the object happens to be an (internal) LNRTChar.
	virtual void SetOffset(LNINT offset);
	
	virtual void SetRTChar(LNStylizedTextBody *stylizedTextBody, LNINT offset);

	virtual LNSTATUS Update( LNINT offset, LNINT *newoffset);
		
private:


	LNINT Offset; // 1-based
};



class LNRTContainer : public LNRTObject
{
	friend class LNRichTextBody;
	friend class LNRTContainerBody;
	friend class LNRTCursorBody;
	friend class LNRTInputParser;
	friend class LNRTObject;
	friend class LNRTObjectBody;
	friend class LNStylizedTextBody;


public:
	
	LNRTContainer();
	LNRTContainer(const LNRTObject &object);
	LNRTContainer(const LNRTContainer &container);

	~LNRTContainer();
	
	LNRTContainer& operator=(const LNRTObject &object);
	LNRTContainer& operator=(const LNRTContainer &container);

	LNRTContainer& operator<<(const LNString &text);
	LNRTContainer& operator<<(const LNRichText &richtext);
	LNRTContainer& operator<<(const LNRTObject &object);

	virtual LNCLASSID GetClassID() const { return LNCLASS_LNRTCONTAINER; }
	
	LNSTATUS Append(const LNString &text);
	LNSTATUS Append(const LNRichText &richtext);
	LNSTATUS Append(const LNRTObject &object);

	LNSTATUS Delete(LNRTCursor *start, LNRTCursor *end);
	LNSTATUS Delete(LNRTCursor *start, LNINT poscount);

	LNSTATUS GetCursor(LNRTCursor *cursor) const;					 
	LNSTATUS GetEndCursor(LNRTCursor *cursor) const;					 

	LNSTATUS GetRichText(const LNRTCursor &start, const LNRTCursor &end,
		LNRichText *richtext) const;
	LNSTATUS GetRichText(const LNRTObject &object,
		LNRichText *richtext) const;

	virtual LNSTATUS GetText(LNString *buf) const;

	LNSTATUS Insert(const LNString &text, LNRTCursor *cursor);

	LNSTATUS SetFontStyle(const LNRTCursor &start, const LNRTCursor &end,
		const LNFontStyle &style, LNSETFONTSTYLEFLAGS mask=LNSETFONTSTYLEFLAGS_ALL);
	LNSTATUS SetFontStyle(const LNRTCursor &cursor, const LNFontStyle &style); 

protected:

	LNRTContainer& operator=(LNRTContainerBody *body);

	virtual LNSTATUS CopyRTObject(const LNRTObject &object, LNBOOL copychildren);

	LNSTATUS Delete(LNRTObject *object);

	virtual LNSTATUS FindFirstInChildren(LNRTTYPE type,
					    LNBOOL *bFound,
					    LNRTObject *nextobj) const;

	virtual LNSTATUS FindLastInChildren(LNRTTYPE type,
					    LNBOOL *bFound,
					    LNRTObject *prevobj) const;

	LNSTATUS GetFirstChild(LNRTObject *object) const;

	LNSTATUS GetRichText(const LNRTCursor &start, LNINT poscount,
		LNRichText *richtext) const;
	LNSTATUS GetRichText(const LNRTCursor &cursor, LNRTTYPE type,
		LNRichText *richtext) const;

	LNRTContainerBody* GetRTContainerBody() const { return (LNRTContainerBody *) Body; }

	LNSTATUS Insert(const LNRichText &richtext, LNRTCursor *cursor);
	LNSTATUS Insert(const LNRTObject &object, LNRTCursor *cursor);

	virtual LNSTATUS SetTopContainer(LNRichTextBody *rtbody); // Called from LNRichTextBody only
};




typedef struct
{
	LNRTObject StartRTObject; 
	LNRTObject EndRTObject;
	LNRTObject ClosestCommonAncestor; 
	LNRTObject ObjectAfterSelection;
} LNRTSelection;



#endif

