//===========================================================================
//
// Module:		LNDB.HPP
//
// Description:	Class library definition for the LNDatabase class.
//
//===========================================================================
//
// Copyright (c) 1996 Lotus Development Corporation. All rights reserved.
// This software is subject to the Lotus Software Agreement, Restricted
// Rights for U.S. government users, and applicable export regulations.
//
//===========================================================================

#ifndef LNDB_HPP
#define LNDB_HPP

#ifndef LNGLOBAL_H
#include "lnglobal.h"
#endif

// Full-text sorting options used by LNFTSearchOptions::Get/SetSortOrder

#ifndef FT_DEFS
#include "ft.h"
#endif

typedef DWORD LNFTSEARCHORDER;

#define LNFTSEARCHORDER_BY_RELEVANCE FT_SEARCH_SCORES
#define LNFTSEARCHORDER_NEWEST_FIRST FT_SEARCH_SORT_DATE
#define LNFTSEARCHORDER_OLDEST_FIRST \
			(FT_SEARCH_SORT_DATE | FT_SEARCH_SORT_ASCEND)

class LNSelectiveReplication {};

//
//	Class definition for LNSearchOptions.
//
class LNSearchOptions : public LNNotesClass
{

public:
	LNSearchOptions();

	LNSearchOptions( const LNSearchOptions &other );

	~LNSearchOptions();

	LNSearchOptions & operator=( const LNSearchOptions &other );

	LNDatetime GetBeginDate() const;

	virtual LNCLASSID GetClassID() const {return LNCLASSID_LNSEARCHOPTIONS;}

	LNDatetime GetCompletedDate() const;

	LNDatetime GetEndDate() const;

	LNINT GetMaxNotes() const;

	LNNOTETYPE GetNoteType() const;

	LNSTATUS SetBeginDate( const LNString &begindate );
	LNSTATUS SetBeginDate( const LNDatetime &begindate );

	LNSTATUS SetEndDate( const LNString &enddate );
	LNSTATUS SetEndDate( const LNDatetime &enddate );

	LNSTATUS SetMaxNotes( LNINT maxnotes );

	LNSTATUS SetNoteType( LNNOTETYPE notetype );

private:
	LNSTATUS SetCompletedDate( TIMEDATE *completeddate );

	TIMEDATE BeginDate;
	TIMEDATE CompletedDate;
	TIMEDATE EndDate;
	LNINT MaxNotes;
	LNNOTETYPE NoteType;
};

//
//	Class definition for LNFTIndexStatistics.
//
class LNFTIndexStatistics : public LNNotesClass
{

public:
	LNFTIndexStatistics();

	LNFTIndexStatistics( const LNFTIndexStatistics &other );

	~LNFTIndexStatistics();

	LNFTIndexStatistics & operator=( const LNFTIndexStatistics &other );

	LNCLASSID GetClassID() const {return LNCLASSID_LNFTINDEXSTATISTICS;}

	LNINT GetNumberOfBytesIndexed() const;

	LNINT GetNumberOfDocsAdded() const;

	LNINT GetNumberOfDocsDeleted() const;

	LNINT GetNumberOfDocsUpdated() const;

private:
	void Assign( FT_INDEX_STATS stats );

	FT_INDEX_STATS Statistics;
};

//
//	Class definition for LNFTSearchOptions.
//
class LNFTSearchOptions : public LNNotesClass
{

public:
	LNFTSearchOptions();

	LNFTSearchOptions( const LNFTSearchOptions &other );

	~LNFTSearchOptions();

	LNFTSearchOptions & operator=( const LNFTSearchOptions &other );

	LNCLASSID GetClassID() const {return LNCLASSID_LNFTSEARCHOPTIONS;}

	LNINT GetMaxDocuments() const;

	LNFTSEARCHORDER GetSortOrder() const;

	LNBOOL GetStemWords() const;

	LNBOOL GetThesaurusWords() const;

	LNSTATUS SetMaxDocuments( LNINT maxdocuments );

	LNSTATUS SetSortOrder( LNFTSEARCHORDER sortorder );

	void SetStemWords( LNBOOL stemwords );

	void SetThesaurusWords( LNBOOL theswords );

private:
	DWORD GetOptions() const {return Options;}

	WORD MaxDocuments;
	DWORD Options;
};

//
//	Class definition for LNDatabase.
//
class LNDatabase : public LNSmartPtr
{
friend class LNNotesSession;
friend class LNNote;
friend class LNNoteArray;

public:
	LNDatabase() : LNSmartPtr() {}
	LNDatabase( const LNDatabase &other ) : LNSmartPtr(other) {}

	~LNDatabase() {}
	
	LNDatabase & operator=( const LNDatabase &other );

	LNBOOL AgentExists( const LNString &agentname ) const;

	LNSTATUS DeleteAgent( const LNString &agentname );
	LNSTATUS DeleteAgent( LNAgent *agent );

	LNSTATUS GetAgent( const LNString &agentname, LNAgent *agent );
	LNSTATUS GetAgent( const NOTEID idnote, LNAgent *agent );
	
	LNSTATUS GetAgents( LNAgentArray *agents );
	
	LNSTATUS CreateDocument( LNDocument *newdoc );
	LNSTATUS CreateDocument( LNDocument &document, LNDocument *newdoc );
	
	LNSTATUS DeleteDocument( LNDocument *document );
	
	LNSTATUS GetDocument( const NOTEID idnote, LNDocument *doc );
	LNSTATUS GetDocuments( LNDocumentArray *docs );

	LNSTATUS DeleteForm( const LNString &formname )
		{return DeleteFormXYZ(formname);}
	LNSTATUS DeleteForm( LNForm *form )
		{return DeleteFormXYZ(form);}
	
	LNBOOL FormExists( const LNString &formname ) const;
	
	LNSTATUS GetForm( const LNString &formname, LNForm *form )
		{return GetFormXYZ(formname,form);}
	LNSTATUS GetForm( const NOTEID idnote, LNForm *form );
	
	LNSTATUS GetForms( LNFormArray *forms );
	
	LNSTATUS CreateNote( LNNOTETYPE notetype, LNNote *newnote );
	LNSTATUS CreateNote( LNNote &note, LNNote *newnote );
	
	LNSTATUS DeleteNote( LNNote *note );

	LNSTATUS GetNote( const NOTEID idnote, LNNote *note );

	LNSTATUS DeleteViewFolder( const LNString &viewname );
	LNSTATUS DeleteViewFolder( LNViewFolder *view );
	
	LNSTATUS GetViewFolder( const LNString &viewname, LNViewFolder *view );
	LNSTATUS GetViewFolder( const NOTEID idnote, LNViewFolder *view );
	
	LNSTATUS GetViewFolders( LNViewFolderArray *views );
	
	LNBOOL ViewFolderExists( const LNString &viewname ) const;
	
	LNSTATUS CreateFTIndex( LNFTINDEXFLAGS options, const LNString &stopfile,
		LNFTIndexStatistics *stats=0 );
		
	LNSTATUS DeleteFTIndex();
	
	LNSTATUS GetFTIndexFlags( LNFTINDEXFLAGS *options );
	
	LNSTATUS GetLastFTIndexed( LNDatetime *lastindex );
	
	LNBOOL IsFTIndexed();
	
	LNSTATUS FTSearch( const LNString &query, LNDocumentArray *results ) const;

	LNSTATUS FTSearch
		( const LNString &query, LNDocumentArray *results,
		  const LNFTSearchOptions &options ) const;

	LNSTATUS FTSearch
		( const LNString &query, LNDocumentArray *results,
		  const LNFTSearchOptions &options, const LNNoteArray &input ) const;
	
	LNSTATUS Search( const LNString &formula, LNNoteArray *results,
		LNSearchOptions *options=0 );
		
	LNSTATUS UpdateFTIndex( LNFTIndexStatistics *stats = 0 );
	
	LNSTATUS Close();
	
	LNSTATUS GetACL( LNACL *acl );
	
	virtual LNCLASSID GetClassID() const {return LNCLASSID_LNDATABASE;}
	
	LNDatetime GetCreated();

	LNText GetDatabaseCatalogCategories();
	
	TIMEDATE * GetDatabaseID();
	
	LNDBOPENFLAGS GetDatabaseOpenFlags() const;
	
	LNDatetime GetDataLastModified();
	
	LNDatetime GetDesignLastModified();
	
	LNBOOL GetDisableBackgroundAgents();
	
	LNBOOL GetDisableReplication();
	
	LNBOOL GetNeverReplicate();

	LNString GetFilename() const;
	
	LNString GetFilepath() const;
	
	LNBOOL GetHideDesign();
	
	LNString GetInheritsFromTemplateName();
	
	LNDatetime GetLastModified();
	
	LNBOOL GetListInDatabaseCatalog();
	
	LNNUMBER GetPercentageUsed();
	
	LNString GetPort() const;
	
	LNINT GetRemoveOldDocumentsDays();
	
	LNBOOL GetRemoveOldDocuments();
	
	TIMEDATE * GetReplicaID();
	
	LNBOOL GetReplicateReceiveSummaries();
	
	LNBOOL GetReplicateSendDeletions();
	
	LNBOOL GetReplicateSendTitleAndCatalogInfo();
	
	LNREPLICATIONPRIORITY GetReplicationPriority();
	
	LNString GetServer() const;

	LNBOOL GetShowInOpenDatabaseDialog();
	
	LNINT GetSize();
	
	LNString GetTemplateName();
	
	LNString GetTitle();
	
	LNDBTYPE GetType();
	
	LNBOOL IsLocal() const { return !IsRemote(); }

	LNBOOL IsOpen() const;
	
	LNBOOL IsRemote() const;

	LNSTATUS Open();
	
	LNSTATUS Open( LNDBOPENFLAGS flags );
	
	LNSTATUS SetDatabaseCatalogCategories( const LNText &txtcategories );
	
	LNSTATUS SetDisableBackgroundAgents( LNBOOL flag );
	
	LNSTATUS SetDisableReplication( LNBOOL flag );
	
	LNSTATUS SetNeverReplicate( LNBOOL flag );

	LNSTATUS SetHideDesign( LNBOOL flag );
	
	LNSTATUS SetInheritsFromTemplateName( const LNString &name );
	
	LNSTATUS SetListInDatabaseCatalog( LNBOOL flag );
		
	LNSTATUS SetRemoveOldDocumentsDays( LNINT olddocdays );
	
	LNSTATUS SetRemoveOldDocuments( LNBOOL flag );
	
	LNSTATUS SetReplicateReceiveSummaries( LNBOOL flag );
	
	LNSTATUS SetReplicateSendDeletions( LNBOOL flag );
	
	LNSTATUS SetReplicateSendTitleAndCatalogInfo( LNBOOL flag );
	
	LNSTATUS SetReplicationPriority( LNREPLICATIONPRIORITY reppriority );
	
	LNSTATUS SetShowInOpenDatabaseDialog( LNBOOL flag );
	
	LNSTATUS SetTemplateName( const LNString &name );

	LNSTATUS SetTitle( const LNString &title );
	
private:
	LNDatabase & operator=( LNDatabaseBody *datab );

	LNSTATUS CreateDatabaseBody( LNNotesSession *session,
		const LNString &filename, const LNString &server );
	
	LNSTATUS DeleteFormXYZ( const LNString &formname );
	LNSTATUS DeleteFormXYZ( LNForm *form );

	LNDatabaseBody * GetDatabaseBody() const {return (LNDatabaseBody *)Body;}
	
	LNSTATUS GetFormXYZ( const LNString &formname, LNForm *form );
};	


#ifndef LNNOTE_HPP
#include "lnnote.hpp"
#endif
	
#endif	//LNDB_HPP

