//===========================================================================
//
// Module:	LNCOMMON.HPP
//
// Description:
//
// 	Class library definition for common classes used by all C++ API classes.
//
//===========================================================================
//
// Copyright (c) 1996 Lotus Development Corporation. All rights reserved.
// This software is subject to the Lotus Software Agreement, Restricted
// Rights for U.S. government users, and applicable export regulations.
//
//===========================================================================

#ifndef LNCOMMON_HPP
#define LNCOMMON_HPP

#ifndef LNGLOBAL_H
#include "lnglobal.h"
#endif

//---------------------------------------------------------------------------
// 	Class name:	LNNotesClass
//
//	Description:
//
//	Abstract base class for all classes in Notes C++ API.
//
//---------------------------------------------------------------------------

class LNNotesClass
{

public:
	virtual ~LNNotesClass();
	static void operator delete( void *object )
		{ LNDelete(object); }
	static void * operator new( size_t size )
		{ return LNNew(size); }	
	virtual LNSTATUS Close() { return LNNOERROR; }
	virtual LNCLASSID GetClassID() const = 0;
	const char * GetClassName()
		{ return GetClassNameText(); }
	virtual const char * GetClassNameText();
	virtual LNNotesSession * GetSession() const
		{ return Session; }
	virtual LNBOOL IsBody() const { return FALSE; }
	virtual LNBOOL IsOnHeap() const;

protected:
	LNNotesClass();
	LNNotesClass( const LNNotesSession *session );
	LNNotesClass( const LNNotesClass &object );
	LNNotesClass & operator=( const LNNotesClass &object );

private:
	LNElement *Element;
	LNNotesSession *Session;

#ifdef LNNOTESCLASS_OBJECT_LIST

	LNNotesClass *Next, *Previous;

#endif

};

//---------------------------------------------------------------------------
// 	Class name:	LNNumber
//
//	Description:
//
//	LNNumber represents a Notes LNNUMBER (double), and provides functions to
//	convert and manipulate LNNUMBER data.
//---------------------------------------------------------------------------

class LNNumber
{

public:
	LNNumber() { Value = 0; }
	LNNumber( const LNNUMBER &number ) { Value = number; }
	LNNumber( const LNString &string, LNSTATUS *error = 0 );
	LNNumber( const LNNumber &number ) { Value = number.Value; }
	~LNNumber() {}

	LNNumber & operator=( const LNNumber &number )
		{ if (&number != this) Value = number.Value; return *this; }
	LNNumber & operator=( const LNNUMBER &number )
		{ Value = number; return *this; }
	LNNumber & operator=( const LNString &string );
	
	operator LNNUMBER() const { return Value; }
	
	LNSTATUS GetText( LNString *text ) const;
	LNSTATUS GetText
		( const LNNumberFormat &fmt, const LNInternational &ifmt,
		  LNString *text ) const;
	LNNUMBER GetValue() const { return Value; }

private:
	LNNUMBER Value;

	LNSTATUS ConvertString( const LNString &string );
	LNSTATUS GetText
		( LNString *text, LNNumberFormat *fmt, LNInternational *ifmt ) const;

};

//---------------------------------------------------------------------------
// 	Class name:	LNNumberFormat
//
//	Description: Display format for number fields in a form or view.
//
//---------------------------------------------------------------------------

class LNNumberFormat : public LNNotesClass
{
friend class LNNumber;

public:
	LNNumberFormat();
	LNNumberFormat( const LNNumberFormat &other );
	~LNNumberFormat() {}
	LNNumberFormat & operator=( const LNNumberFormat &other );
	LNINT GetDigits() const
		{return (LNINT)NumberFormat.Digits;}
	LNNUMBERFORMAT GetFormat() const
		{return (LNNUMBERFORMAT)NumberFormat.Format;}
	LNBOOL GetParentheses() const;
	LNBOOL GetPercentage() const;
	LNBOOL GetPunctuated() const;
	LNSTATUS SetDigits( LNINT digits );
	LNSTATUS SetFormat( LNNUMBERFORMAT format );
	void SetParentheses( LNBOOL parentheses );
	void SetPercentage( LNBOOL percentage );
	void SetPunctuated( LNBOOL punctuated );

private:
	NFMT * GetNFMT()
	{return &NumberFormat;}
	
	NFMT NumberFormat;	// Structure that contains all number display information.
};

//---------------------------------------------------------------------------
// 	Class name:	LNObjectList
//
//	Description:
//
//	LNObjectList manages an unsorted doubly-linked list of objects, and is used
//	by classes that need to track objects that they've allocated, so that they
//	can destroy those objects later. This class is for internal use only.
//
//---------------------------------------------------------------------------

class LNObjectList : public LNNotesClass
{

public:
	LNObjectList();
	~LNObjectList();

	virtual LNSTATUS AddObject( LNNotesClass *object );
	void DeleteAllObjects();
	LNSTATUS DeleteObject( LNNotesClass *object );
	virtual LNCLASSID GetClassID() const
		{ return LNCLASSID_LNOBJECTLIST; }
	LNINT GetCount() const
		{ return Count; }
	LNNotesClass * GetFirst() const;
	LNNotesClass * GetLast() const;
	LNNotesClass * GetNext( const LNNotesClass * object ) const;
	LNNotesClass * GetPrevious( const LNNotesClass * object ) const;
	LNSTATUS RemoveObject( LNNotesClass *object );

protected:
	LNINT Count;
	LNElement *First, *Last;

	void AddElement( LNElement *element );
	LNSTATUS CheckObject( const LNNotesClass *object ) const;
	void DeleteElement( LNElement *element );
	LNSTATUS GetElement
		( const LNNotesClass *object, LNElement **element ) const;

	// copy constructor and assignment operator are undefined to prevent
	// anyone from using them

private:
	LNObjectList( const LNObjectList &list );
	LNObjectList & operator=( const LNObjectList &list );
};

//---------------------------------------------------------------------------
// 	Class name:	LNSmartPtr
//
//	Description:
//
//	LNSmartPtr is the base class for all smart pointer classes. Note that the
//	copy constructor and assignment operator in each derived class must invoke
//	the base class versions. Note also that LNSmartPtr does not inherit from
//	LNNotesClass, to keep it as small as possible.
//---------------------------------------------------------------------------

class LNSmartPtr
{
public:
	LNSmartPtr() { Body = 0; }
	LNSmartPtr( LNBody *body );
	LNSmartPtr( const LNSmartPtr &ptr );
	~LNSmartPtr() { DeleteBody(); }
	LNSmartPtr & operator=( const LNSmartPtr &ptr );
	LNSmartPtr & operator=( LNBody *body );
	virtual LNBOOL DeleteBody();
	virtual LNCLASSID GetClassID() const = 0;
	const char * GetClassName()
		{ return GetClassNameText(); }
	virtual const char * GetClassNameText();
	virtual LNNotesSession * GetSession() const
		{ return (Body ? ((LNNotesClass *) Body)->GetSession() : 0); }
	LNBOOL IsNull() const { return Body == 0; }

protected:
	LNBody *Body;
};


#endif	//LNCOMMON_HPP

