/***********************************************************************
IE_EX4.C: Import/Export example program #4

Copyright (c) 1992-8 by Lotus Development Corporation, an IBM subsidiary.
All rights reserved.

This source code, any derivatives of it, and the resulting object code
may be freely used and/or copied, so long as this copyright notice is
included and remains unmodified.  Lotus Development Corporation does not
make any representation or warranty, express or implied with respect to
this source code.  Lotus Development Corporation disclaims any liability
whatsoever for any use of such code.

************************************************************************

    This program is an example of how to parse an Import/Export message
    format file.  It is assumed that the file being imported was
    created using Export with the ITEMSIZE and END/1 parameters and
    without the FILES/DETACH parameter, which means that message
    section sizes are always present, the file contains exactly one
    message, and file items are written into the main file.  Because
    the file is assumed to have come from Export, the program can make
    additional assumptions, such as the fact that a primary text item
    will have a "Text item:" line preceeding it.  This program also does
    not handle graphics or fax items, but it does handle forward items.

    The actual function of the program is fairly useless in order to
    keep it to a manageble size.  Each of the header fields that is
    understood is simply written to the standard output, as are the
    item types and titles.  The actual items are simply skipped.  The
    usage is:
        ie_ex4 infile

            where infile is the input file

    IE_EX4 returns an ERRORLEVEL of 0 for success and 1 for any error.

    Build with Microsoft C version 5.1 or later using any memory model.
    Specify at least 4KB of stack space for the resulting program.
***********************************************************************/


// Include file(s)
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Define constants
#define BUFFSIZE    1024        // Size of various buffers
#define CTRLZ       '\x1A'      // Ctrl-Z character
#define TITLSIZE    60          // Item title size

// Define return statuses for matchLine, parseFwdLine, and parseHdrLine
#define STAT_UNKNWN 0           // Pattern unknown or not found
#define STAT_FOUND  1           // Pattern found (and not "special")
#define STAT_FNDSP1 2           // "Special" pattern #1 found
#define STAT_FNDSP2 3           // "Special" pattern #2 found
#define STAT_FNDERR 4           // Pattern found with error

// Declare prototypes
int matchLine(char *inBuff, char *patt, char *outBuff, unsigned
              outBuffSiz);
int parseFwdLine(char *line);
int parseHdrLine(char *line);
int parseSize(char *buff, long *sizeP);


/***********************************************************************
IE_EX4 main routine (described above)
***********************************************************************/
int main(int argc, char *argv[])
{
    char buff[BUFFSIZE], title[TITLSIZE + 1];
    FILE *fpIn;
    int c, state;
    long fileLen, frwdLen, ignrLen, itemLen;
    unsigned buffUsed;

    // Check argument count
    if (argc != 2) {
        fprintf(stderr, "Usage: ie_ex4 infile\n");
        return  1;
    }

    // Open input file (binary mode)
    if ((fpIn = fopen(argv[1], "rb")) == NULL) {
        fprintf(stderr, "ie_ex4: Error opening %s\n", argv[1]);
        return  1;
    }

    // Determine input file length
    if (    fseek(fpIn, 0L, SEEK_END)       ||
            (fileLen = ftell(fpIn)) == -1L  ||
            fseek(fpIn, 0L, SEEK_SET)       ) {
        fprintf(stderr,
            "ie_ex4: Error determining input file length\n");
        fclose(fpIn);
        return  1;
    }

    // Loop for input, one line at a time...
    state = 0;
    frwdLen = 0L;
    while (fileLen) {

        // Get a line from the input file
        buffUsed = 0;
        while (fileLen--) {
            if ((c = fgetc(fpIn)) == EOF) {
                fprintf(stderr, "ie_ex4: Error reading input file\n");
                fclose(fpIn);
                return  1;
            }
            if (frwdLen) {
                frwdLen--;
            }
            if (c == '\n') {
                break;
            }
            if (c != '\r'  &&  c != '\0'  &&  buffUsed < BUFFSIZE) {
                buff[buffUsed++] = (char)c;
            }
        }

        // Remove trailing ctrl-Z's (at end of file) and spaces
        if (fileLen <= 0) {
            while (buffUsed  &&  buff[buffUsed - 1] == CTRLZ) {
                buffUsed--;
            }
        }
        while (buffUsed  &&  buff[buffUsed - 1] == ' ') {
            buffUsed--;
        }
        buff[buffUsed] = '\0';

        // Skip blank lines
        if (!buffUsed) {
            continue;
        }

        // Handle the input line, depending on our state
        switch (state) {

            // Initial state: Look for "Message:"
            case 0:
                if (matchLine(buff, "message:", NULL,
                        0) != STAT_FOUND) {
                    fprintf(stderr,
                        "ie_ex4: Missing \"Message:\" line\n");
                    fclose(fpIn);
                    return  1;
                }
                state++;
                break;

            // Look for size after "Message:"
            case 1:
                if (!parseSize(buff, &ignrLen)) {
                    fprintf(stderr,
                        "ie_ex4: Bad size after \"Message:\"\n");
                    fclose(fpIn);
                    return  1;
                }
                state++;
                break;

            // Parse header lines (go to next state if "Contents:")
            case 2:
                switch (parseHdrLine(buff)) {
                                // Keep going after unknown lines
                    case STAT_UNKNWN:
                    case STAT_FOUND:
                        break;
                    case STAT_FNDSP1:
                        state++;
                        break;
                    default:
                        fclose(fpIn);
                        return  1;
                }
                break;

            // Parse size after "Contents:"
            case 3:
                if (!parseSize(buff, &ignrLen)) {
                    fprintf(stderr,
                        "ie_ex4: Bad size after \"Contents:\"\n");
                    fclose(fpIn);
                    return  1;
                }
                state++;
                break;

            // Parse item type and title (or "Message:") after first
            //    making sure there is no pending forward item -- next
            //    state is 4 (if extra text after forward item end
            //    separator), 1 (if "Message:"), 5 (if text or file
            //    item), or 6 (if forward item)
            case 4:
                if (frwdLen) {
                    fprintf(stderr, "ie_ex4: Extra after forward item "
                        "end separator\n");
                    break;
                }
                if (matchLine(buff, "message:", NULL,
                        0) == STAT_FOUND) {
                    state = 1;
                }
                else if (matchLine(buff, "text item:", title,
                        sizeof(title)) == STAT_FOUND) {
                    printf("Text item with title \"%s\" will be "
                        "skipped\n", title);
                    state++;
                }
                else if (matchLine(buff, "file item:", title,
                        sizeof(title)) == STAT_FOUND) {
                    printf("File item with title \"%s\" will be "
                        "skipped\n", title);
                    state++;
                    // Note that the date and time in the item title are
                    //    formatted based on the country information of
                    //    the author (if FILES/DETACH is used on Export,
                    //    this will already have been handled correctly)
                }
                else if (matchLine(buff, "forward item:", NULL,
                        0) == STAT_FOUND) {
                    printf("Forward item -----------------\n");
                    state = 6;
                    break;
                }
                else {
                    fprintf(stderr,
                        "ie_ex4: Missing item keyword line\n");
                    fclose(fpIn);
                    return  1;
                }
                break;

            // Parse size after "Text item:" or "File item:" and skip
            //    it -- next state is 4
            case 5:
                if (!parseSize(buff, &itemLen)) {
                    fprintf(stderr, "ie_ex4: Bad item size\n");
                    fclose(fpIn);
                    return  1;
                }
                if (itemLen > fileLen) {
                    fprintf(stderr, "ie_ex4: Item larger than rest of "
                        "input file\n");
                    fclose(fpIn);
                    return  1;
                }
                if (fseek(fpIn, itemLen, SEEK_CUR)) {
                    fprintf(stderr, "ie_ex4: Error skipping item\n");
                    fclose(fpIn);
                    return  1;
                }
                fileLen -= itemLen;
                state--;
                break;

            // Parse size after "Forward item:"
            case 6:
                if (!parseSize(buff, &frwdLen)) {
                    fprintf(stderr, "ie_ex4: Bad forward item size\n");
                    fclose(fpIn);
                    return  1;
                }
                if (frwdLen > fileLen) {
                    fprintf(stderr, "ie_ex4: Item larger than rest of "
                        "input file\n");
                    fclose(fpIn);
                    return  1;
                }
                state++;
                break;

            // Parse initial forward item separator line
            case 7:
                if (parseFwdLine(buff) != STAT_FNDSP1) {
                    fprintf(stderr, "ie_ex4: Missing forward item "
                        "separator line\n");
                    fclose(fpIn);
                    return  1;
                }
                state++;
                break;

            // Parse forward item lines (goto state 4 if end separator
            //    line)
            case 8:
                switch (parseFwdLine(buff)) {
                                // Keep going after unknown lines
                    case STAT_UNKNWN:
                    case STAT_FOUND:
                    case STAT_FNDSP1:
                        break;
                    case STAT_FNDSP2:
                        printf("End of forward item ----------\n");
                        state = 4;
                        break;
                    default:
                        fclose(fpIn);
                        return  1;
                }
                break;

            // Should never get here
            default:
                break;
        }
    }

    // Make sure everything parsed
    if (state != 4) {
        fprintf(stderr, "ie_ex4: File ended in invalid state\n");
        fclose(fpIn);
        return  1;
    }

    // Close input file (ignore errors) and exit with no error
    fclose(fpIn);
    return  0;
} // main


/***********************************************************************
Routine    : matchLine
Description: Determine if the specified buffer begins with the specified
           : pattern.  There is a match if the pattern is at the start
           : of the buffer (after skipping leading spaces).  The
           : comparison is case-insensitive.  The characters in the
           : buffer after the pattern are returned in the specified
           : output buffer (if any).
Inputs     : inBuff     buffer to parse (null-terminated, leading and
           :            trailing spaces are ignored)
           : patt       pattern to check for (null-terminated)
           : outBuff    buffer to place text that follows the pattern in
           :            (spaces between pattern and text are skipped,
           :            this argument ignored if outBuffSiz is zero)
           : outBuffSiz size of outBuff (zero means no text is allowed
           :            to follow the pattern, at most outBuffSiz-1
           :            characters can be returned because of the
           :            null-terminating byte)
Returns    : STAT_xxxxxx status (errors include unexpected characters or
           : too many characters following pattern)
***********************************************************************/
int matchLine(char *inBuff, char *patt, char *outBuff, unsigned
              outBuffSiz)
{
    unsigned len;

    // Handle possible invalid call
    if (outBuff == NULL) {
        outBuffSiz = 0;
    }

    // See if the pattern matches (return if not)
    while (*inBuff == ' ') {
        inBuff++;
    }
    len = strlen(patt);
    if (strnicmp(inBuff, patt, len)) {
        return  STAT_UNKNWN;
    }

    // Determine the length of the following characters and return if
    //    there's a problem
    inBuff += len;
    while (*inBuff == ' ') {
        inBuff++;
    }
    len = strlen(inBuff);
    while (len  &&  inBuff[len - 1] == ' ') {
        len--;
    }
    if (!outBuffSiz && len) {
        return  STAT_FNDERR;
    }
    if (outBuffSiz  &&  (len + 1) > outBuffSiz) {
        return  STAT_FNDERR;
    }

    // Return the following characters
    memmove(outBuff, inBuff, len);
    outBuff[len] = '\0';
    return  STAT_FOUND;
} // matchLine


/***********************************************************************
Routine    : parseFwdLine
Description: Parse the specified forward item line and return it's
           : legality.  In this example program, legal forward item
           : lines (except for the end separator) are simply written to
           : the standard output, while unknown and illegal lines are
           : written to the standard error.
Inputs     : line       forward item line to parse (null-terminated)
Returns    : STAT_xxxxxx status (special pattern #1 is section start
           : separator line and special pattern #2 is section end
           : separator line)
***********************************************************************/
int parseFwdLine(char *line)
{
    unsigned len;

    // Handle non-separator lines
    if (*line != '-') {
        switch (parseHdrLine(line)) {
            case STAT_UNKNWN:
            case STAT_FNDSP1:   // Contents: is unknown for forwarding
                return  STAT_UNKNWN;
            case STAT_FOUND:
                return  STAT_FOUND;
            default:
                return  STAT_FNDERR;
        }
    }

    // Skip leading and trailing dashes and spaces on separator lines
    while (*line == '-'  ||  *line == ' ') {
        line++;
    }
    len = strlen(line);
    while (len  &&  (line[len - 1] == '-'  ||  line[len - 1] == ' ')) {
        len--;
    }

    // Output and return separator lines
    if (len) {
        printf("Forward item section: \"%.*s\"\n", len, line);
        return  STAT_FNDSP1;
    }
    return  STAT_FNDSP2;
} // parseFwdLine


/***********************************************************************
Routine    : parseHdrLine
Description: Parse the specified header line and return it's legality.
           : In this example program, legal header lines (except
           : "Contents:") are simply written to the standard output,
           : while unknown and illegal lines are written to the standard
           : error.  Note that the date and time in the "Date:" header
           : line are always formatted as "mm/dd/yy hh:mm?m" by Export,
           : regardless of country.
Inputs     : line       header line to parse (null-terminated)
Returns    : STAT_xxxxxx status (special pattern #1 is "Contents:")
***********************************************************************/
int parseHdrLine(char *line)
{
    static struct {
        char *patt;             // Pattern to match
        unsigned len;           // Allowed text length
    } patts[] = {
        { "contents:", 0 },     // Must be 1st (index 0)
        { "priority:", 7 },     // Must be 2nd (index 1)
        { "from:", 257 },
        { "forwarded by:", 257 },
        { "to:", 257 },
        { "cc:", 257 },
        { "bcc:", 257 },
        { "*to:", 257 },
        { "*cc:", 257 },
        { "not deliverable to:", 257 },
        { "subject:", 61 },
        { "date:", 30 },
        { "rrq:", 73 },
        { "rrt:", 73 },
        { "receipt requested", 0 },
        { "", 0 }               // Must be last
    };
    char buff[BUFFSIZE];
    unsigned i;

    // Loop for each possible pattern...
    for (i=0; patts[i].patt[0]; i++) {
        switch (matchLine(line, patts[i].patt, buff, patts[i].len)) {

            // This wasn't the pattern, so try the next one
            case STAT_UNKNWN:
                break;

            // This was the pattern (handle "Contents:" and "Priority:"
            //    specially)
            case STAT_FOUND:
                if (i == 0) {
                    return  STAT_FNDSP1;
                }
                else if (i != 1                          ||
                        matchLine(buff, "urgent", NULL,
                            0) == STAT_FOUND             ||
                        matchLine(buff, "low"   , NULL,
                            0) == STAT_FOUND             ||
                        matchLine(buff, "normal", NULL,
                            0) == STAT_FOUND             ) {
                    printf("Header field \"%s\" is \"%s\"\n",
                        patts[i].patt, buff);
                    return  STAT_FOUND;
                }
                // Priority was invalid -- fall through to error case

            // This was the pattern, but there was an error
            default:
                fprintf(stderr,
                    "ie_ex4: Error parsing header line \"%s\"\n", line);
                return  STAT_FNDERR;
        }
    }

    // Unknown header line
    fprintf(stderr, "ie_ex4: Unknown header line \"%s\"\n", line);
    return  STAT_UNKNWN;
} // parseHdrLine


/***********************************************************************
Routine    : parseSize
Description: Parse the specified buffer as a message section size, and
           : return the value (indirectly) and the success status
           : (directly).
Inputs     : buff       buffer containing the size to parse
           :            (null-terminated)
           : sizeP      pointer to a variable to receive the size
Returns    : Zero (false) if illegal size
***********************************************************************/
int parseSize(char *buff, long *sizeP)
{
    char *end;
    unsigned long result;

    // Parse the number
    result = strtoul(buff, &end, 10);

    // If nothing parsed or it overflowed a signed long, it's a error
    if (end == buff  ||  (long)result < 0) {
        return  0;
    }

    // Make sure there is nothing after it except spaces
    while (*end == ' ') {
        end++;
    }
    if (*end) {
        return  0;
    }

    // It must be okay
    *sizeP = result;
    return  1;
} // parseSize

// End of IE_EX4.C
