/**
 * @fileoverview Advanced Expand/Collapse Control for hierarchical data visualization
 * This module provides functionality to create expandable/collapsible sections in lists and crosstabs
 *
 * @requires jquery
 */
define(["jquery"], function () {
    "use strict";

    /**
     * @constructor
     * @description Constructor for the expandCollapse control
     */
    function expandCollapse(){};

    /**
     * @function initialize
     * @description Initializes the expand/collapse functionality on the specified list control
     * @param {Object} oControlHost - The host control object containing configuration
     * @param {Function} fnDoneInitializing - Callback function to execute when initialization is complete
     */
    expandCollapse.prototype.initialize = function( oControlHost, fnDoneInitializing ){
      
    var o = oControlHost.configuration
      , list = oControlHost.page.getControlByName(o["listName"])
      , listElm = list.element
      , startState = o['startState']
      ;

    // Type 1: List with group headers implementation
    if(o["type"]==1){
    /**
     * Type 1: List with group headers implementation
     *
     * This method works with lists that use group headers for hierarchical data.
     * Requirements:
     * - List must have grouping applied to create hierarchy
     * - Group header cells must contain <span class="expand"> elements
     *
     * Implementation details:
     * 1. Finds all rows with expand spans
     * 2. Uses rowSpan to determine child rows
     * 3. Sets "parent" attribute on cells to track hierarchy
     * 4. Initially hides all child rows based on startState
     */
    $(listElm).find('tr:has(span[class="expand"])').each(
      function() {
        var r = this.rowIndex
          , a = this.nextSibling.firstChild.rowSpan;
        
        // Process each row within the rowSpan range
        for (var i=r+1; i<r+a+1; ++i){
          var id = $(this).find('td:has(span[class="expand"])')[0].id
          // Set parent attribute to track hierarchy
          $(listElm.rows[i]).find('TD').attr(
              'parent'
            , function(){
              return $(this).attr('parent')? $(this).attr('parent')+id +'|': id+'|'
            }
          );
          // Initially hide all child cells
          if(startState !== 'expanded') {
            $(listElm.rows[i]).find('TD').hide();
          }
        }
      }
    );
    }

    // Type 2: Crosstab with left padding implementation
    if(o["type"]==2){
    /**
     * Type 2: Crosstab with left padding implementation
     *
     * This method works with crosstabs that use left padding to indicate hierarchy levels.
     * Requirements:
     * - Each row represents an item in the hierarchy
     * - Left padding is used to indicate hierarchy level
     * - Cells must contain <span class="expand"> elements
     *
     * Implementation details:
     * 1. Finds all rows with expand spans
     * 2. Compares left padding between rows to determine parent-child relationships
     * 3. Sets "parent" attribute on cells to track hierarchy
     * 4. Removes expand spans from rows with no children
     * 5. Initially hides/shows child rows based on startState
     */
 
 
 
    // Process each row with an expand span
    $(listElm).find('tr:has(span[class="expand"])').each(
      function() {
        var r = this.rowIndex
          , a = listElm.rows.length
          , lpad = parseInt($($(this).find('td:has(span[class="expand"])')[0]).css('padding-left'))
          , itr=0
          ;
        
        // Remove expand span if this is the last row (no children possible)
        if(r+1==a) $(this).find('span[class="expand"]').remove();
        
        // Process all potential child rows
        for (var i=r+1; i<a; ++i){
          var id = $(this).find('td:has(span[class="expand"])')[0].id
            , td = $(listElm.rows[i]).find('TD')
          
          // Check if this row is at same or higher level (not a child)
          if(parseInt($(td[0]).css('padding-left'))<=lpad) {
            // Remove expand span if no children were found
            if(itr==0) $(this).find('span[class="expand"]').remove();
            i=a; // Exit the loop
          }
          else {
            // This is a child row - increment counter and set parent attribute
            ++itr;
            $(td).attr(
                'parent'
              , function(){
              return $(this).attr('parent')? $(this).attr('parent')+id+'|': id+'|'
            }
          );
          }
          
          // Hide child rows if startState is not 'expanded'
          if(startState !== 'expanded') {
            $(listElm.rows[i]).find('TD').hide();
          }
        }
      }
    );
    }

    /**
     * Common initialization for both types
     * Sets up the expand/collapse indicators and click handlers
     */
    
    // Set initial expand/collapse indicators based on startState
    if(startState == 'expanded') {
      $(listElm).find('span[class="expand"]').text('- ');
    }
    else {
      $(listElm).find('span[class="expand"]').text('+ ');
    }

    // Style and set initial state for expandable cells
    var tds = $(listElm).find('td:has(span[class="expand"])');
    tds.css( 'cursor', 'pointer' );
    if(startState == 'expanded') {
      tds.attr( 'state', 'collapse' );
    }
    else {
      tds.attr( 'state', 'expand' );
    }

    /**
     * Click handler for expandable cells
     * Toggles visibility of child elements and updates indicators
     */
    $(listElm).find('td:has(span[class="expand"])').click(function(){
      var p = $(this).attr('parent')?$(this).attr('parent'):'';
     
      if($(this).attr('state')=='expand'){
        // Expand action: Show direct children with animation
        $(listElm).find('TD[parent="'+p+this.id+'|"]').show(200);
        $(this).attr('state','collapse');
        $(this).find('span[class="expand"]').text('- ');
      }
      else {
        // Collapse action: Hide all descendants with animation
        $(listElm).find('TD[parent*="'+this.id+'|"]').hide(200);
        // Reset state of all descendant expandable cells
        $(listElm).find('TD[parent*="'+this.id+'|"]:has(span[class="expand"])').attr('state','expand');
        // Update expand/collapse indicators
        $(this).find('span[class="expand"]').text('+ ');
        $($(listElm).find('TD[parent*="'+this.id+'|"]:has(span[class="expand"])')).find('span[class="expand"]').text('+ ');
        
        $(this).attr('state','expand');
      }
    });
    
    // Store initialization state in session storage
    sessionStorage.setItem(this.ctrlFrom+'secondaryRun',1);
    
    // Signal that initialization is complete
    fnDoneInitializing();
    };

    return expandCollapse;
  }
);
