/*
 * %Z%%W% %I%
 *
 * =========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 * (C) Copyright IBM Corp. 2007,2024. All Rights Reserved
 * 
 * DISCLAIMER: 
 * The following [enclosed] code is sample code created by IBM 
 * Corporation.  This sample code is not part of any standard IBM product 
 * and is provided to you solely for the purpose of assisting you in the 
 * development of your applications.  The code is provided 'AS IS', 
 * without warranty of any kind.  IBM shall not be liable for any damages 
 * arising out of your use of the sample code, even if they have been 
 * advised of the possibility of such damages.
 * =========================================================================
 */
package com.ibm.jzos.sample.dfsort;

import java.io.BufferedOutputStream;
import java.util.Iterator;
import java.util.List;

import com.ibm.jzos.DfSort;
import com.ibm.jzos.RDWOutputRecordStream;
import com.ibm.jzos.RcException;

/**
 * This example creates an instance of {@link DfSort} to sort variable record data
 * generated by java and store the output in a Dataset.
 * <p>
 * Arguments (supplied as key=value pairs):
 * <dl>
 * <dt>sortout=&lt;dsn&gt;</dt>
 * <dd>The name of a RECFM=V dataset to receive the sorted data.  Supplied to DFSORT as SORTOUT.</dd>
 * <dt>numrecs=&lt;nn&gt;</dt>
 * <dd>The number of records to generate.</dd>
 * <dt>[shareas=&lt;yes | no&gt;]</dt>
 * <dd>Determines that address space for DFSORT.  If yes, DFSORT will run in the same address
 * space as the JVM.  If no, it will run in a separate address space.</dd>
 * <dt>[loglevel=&lt;n&gt;]</dt>
 * <dd>Sets the logging level for the child process and prints the resulting
 * child process output is written to System.err.  The valid levels are those defined
 * in the ZUtil.LOG_* constants.</dd>
 * </dl>
 * <p>
 * Concepts illustrated:
 * <ul>
 * <li>Generate unsorted variable length record data in Java
 * <li>Write these records on-the-fly to DFSORT.
 * </ul>
 */
public class DfSortVariableJavaToDataset {

	private static final String RECORD_DATA = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

	public static void main(String[] args) throws Exception {
		DfSortArguments dfsortArgs = new DfSortArguments(args);
    	if (dfsortArgs.getSortoutDataset() == null || dfsortArgs.getNumrecs() < 0) {
    		System.err.println("Usage: " + DfSortVariableJavaToDataset.class.getName() + " sortout=<dsn> numrecs=<nn> [shareas=<yes|NO>]");
       		System.err.println("\tsortout is an existing RECFM=V DSN to hold DFSORT output");
    		System.err.println("\tnumrecs is the number of records to generate");
    		System.err.println("\tshareas = yes: DFSORT executes in the same address space as the JVM");
			System.exit(4);
    	}
    	
		DsInfo outDs = dfsortArgs.getSortoutDataset();
		if (!outDs.isVariableRecfm()) {
			System.err.println("Dataset " + outDs.getFullyQualifiedDsn() + " is not RECFM=V");
			System.exit(8);
		}
		
		doSort(dfsortArgs);

	}
	
	private static void doSort(DfSortArguments dfsortArgs) throws Exception {
		DsInfo outDs = dfsortArgs.getSortoutDataset();

		DfSort dfSort = new DfSort();
				
		if (dfsortArgs.getLogLevel() != -1) {
			dfSort.setLoggingLevel(dfsortArgs.getLogLevel());
		}

		//Direct DFSORT to read input from Java.
		dfSort.setInputStreamHasRdws();
		
		//Direct DFSORT to send its output (SORTOUT) to the supplied DSN.
		dfSort.addAllocation("alloc fi(sortout) da("+outDs.getFullyQualifiedDsn()+") reuse shr msg(2)");
		
		//For this example, we sort the first 25 bytes of the input records.  
		//Since this SORTIN is RECFM=V, the record data begins in column 5 
		//(the first 4 bytes make up the RDW).  The VLSHRT option is used to 
		//allow records smaller than the sort field length.  Results are in 
		//ascending order (A)
		dfSort.addControlStatement("OPTION VLSHRT");
		dfSort.addControlStatement("SORT FIELDS=(5,25,CH,A)");
		
		//We also need to tell DFSORT the RECFM and LRECL of the incoming records
		dfSort.addControlStatement("RECORD TYPE=V,LENGTH=("+outDs.getLrecl()+")");
		
		//Specify whether the DFSORT child process should be run in a separate address space.
		//This allows multiple DFSORT processes to run simultaneously as each instance of
		//the DFSORT DDs (SORTIN, SORTOUT, etc...) will be in their own address space.
		dfSort.setSameAddressSpace(dfsortArgs.isSameAddressSpace());
		
		long startTime = System.currentTimeMillis();

		//Kick off the sort.  
		dfSort.execute();

		//Once the child starts, open an RDWOutputRecordStream on dfSort's stdin
		//and write generated test records to it.
		RDWOutputRecordStream rdwos = new RDWOutputRecordStream(
										new BufferedOutputStream(dfSort.getChildStdinStream()));
		byte[] byte_data = RECORD_DATA.getBytes();
		byte[] record = new byte[outDs.getLrecl()];
		try {
			for (int i=0; i<dfsortArgs.getNumrecs(); i++) {
				int s = i % (byte_data.length - 1);
				for (int j=0; j<record.length; j++,s++) {
					record[j] = byte_data[s % (byte_data.length - 1)];
				}
				// Make record lengths vary up to lrecl (minus RDW)
				int dataLen = Math.max(1, i % outDs.getLrecl() - 4);
				// Write the generated variable length record.
				// The RDWOutputStream will prefix the data by an RDW expected by DFSORT
				rdwos.write(record, 0, dataLen); 
			}
		} catch (Exception ioe) {
			ioe.printStackTrace();
		} finally {
			rdwos.close();
		}
				
		//Wait for dfSort to finish and check the result
		int rc =0;
		try {
			rc = dfSort.getReturnCode();
		} catch (RcException rce) {
			System.out.println("Caught RcException: " + rce.getMessage());
			rc = -1;
		}
		long runtime = System.currentTimeMillis() - startTime;

		if (rc != 0 || dfsortArgs.getLogLevel() >= 0) {
			List<?> stderrLines = dfSort.getStderrLines();
			for (Iterator<?> i=stderrLines.iterator(); i.hasNext(); ) {
				System.err.println(i.next());
			}
		}

		System.out.println("RC=" + rc + " TIME=" + runtime + " RECORD COUNT=" + dfsortArgs.getNumrecs() + " "
				+ DfSortVariableJavaToDataset.class.getName());		
	}

}
