/*
 * %Z%%W% %I%
 *
 * =========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 * (C) Copyright IBM Corp. 2007. All Rights Reserved
 *
 * DISCLAIMER:
 * The following [enclosed] code is sample code created by IBM
 * Corporation.  This sample code is not part of any standard IBM product
 * and is provided to you solely for the purpose of assisting you in the
 * development of your applications.  The code is provided 'AS IS',
 * without warranty of any kind.  IBM shall not be liable for any damages
 * arising out of your use of the sample code, even if they have been
 * advised of the possibility of such damages.
 * =========================================================================
 */
package com.ibm.jzos.sample.dfsort;

import java.util.StringTokenizer;

import com.ibm.jzos.ZFileException;

/**
 * Helper class for parsing key=value pair arguments into a usable
 * form for the DFSORT sample programs.
 * <p>
 * An instance of DfSortArguments is constructed with the String[] arguments
 * in the form <parm=value>, as provided to one of the DFSORT sample
 * classes in this package.  It will parse and provide
 * accessors the following information:
 * <ul>
 * <li><code>DsInfo getSortinDataset()</code> from: "sortin=dsn"</li>
 * <li><code>DsInfo getSortoutDataset()</code> from: "sortout=dsn"</li>
 * <li><code>boolean isSameAddressSpace()</code> from: "shareas=yes/no"</li>
 * <li><code>String getEncoding()</code> from: encoding=value</li>
 * <li><code>int getLrecl()</code> from: lrecl=nnn</li>
 * <li><code>int getNumrecs()</code> from: numrecs=mmm</li>
 * <li><code>int getLogLevel()</code> from: loglevel=n</li>
 * </ul>
 */
public class DfSortArguments {

    private DsInfo sortinDataset;
    private DsInfo sortoutDataset;
    private boolean sameAddressSpace = false;
    private int lrecl = -1;
    private int numrecs = -1;
    private String encoding;
    private int logLevel = -1;

    /**
     * Construct and parse a list of arguments as described by the class description.
     * @param args an array of key=value arguments
     * @throws ZFileException if a dataset cannot be opened
     */
    public DfSortArguments(String args[]) throws ZFileException {
        for (int i = 0; i<args.length; i++) {
            StringTokenizer strtok = new StringTokenizer(args[i],"=");
            if (strtok.countTokens() != 2) {
                throw new IllegalArgumentException("Argument '" + args[i] + "' not in the form: name=value");
            }
            String name = strtok.nextToken();
            String value = strtok.nextToken();
            if ("sortin".equalsIgnoreCase(name)) {
                sortinDataset = new DsInfo(value);
            } else if ("sortout".equalsIgnoreCase(name)) {
                sortoutDataset = new DsInfo(value);
            } else if ("shareas".equalsIgnoreCase(name)) {
                sameAddressSpace = ("yes".equalsIgnoreCase(value));
            } else if ("lrecl".equalsIgnoreCase(name)) {
                lrecl = Integer.parseInt(value);
            } else if ("numrecs".equalsIgnoreCase(name)) {
                numrecs = Integer.parseInt(value);
            } else if ("encoding".equalsIgnoreCase(name)) {
                encoding = value;
            } else if ("loglevel".equalsIgnoreCase(name)) {
                logLevel = Integer.parseInt(value);
            } else {
                throw new IllegalArgumentException("Argument '" + name + "' not recognized.");
            }
        }
    }

    /**
     * Answer a description of the sortin dataset.  Answers null if not supplied.
     * @return DsInfo
     */
    public DsInfo getSortinDataset() {
        return sortinDataset;
    }

    /**
     * Answer a description of the sortout dataset.  Answers null if not
     * supplied.
     * @return DsInfo
     */
    public DsInfo getSortoutDataset() {
        return sortoutDataset;
    }

    /**
     * Answer true if DFSORT should be spawned in the same address space.  Answers
     * false by default.
     * @return boolean
     */
    public boolean isSameAddressSpace() {
        return sameAddressSpace;
    }

    /**
     * Answer an encoding name for encoding/decoding logical records.  Answers
     * null if not supplied.
     * @return String
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * Answer the LRECL used for the sample.   Answers -1 if
     * not supplied.
     * @return int
     */
    public int getLrecl() {
        return lrecl;
    }

    /**
     * Answer the number of records to be used/generated by the sample.
     * Answers -1 if not supplied.
     * @return int
     */
    public int getNumrecs() {
        return numrecs;
    }

    /**
     * Answer the logging level to use for the child process.  Answers -1 if
     * not supplied.
     * @return int
     */
    public int getLogLevel() {
        return logLevel;
    }

}
