#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# --------------------------------------------- 



disk_check(){
    local disk_space=$1
    local disk_limit=$2
    # lets check that df gives us a number with G
    if [[  $disk_space =~ [0-9]*G ]]; then
        # remove trailing G
        local disk_space_gb=${disk_space%%[G]*}
        if (( $disk_space_gb < $disk_limit )); then
            return 1
        fi
    else
        return 2
    fi
}

check_disk_space() {
    # check the general size of the filesystem and warn if it's under 8G
    local total_disk_space=$( df -h /  | awk 'NR==2 {print $4}')
    disk_check $total_disk_space 10
    local disk_check_result=$?
    # if disk_check 1 the space is under the limit
    if [[ $disk_check_result -eq 1 ]]; then
        terminal_warn "There might not be enough disk space on machine. This could affect running of ${PRODUCT_NAME} images." "warn-low-disk-space"
    fi

    # if disk_check 2 the space is not in G
    if [[ $disk_check_result -eq 2 ]]; then
        terminal_warn "Unable to determine disk space on machine. This could affect running of ${PRODUCT_NAME} images." "unable-to-determine-disk"
    fi

    # this is a check of where the images will be saved to make sure filesystem is large enough
    # check space in ${HOME}/.local/shared/containers/storage if being run as non-root
    PATH_TO_CHECK="/var"
    if [[ "$EUID" -ne 0 ]]; then
        PATH_TO_CHECK=$(podman info --format '{{.Store.GraphRoot}}')
        if [[ "${TA_LOCAL_TEMP_DOCKER_INSTALL}" == "true" ]] && ! is_mac_os; then
            PATH_TO_CHECK="/var/lib/docker"
        fi
    fi
    log "Checking $PATH_TO_CHECK has enough space"
    if is_mac_os; then
        local disk_space=$(podman machine ssh -- df -h "$PATH_TO_CHECK" | tail -1 | awk '{print $4}')
    else
        local disk_space=$( df -h "$PATH_TO_CHECK" | awk 'NR==2 {print $4}')
    fi
    log "Found disk space of ${disk_space} on ${PATH_TO_CHECK}"
    # need 5G for the images
    disk_check $disk_space 10
    local disk_check_result=$?
    # if disk_check 1 the space is under the limit
    if [[ $disk_check_result -eq 1 ]]; then
        terminal_error "Not enough disk space on ${PATH_TO_CHECK}. Please free up at least 10GB for ${PRODUCT_NAME} images and generated data." "not-enough-disk"
        exit 1
    fi

    # if disk_check 2 the space is not in G
    if [[ $disk_check_result -eq 2 ]]; then
        terminal_error "Unable to determine disk space on ${PATH_TO_CHECK}. Please free up at least 10GB ${PRODUCT_NAME} images and generate data." "unable-to-determine-disk"
        exit 1

    fi
}


check_rootless(){
	# Skip if podman isnt installed
	if [[ "${TA_LOCAL_TEMP_PODMAN_INSTALL}" != "true" ]]; then
		return
	fi	
	
	# Experienced some issues with running rootless machine on mac - force rootful for now 
	if is_mac_os; then
        log "OS is MAC - need to be rootful"
		if ! podman_machine_default_is_root; then
			header "Rootless podman machine detected" 
			to_terminal "Please update default machine to rootful with the following commands:"
			to_terminal "  podman machine stop"
			to_terminal "  podman machine set --rootful"
			to_terminal "  podman machine start"
			to_terminal "Then re-launch the script."
			exit 1
		fi
	else
		# Need to update image trust for 'icr.io' - `podman image trust set -t accept icr.io`
		# We need to update the ports from privilaged to something else e.g.3000
		# Linux - is script being run as root?
		# Need to check podman image trust for 'icr.io'
		# need to check port in .configuration file
		if [[ "$EUID" -ne 0 ]] && ( ! podman_image_trust_is_set ||  check_ui_port_privilege ) ; then
			header "${PRODUCT_NAME} is trying to be run with rootless user, configuration required for non-root user"
			to_terminal "The following configuration is needed:"
            
            if ! check_policy_json_present; then
                to_terminal "• Create containers config directory for user config: 'mkdir -p $HOME/.config/containers'"
                to_terminal "• Copy policy file if it exists: 'cp /etc/containers/policy.json $HOME/.config/containers/policy.json'"
                to_terminal "• Create policy file if it doesn't exist: 'touch  $HOME/.config/containers/policy.json'"
            fi

            if ! should_linger_be_enabled $UID; then
                to_terminal "• Enable lingering for user. Allows user to run long-running services independently of their login sessions: 'loginctl enable-linger "$UID"'"
            fi
			
            if ! podman_image_trust_is_set; then
				to_terminal "• Set image trust: 'podman image trust set -t accept icr.io'"
			fi

			if check_ui_port_privilege; then
				to_terminal "• Change UI port from privileged to non-privileged (e.g., 443→3000) in 'scripts/.configuration'"
			fi

            if check_resources_enabled; then
				to_terminal "• Disable resources for containers in 'scripts/.configuration'. If you want to use resources this may require manual updating the CPU cgroup controller in the user slice, which will require root permissions "
            fi
			

			header "Auto-Configuration Available"
			to_terminal "We can automatically apply these changes for you."
			
			prompt_and_read "\n\tWould you like to auto-configure ${PRODUCT_NAME} for use by rootless user? (y/n)" REPLY
			if [[ "$REPLY" =~ ^[Yy]$ ]]; then		
				run_with_spinner "enable_lingering; create_policy; podman_set_trust; update_ui_port; update_resources; sleep 2" \
                 "Applying changes and restarting" \
                 "Configuring"
				exit 255
			else
				to_terminal "Auto-configuration skipped. Please manually configure before restarting if you want to be able to run as a rootless user."
				to_terminal "Otherwise you need to run ${PRODUCT_NAME} as root."
				exit 1
			fi
		fi
	fi
}

is_mac_os() {
	if [[ "$OSTYPE" == "darwin"* ]]; then
		return 0
	else
		return 1
	fi
}