#!/usr/bin/env bash
# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------
PARENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LOG_DIR="${PARENT_DIR}/logs"
TA_LOCAL_CONFIG=scripts/.configuration
TA_LOCAL_CONFIG_DEV=scripts/.configuration.dev
source ./scripts/logging.sh
setup_background_logging
# If linux check linger and enable at start for podman and check podman 
source ./scripts/launch_utils.sh
source ./scripts/configuration_utils.sh
if ! should_linger_be_enabled; then
    enable_lingering
fi 
source ./scripts/helper.sh
sourceConfig
# Needs to be before createPodmanAlias as it sets the product name
source ./scripts/getProductName.sh 	
getProductName
getProductDoc
getProductShort
source ./scripts/createPodmanAlias.sh
createPodmanAlias
source scripts/acceptLic.sh
source ./scripts/findHost.sh
getHostIp
source ./scripts/spinner.sh
source ./scripts/podman_machine_utils.sh
source ./scripts/container_manager_utils.sh
 

trap 'to_terminal "Script interrupted by user (Ctrl+C or SIGTERM)."; exit 130' INT TERM

#### Usage ####
PASSED_PARAM="${1:-}"
case $PASSED_PARAM in
  -h|--h|--help|-help)
	to_terminal "Welcome to ${PRODUCT_NAME} Local."
	to_terminal "To install ${PRODUCT_NAME} Local you need docker or podman installed."
	exit 0
    ;;
esac

#run scripts at relative path
scriptdir="$(dirname "$0")"
cd "$scriptdir"


# main logic in function the will be run in subshell. Makes restarting cleaner
main(){
	## Check cores of physical machine min needed is 4
	## check minimum number of cores
	num_cores=$(nproc 2>/dev/null || sysctl -n hw.logicalcpu)
	if [[ "$num_cores" -lt "4" ]]; then
		to_terminal "Not enough CPU cores. You need at least 4 CPU cores to run."
		exit
	fi

	## if we find both docker and podman then we exit 
	if ( is_docker_installed && is_podman_installed ); then
		status_header
		to_terminal "${PRODUCT_NAME} does not run when both Podman and Docker are on the same machine."
		to_terminal "Please remove one, for example, 'mv /usr/bin/docker /usr/bin/docker_backup'" 
		to_terminal "Once removed re-run the launch.sh script"
		exit
	fi

	## check docker running
	is_cm_running

	## check docker running - docker can be docker or podman - depends on alias
	docker_interpretation=$(type -a docker 2>&1)
	if [ -z "$docker_interpretation" ] || [[ $docker_interpretation =~ "no docker in" ]]; then
		status_header
		to_terminal "Docker not installed. Please install docker and re-run the script."
		to_terminal "https://docs.docker.com/install/"
		exit
	fi

	# Disk space requirement
	check_disk_space

	# Root/less checks
	check_rootless


	if [[ -z ${TA_LOCAL_RUNTIME_HOST_IP}  ]]; then
		case "$image_status" in
			"some-present")
				status_header
				to_terminal "Unable to detect a Public IP Address and not all ${PRODUCT_NAME} docker images are pulled."
				prompt_and_read "
					\t1) Working in an Air Gapped Environment\n
					t2) Quit." CHOICE
				case $CHOICE in
					1) ./scripts/fileImages.sh ;;
					2) exit ;;
				esac
				exit
				;;
		esac
	fi

	############################################# MAC ################################################################################

	## check if system is a mac

	if uname -s | grep 'Darwin' >/dev/null; then
		if [[ ${TA_LOCAL_TEMP_PODMAN_INSTALL} = true ]]; then
			check_podman_machine_resources
		fi

		# Is this used??
		host_to_find=$(ifconfig | grep -Fv -- "-->" | grep "inet " | grep -E -o "([0-9]{1,3}[\.]){3}[0-9]{1,3}" | grep -Fv 127 | grep -Fv 255 | grep -Fv 0.0.0.0)

		## get ID of container
		server_ta=$(docker ps -a | grep transformation-advisor-server | awk '{print $1}')

		ta_ui_host=""
		new_ta_ui_host=""
		## check if TA is running
		server_ta_running=$(docker ps | grep transformation-advisor-server | awk '{print $1}')

		if [[ -n $server_ta_running ]]; then # means TA is running
			ta_ui_host=$(docker exec -it $server_ta sh -c "env | grep TA_PUBLIC_ACCESSIBLE_API" | awk -F // '{print $2}' | awk -F : '{print $1}')
			new_ta_ui_host="${ta_ui_host//[[:space:]]/}"
		fi

		if [ -e ./.license_accepted ]; then
			# if the length of $server_ta_running is non-zero
			if [[ (-n $server_ta_running && (${TA_LOCAL_RUNTIME_HOST_IP} != ${new_ta_ui_host}))]]; then
				to_terminal "Host mismatch...correcting......."
				./scripts/stopBeforeInstall.sh
				./scripts/installLocalMismatch.sh
			fi
		fi


	################################################## END OF MAC ########################################################################################################
	fi


	## If a license has been accepted check images
	if [[ -e ./.license_accepted ]]; then

		# todo finish up the cases for when a license is present i.e some/none etc
		case "$image_status" in
			"some-present")
				status_header
				to_terminal "Not all ${PRODUCT_NAME} docker images are pulled."
				to_terminal "Select option 1 to install ${PRODUCT_NAME}."
				prompt_and_read "
					\n\t1) Install ${PRODUCT_NAME}\n
					\t2) Quit." CHOICE
				case $CHOICE in
					1)
						run_with_spinner "./scripts/cleanUpBeforeInstall.sh" "Cleaning up before installing ${PRODUCT_NAME}" "Cleaning"
						./scripts/installLocal.sh
						sleep 3
						./scripts/renameDockerName.sh&>/dev/null
						;;
					2) exit ;;

					esac
				exit
				;;
			"none-present")
				header "Status"
				to_terminal "${PRODUCT_NAME} images are not pulled even though there is a license file."
				run_with_spinner 'rm -rf $PARENT_DIR/.license_accepted; sleep 2' "Removing file and RESTARTING Installation" "Restarting"
				# rather than spawn new process, exit with restart code
				# means we don't have to deal with env persitance etc
				exit 255
				;;
		esac
		header "License Exists"

	else
			licenseType
			acceptLicense "$PASSED_PARAM"
	fi


	## Launch the script for user options etc.
	if [ -e ./.license_accepted ]; then
		./scripts/launch.sh "$PASSED_PARAM"
	fi


}


# main loop, allows easy restart
while true; do
    (main "$@")
    res=$?
    if (( res != 255 )); then
        break
    fi
done
