#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# --------------------------------------------- 


# Function to check if available disk space is greater than a specified limit
# Parameters:
#   $1 - Available disk space in MB
#   $2 - Minimum required disk space in MB
# Returns:
#   0 - If available space is greater than or equal to the limit
#   1 - If available space is less than the limit
disk_check_mb(){
    local disk_space_mb=$1
    local disk_limit_mb=$2
    log "Checking disk space $disk_space_mb MB is greater than limit $disk_limit_mb MB"
    if (( disk_space_mb < disk_limit_mb )); then
        return 1
    fi
    return 0
}

# Function to verify sufficient disk space for running container images
# This function performs two disk space checks:
# 1. Checks the root filesystem for general system requirements
# 2. Checks the container storage location for image storage requirements
check_disk_space() {
    # First check: Verify root filesystem has enough space (minimum 10GB)
    # Get available disk space in MB on the root filesystem
    local total_disk_space_mb=$(df -m / | awk 'NR==2 {print $4}')
    # Check if available space meets the minimum requirement of 10GB (10240MB)
    disk_check_mb "$total_disk_space_mb" 10240  # 10GB in MB
    local disk_check_result=$?
    # If available space is below the minimum requirement, display a warning
    if [[ $disk_check_result -eq 1 ]]; then
        terminal_warn "There might not be enough disk space on machine. This could affect running of ${PRODUCT_NAME} images. Please free up space." "warn-low-disk-space"
    fi

    # Second check: Verify container storage location has enough space
    # Determine the appropriate path to check based on user permissions and OS
    PATH_TO_CHECK="/var"  # Default path for root user
    if [[ "$EUID" -ne 0 ]]; then
        # For non-root users, get the podman storage location
        PATH_TO_CHECK=$(podman info --format '{{.Store.GraphRoot}}')
        # If using temporary Docker installation on non-Mac systems, use Docker's storage path
        if [[ "${TA_LOCAL_TEMP_DOCKER_INSTALL}" == "true" ]] && ! is_mac_os; then
            PATH_TO_CHECK="/var/lib/docker"
        fi
    fi
    log "Checking $PATH_TO_CHECK has enough space"
    
    # Get available disk space in MB for the container storage location
    local disk_space_mb
    if is_mac_os; then
        # On macOS, need to SSH into the podman machine to check disk space
        disk_space_mb=$(podman machine ssh -- df -m "$PATH_TO_CHECK" | awk 'NR==2 {print $4}')
    else
        # On Linux, directly check the filesystem
        disk_space_mb=$(df -m "$PATH_TO_CHECK" | awk 'NR==2 {print $4}')
    fi
    log "Found disk space of ${disk_space_mb} MB on ${PATH_TO_CHECK}"
    
    # Check if available space meets the minimum requirement of 5GB (5120MB) for container images
    disk_check_mb "$disk_space_mb" 5120  # 5GB in MB
    local disk_check_result=$?
    # If available space is below the minimum requirement, display an error and exit
    if [[ $disk_check_result -eq 1 ]]; then
        terminal_error "Not enough disk space on ${PATH_TO_CHECK}. Please free up at least 10GB for ${PRODUCT_NAME} images and generated data." "not-enough-disk"
        exit 1
    fi
}

check_rootless(){
	# Skip if podman isnt installed
	if [[ "${TA_LOCAL_TEMP_PODMAN_INSTALL}" != "true" ]]; then
		return
	fi	
	
	# Experienced some issues with running rootless machine on mac - force rootful for now 
	if is_mac_os; then
        log "OS is MAC - need to be rootful"
		if ! podman_machine_default_is_root; then
			header "Rootless podman machine detected" 
			to_terminal "Please update default machine to rootful with the following commands:"
			to_terminal "  podman machine stop"
			to_terminal "  podman machine set --rootful"
			to_terminal "  podman machine start"
			to_terminal "Then re-launch the script."
			exit 1
		fi
	else
		# Need to update image trust for 'icr.io' - `podman image trust set -t accept icr.io`
		# We need to update the ports from privilaged to something else e.g.3000
		# Linux - is script being run as root?
		# Need to check podman image trust for 'icr.io'
		# need to check port in .configuration file
		if [[ "$EUID" -ne 0 ]] && ( ! podman_image_trust_is_set ||  check_ui_port_privilege ) ; then
			header "${PRODUCT_NAME} is trying to be run with rootless user, configuration required for non-root user"
			to_terminal "The following configuration is needed:"
            
            if ! check_policy_json_present; then
                to_terminal "• Create containers config directory for user config: 'mkdir -p $HOME/.config/containers'"
                to_terminal "• Copy policy file if it exists: 'cp /etc/containers/policy.json $HOME/.config/containers/policy.json'"
                to_terminal "• Create policy file if it doesn't exist: 'touch  $HOME/.config/containers/policy.json'"
            fi

            if ! should_linger_be_enabled $UID; then
                to_terminal "• Enable lingering for user. Allows user to run long-running services independently of their login sessions: 'loginctl enable-linger "$UID"'"
            fi
			
            if ! podman_image_trust_is_set; then
				to_terminal "• Set image trust: 'podman image trust set -t accept icr.io'"
			fi

			if check_ui_port_privilege; then
				to_terminal "• Change UI port from privileged to non-privileged (e.g., 443→3000) in 'scripts/.configuration'"
			fi

            if check_resources_enabled; then
				to_terminal "• Disable resources for containers in 'scripts/.configuration'. If you want to use resources this may require manual updating the CPU cgroup controller in the user slice, which will require root permissions "
            fi
			

			header "Auto-Configuration Available"
			to_terminal "We can automatically apply these changes for you."
			
			prompt_and_read "\n\tWould you like to auto-configure ${PRODUCT_NAME} for use by rootless user? (y/n)" REPLY
			if [[ "$REPLY" =~ ^[Yy]$ ]]; then		
				run_with_spinner "enable_lingering; create_policy; podman_set_trust; update_ui_port; update_resources; sleep 2" \
                 "Applying changes and restarting" \
                 "Configuring"
				exit 255
			else
				to_terminal "Auto-configuration skipped. Please manually configure before restarting if you want to be able to run as a rootless user."
				to_terminal "Otherwise you need to run ${PRODUCT_NAME} as root."
				exit 1
			fi
		fi
	fi
}

is_mac_os() {
	if [[ "$OSTYPE" == "darwin"* ]]; then
		return 0
	else
		return 1
	fi
}