#!/usr/bin/env bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------

source ./logging.sh
source ./spinner.sh

# source configure before installing
TA_LOCAL_CONFIG=./.configuration
TA_LOCAL_CONFIG_DEV=./.configuration.dev

source ./helper.sh
sourceConfig

source ./localTempPodman.sh
source ./.security_config
source ./container_manager_utils.sh

trap 'exit 130' INT TERM

get_image_container_id(){
        local include_all=0
        local image_name=""

        # Parse options: only -a is supported
        while getopts "a" opt; do
                case $opt in
                a) include_all=1 ;;
                *) return 1 ;;
                esac
        done
        # - OPTIND is the index of the next argument to be processed by getopts
        # - After the getopts loop, OPTIND-1 equals the number of options processed
        # - This effectively removes all processed options, making $1 the first non-option argument
        shift $((OPTIND-1))

        image_name="${1:-}"
        if [[ -z "$image_name" ]]; then
                echo "Error: No image name provided" >&2
                return 1
        fi

        if (( include_all )); then
                docker ps -aqf "name=$image_name"
        else
                docker ps -qf "name=$image_name"  # Running only (default)
        fi
}

get_server_id(){
        get_image_container_id "$@" "taserver"
}

get_ui_id(){
        get_image_container_id "$@" "taui"
}

get_graph_id(){
        get_image_container_id "$@" "tagraph"
}

get_db_id(){
        get_image_container_id "$@" "tadb"
}

pull_image(){
        local image_for_pull=$1
        local service_name=$2
        docker pull $image_for_pull
        if [[ $? -ne 0 ]]; then
                set_spinner_error_msg "Error pulling $service_name image. Look for any errors related to ${image_for_pull} in the log"
                return 1
        fi
}

function imagePull {
        if [[ "$image_status" == "none-present" ]] ; then
                header "Pulling Images"
                to_terminal "\t%s%s\n\n" "Pulling ${PRODUCT_NAME} images can take 5 minutes per image depending on network" 
                run_with_spinner 'pull_image "$TA_LOCAL_CONFIG_COUCH_IMAGE" "DB"' \
                        "Pulling DB Image" \
                        "Downloading"
                run_with_spinner 'pull_image "$TA_LOCAL_CONFIG_SERVER_IMAGE" "ADVISOR"' \
                        "Pulling ADVISOR Image" \
                        "Downloading"
                run_with_spinner 'pull_image "$TA_LOCAL_CONFIG_UI_IMAGE" "UI"' \
                        "Pulling UI Image" \
                        "Downloading"
                run_with_spinner 'pull_image "$TA_LOCAL_CONFIG_NEO4J_IMAGE" "NEO4J"' \
                        "Pulling NEO4J Image" \
                        "Downloading"
        fi
}

use_resources_if_enabled(){
        # If RESOURCES is true, set CPU and memory for the containers
        # We want to use the resources so the default in .configuration is true
        if [[ "${RESOURCES}" == true ]]; then
                source ./configHelper.sh
        fi
}

function imageRun {

        use_resources_if_enabled

        # we prioritise to use docker if available
        if [[ ${TA_LOCAL_TEMP_DOCKER_INSTALL} = true ]]; then

                if uname -s | grep 'Darwin' >/dev/null; then
                        # sed commands modified for mac
                        sed -i '' -e  's/<contServer>/taserver/g' .env
                        sed -i '' -e  's/<contUI>/taui/g' .env
                        sed -i '' -e  's/<contDB>/tadb/g' .env
                        sed -i '' -e  's/<contGraph>/tagraph/g' .env
                        sed -i '' -e 's/<internal_port>/'${TA_LOCAL_INTERNAL_SERVER_PORT}'/g' .env
                        sed -i '' -e 's/<server_port>/'${TA_EXTERNAL_PORT_SERVER}'/g' .env
                else
                        sed -i 's/<contServer>/taserver/g' .env
                        sed -i 's/<contUI>/taui/g' .env
                        sed -i 's/<contDB>/tadb/g' .env
                        sed -i 's/<contGraph>/tagraph/g' .env
                        sed -i 's/<internal_port>/'${TA_LOCAL_INTERNAL_SERVER_PORT}'/g' .env
                        sed -i 's/<server_port>/'${TA_EXTERNAL_PORT_SERVER}'/g' .env
                fi

                if uname -s | grep 'Darwin' >/dev/null; then

                        docker network create ta_local 2>/dev/null
                        docker run  -d --expose ${TA_LOCAL_INTERNAL_DB_PORT}  --net ta_local --name tadb "${tadbResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../data:/opt/couchdb/data" "${TA_LOCAL_CONFIG_COUCH_IMAGE}" #2>/dev/null

                        docker run  -d -p ${TA_EXTERNAL_UI_PORT}:${TA_LOCAL_INTERNAL_UI_PORT} --net ta_local --name taui "${tauiResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_UI_IMAGE}" #2>/dev/null

                        docker run  -d -p ${TA_EXTERNAL_PORT_SERVER}:${TA_LOCAL_INTERNAL_SERVER_PORT} --net ta_local --name taserver  "${taserverResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_SERVER_IMAGE}" #2>/dev/null

                        docker run  -d --expose 7687 --net ta_local --name tagraph "${tagraphResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../graph_data:/data" "${TA_LOCAL_CONFIG_NEO4J_IMAGE}" #2>/dev/null

                else

                        if uname -r | grep 'el7' >/dev/null; then
                                PRIVILEGED="--privileged"
                        fi

                        docker network create ta_local 2>/dev/null 
                        docker run $PRIVILEGED -d --expose ${TA_LOCAL_INTERNAL_DB_PORT}  --net ta_local --name tadb "${tadbResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../data:/opt/couchdb/data:Z" "${TA_LOCAL_CONFIG_COUCH_IMAGE}" #2>/dev/null

                        docker run $PRIVILEGED -d -p ${TA_EXTERNAL_UI_PORT}:${TA_LOCAL_INTERNAL_UI_PORT} --net ta_local --name taui "${tauiResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_UI_IMAGE}" #2>/dev/null

                        docker run $PRIVILEGED -d -p ${TA_EXTERNAL_PORT_SERVER}:${TA_LOCAL_INTERNAL_SERVER_PORT} --net ta_local --name taserver  "${taserverResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_SERVER_IMAGE}" #2>/dev/null

                        docker run $PRIVILEGED -d --expose 7687 --net ta_local --name tagraph "${tagraphResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../graph_data:/data:Z" "${TA_LOCAL_CONFIG_NEO4J_IMAGE}" #2>/dev/null

                fi
        else
                if uname -s | grep 'Darwin' >/dev/null; then
                        # sed commands modified for mac
                        sed -i '' -e  's/<contServer>/localhost/g' .env
                        sed -i '' -e  's/<contUI>/localhost/g' .env
                        sed -i '' -e  's/<contDB>/localhost/g' .env
                        sed -i '' -e  's/<contGraph>/localhost/g' .env
                        sed -i '' -e 's/<internal_port>/'${TA_LOCAL_INTERNAL_SERVER_PORT}'/g' .env
                        sed -i '' -e 's/<server_port>/'${TA_EXTERNAL_PORT_SERVER}'/g' .env
                else
                        sed -i 's/<contServer>/localhost/g' .env
                        sed -i 's/<contUI>/localhost/g' .env
                        sed -i 's/<contDB>/localhost/g' .env
                        sed -i 's/<contGraph>/localhost/g' .env
                        sed -i 's/<internal_port>/'${TA_LOCAL_INTERNAL_SERVER_PORT}'/g' .env
                        sed -i 's/<server_port>/'${TA_EXTERNAL_PORT_SERVER}'/g' .env
                fi

                # If TA_READ_VALUE_FROM_ENV_VAR is empty or false, and user is using podman, mount secrets into container
                # secretHelper contains the --secret option cmds for 4 container images
                ta_read_value_from_env_var=`grep TA_READ_VALUE_FROM_ENV_VAR .env | cut -d'=' -f 2-`
                if [[ -z "${ta_read_value_from_env_var}" || "${ta_read_value_from_env_var}" == false ]] && [[ "${TA_LOCAL_TEMP_PODMAN_INSTALL}" == true ]]; then
                        source ./secretHelper.sh
                fi

                if [[ -f ./.machine_name ]]; then


                        sleep 5
                        podman run -p 2222:5984 -d --name tadb "${tadbSecretOptions[@]}" --env-file ./env_couch.list -v "$(pwd)/../data:/opt/couchdb/data:Z" "${TA_LOCAL_CONFIG_COUCH_IMAGE}"

                        podman run -p 7474:7474 -p 7687:7687 -d --name tagraph "${tagraphSecretOptions[@]}" --env-file ./env_graph.list -v "$(pwd)/../graph_data:/data:Z"  "${TA_LOCAL_CONFIG_NEO4J_IMAGE}"

                        podman run -p 2220:9080 -d --name taserver "${taserverSecretOptions[@]}" --env-file ./env_server.list "${TA_LOCAL_CONFIG_SERVER_IMAGE}"

                        podman run -p 3000:3000 -d  --name taui "${tauiSecretOptions[@]}" --env-file ./env_ui.list "${TA_LOCAL_CONFIG_UI_IMAGE}"

                elif uname -s | grep 'Darwin' >/dev/null; then

                        podman pod create -n ta_pod -p ${TA_EXTERNAL_UI_PORT}:${TA_LOCAL_INTERNAL_UI_PORT} -p ${TA_EXTERNAL_PORT_SERVER}:${TA_LOCAL_INTERNAL_SERVER_PORT}

                        podman run  -d --pod ta_pod --name tadb "${tadbSecretOptions[@]}" "${tadbResourcesOptions[@]}" --env-file ./.env -v "tadb:/opt/couchdb/data" "${TA_LOCAL_CONFIG_COUCH_IMAGE}"

                        podman run  -d --pod ta_pod --name tagraph "${tagraphSecretOptions[@]}" "${tagraphResourcesOptions[@]}" --env-file  ./.env -v "tagraph:/data"  "${TA_LOCAL_CONFIG_NEO4J_IMAGE}"

                        podman run -d --pod ta_pod --name taserver "${taserverSecretOptions[@]}" "${taserverResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_SERVER_IMAGE}"

                        podman run -d --pod ta_pod --name taui "${tauiSecretOptions[@]}" "${tauiResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_UI_IMAGE}"


                else
                        podman pod create -n ta_pod -p ${TA_EXTERNAL_UI_PORT}:${TA_LOCAL_INTERNAL_UI_PORT} -p ${TA_EXTERNAL_PORT_SERVER}:${TA_LOCAL_INTERNAL_SERVER_PORT}

                        podman run  -d --pod ta_pod --name tadb "${tadbSecretOptions[@]}" "${tadbResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../data:/opt/couchdb/data:Z" "${TA_LOCAL_CONFIG_COUCH_IMAGE}"

                        podman run  -d --pod ta_pod --name tagraph "${tagraphSecretOptions[@]}" "${tagraphResourcesOptions[@]}" --env-file  ./.env -v "$(pwd)/../graph_data:/data:Z"  "${TA_LOCAL_CONFIG_NEO4J_IMAGE}"

                        podman run -d --pod ta_pod --name taserver "${taserverSecretOptions[@]}" "${taserverResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_SERVER_IMAGE}"

                        podman run -d --pod ta_pod --name taui "${tauiSecretOptions[@]}" "${tauiResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_UI_IMAGE}"

                fi

        cp ./env_ui.list_orig ./env_ui.list
        cp ./env_server.list_orig ./env_server.list

        fi

}

function routeViaHost {
        use_resources_if_enabled

        sed -i 's/<contServer>/localhost/g' .env
        sed -i 's/<contUI>/localhost/g' .env
        sed -i 's/<contDB>/localhost/g' .env
        sed -i 's/<contGraph>/localhost/g' .env
        sed -i 's/<internal_port>/'${TA_LOCAL_INTERNAL_SERVER_PORT}'/g' .env
        sed -i 's/<server_port>/'${TA_EXTERNAL_PORT_SERVER}'/g' .env

        podman pod create -n ta_pod --network host

        podman run  -d --pod ta_pod --name tadb "${tadbSecretOptions[@]}" "${tadbResourcesOptions[@]}" --env-file ./.env -v "$(pwd)/../data:/opt/couchdb/data:Z" "${TA_LOCAL_CONFIG_COUCH_IMAGE}"

        podman run  -d --pod ta_pod --name tagraph "${tagraphSecretOptions[@]}" "${tagraphResourcesOptions[@]}" --env-file  ./.env -v "$(pwd)/../graph_data:/data:Z"  "${TA_LOCAL_CONFIG_NEO4J_IMAGE}"

        podman run -d --pod ta_pod --name taserver "${taserverSecretOptions[@]}" "${taserverResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_SERVER_IMAGE}"

        podman run -d --pod ta_pod --name taui "${tauiSecretOptions[@]}" "${tauiResourcesOptions[@]}" --env-file ./.env "${TA_LOCAL_CONFIG_UI_IMAGE}"

        cp ./env_ui.list_orig ./env_ui.list
        cp ./env_server.list_orig ./env_server.list

}

function startLocal {

        ### Get the Docker Container IDs

        ui=$(get_ui_id -a)
        couch=$(get_db_id -a)
        server=$(get_server_id -a)
        neo4j=$(get_graph_id -a)

        ### start containers
        run_with_spinner "docker start $couch" "Starting DB Image" "Starting" "STARTED"
        run_with_spinner "docker start $server" "Starting SERVER Image" "Starting" "STARTED"
        run_with_spinner "docker start $ui" "Starting UI Image" "Starting" "STARTED"
        run_with_spinner "docker start $neo4j" "Starting NEO4J Image" "Starting" "STARTED"


}

function stopTA {
        log "Stopping running containers"
        ui=$(get_ui_id)
        couch=$(get_db_id)
        server=$(get_server_id)
        graph=$(get_graph_id)

        #### Stop All  Docker Containers
        for a in $ui; do
                log "Stopping UI container"
                docker kill $a 2>/dev/null 
        done

        for b in $couch; do
                log "Stopping DB container"
                docker kill $b 2>/dev/null 
        done

        for c in $server; do
                log "Stopping SERVER container"
                docker kill $c 2>/dev/null 
        done

        for d in $graph; do
                log "Stopping NEO4J container"
                docker stop $d 2>/dev/null 
        done

        podRun=$(which podman 2>&1)
        if [[  "$podRun" ||  $podRun =~ "/usr/bin/podman" ]] && [[  $podRun =~ "/bin/podman" ]]  && [[ $podRun =~ "/usr/sbin/podman" ]] ; then
                pod=$(docker pod ps | grep ta_pod | awk '{print $1}')

                for e in $pod; do
                        log "Stopping removing pod $e"
                        docker pod rm -f $e 2>/dev/null
                done
        fi

}

function removeContainers {
        log "Removing containers"
        ui=$(get_ui_id -a)
        couch=$(get_db_id -a)
        server=$(get_server_id -a)
        graph=$(get_graph_id -a)

        #### Remove Containers ###################

        for e in $ui; do
                log "Removing UI container"
                docker rm -f $e
        done

        for f in $couch; do
                log "Removing DB container"
                docker rm -f $f
        done

        for g in $server; do
                log "Removing SERVER container"
                docker rm -f $g
        done

        for h in $graph; do
                log "Removing NEO4J container"
                docker rm -f $h
        done

}

function removeVolumes {
        if [[ ${TA_LOCAL_TEMP_DOCKER_INSTALL} = true ]]; then
                dbVol=`docker volume ls | grep tadb | awk '{print $2}'`
                if [[ ! -z $dbVol ]] ; then
                        docker volume rm tadb tagraph
                fi
        fi

        if [[ ${TA_LOCAL_TEMP_PODMAN_INSTALL} = true ]]; then
                dbVol=`docker volume ls | grep tadb | awk '{print $2}'`
                if [[ ! -z $dbVol ]] ; then
                        docker volume rm tadb tagraph
                fi
        fi
}

