#!/usr/bin/env bash
# Created by Huang, Fuguo (aka ken) on 18.2.2021.

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------
function importSecurityEnv() {
	if [[ ! -d "../logs" ]]; then
		mkdir ../logs
	fi

	SECURITY_CONFIG=./.security_config
	SECURITY_CONFIG_DEV=./.security_config.dev

	# if there is dev file, use it and exit
	if [ -f "$SECURITY_CONFIG_DEV" ]; then
		log "$SECURITY_CONFIG_DEV is provided. Using dev file."
		source "$SECURITY_CONFIG_DEV"
		return
	fi

	# if no dev file and no prod file, do not enable authentication and exit
	# default to not auth turned on
	export TA_AUTH_LIBERTY_DISABLED=true
	export TA_AUTH_UI_DISABLED=true
	if [ ! -f "$SECURITY_CONFIG" ]; then
		return
	fi

	source "$SECURITY_CONFIG"
}

# import neo4j password file
# contains neo4j password
function importNeo4jPassEnv() {
	if [[ ! -d "../logs" ]]; then
  		mkdir ../logs
	fi
	NEO4J_PASS_FILE=./.neo4j_pass

	if [ -f "$NEO4J_PASS_FILE" ]; then
		source "$NEO4J_PASS_FILE"
	fi
}

# create .neo4j_pass file containing encrypted password and key to decrypt it
function createNeo4jPassFile() {
	getOpensslVersion
	TA_LOCAL_ENCRYPTED_NEO4J_PASSWORD=$(echo $TA_LOCAL_NEO4J_PASSWORD | openssl enc -aes-256-cbc -a -k $TA_LOCAL_KEY_PASS -pbkdf2)

	cat > .neo4j_pass <<-EOF
# created on $(date +"%H:%M:%S")
TA_LOCAL_STORED_NEO4J_ENCRYPTED_PASS=$TA_LOCAL_ENCRYPTED_NEO4J_PASSWORD
TA_LOCAL_STORED_NEO4J_PASS_KEY=$TA_LOCAL_KEY_PASS
EOF
}
# decrypt neo4j password from the .neo4j_pass file and export to environment variables
function setNeo4jPassHelper() {
	getOpensslVersion
	TA_LOCAL_NEO4J_PASSWORD=$(echo $TA_LOCAL_STORED_NEO4J_ENCRYPTED_PASS | openssl enc -aes-256-cbc -d -a -k $TA_LOCAL_STORED_NEO4J_PASS_KEY -pbkdf2)
	export TA_LOCAL_NEO4J_PASSWORD=$TA_LOCAL_NEO4J_PASSWORD
	export TA_LOCAL_DECRYPT_NEO4J_PASSWORD=$TA_LOCAL_NEO4J_PASSWORD
}

# set neo4j password, either generate it or reuse it
function setNeo4jPass() {
  	if [[ -z "${TA_LOCAL_STORED_NEO4J_ENCRYPTED_PASS:-}" || -z "${TA_LOCAL_STORED_NEO4J_PASS_KEY:-}" ]]; then
  		# password not set
  		export TA_LOCAL_KEY_PASS=$pass_key
  		export TA_LOCAL_NEO4J_PASSWORD=$admin_pwd
  		createNeo4jPassFile
  	else
  		# password exists in the file
  		setNeo4jPassHelper
  	fi
}

function setDbPasswords() {
	admin_pwd=$(openssl rand -base64 29 | tr -d "=+/" | cut -c1-25)
	pass_key=$(openssl rand -base64 29 | tr -d "=+/" | cut -c1-25)
	user_pwd=$(openssl rand -base64 29 | tr -d "=+/" | cut -c1-25)
	export TA_LOCAL_COUCHDB_PASSWORD=$admin_pwd
	export TA_LOCAL_DB_PASSWORD=$admin_pwd
	export TA_LOCAL_TA_DB_NONADMIN_PASSWORD=$user_pwd

	# Neo4J
	# set random password only if the stored file is not present, or the key is not present or wrong key
	# $NEO4J_PASS and $TA_LOCAL_STORED_NEO4J_PASS_KEY are set by importNeo4jPassEnv
	setNeo4jPass
}

# return openssl version in digits only
# e.g. OpenSSL 1.1.1i  8 Dec 2020 will return 111
function getOpensslVersion() {
	v=$(openssl version | awk '{print $2}')

	IFS=".$IFS"

	read majar minor patch <<<"$v"

	# grep digits from string
	patch_1=$(echo $patch | grep -o '[0-9]\+')

	export TA_TEMP_CURRENT_OPENSSL_VERSION="$majar$minor$patch_1"
}

function setApiKey() {
	if [[ -z "${TA_API_KEY}" ]]; then
		# TA_API_KEY is unset, will generate this value
		ta_api_key=$(openssl rand -base64 29 | tr -d "=+/" | cut -c1-25)
		export TA_API_KEY=$ta_api_key
	fi
}

function setAesKeyIV() {

	getOpensslVersion

	if [[ -z "$TA_AES_IV" || -z "$TA_AES_KEY" ]]; then
		key_iv_pair=$(openssl enc -d -a -aes-256-cbc -pbkdf2 -nosalt -P -pass pass:${TA_API_KEY} | tr -d '\n')
		
		# Set the delimiter
		IFS='iv ='

		#Read the split strings into an array based on the delimiter
		read -a string_array <<<"$key_iv_pair"
		# sample out put
		# key=E9FDB9B39A0D0E66BEB70DDBBF48CA9C908313E986C45D24FAA7E1D5347D7651iv =DEE4909389A4CD1E200A4A56A7AFD389
		# key
		# E9FDB9B39A0D0E66BEB70DDBBF48CA9C908313E986C45D24FAA7E1D5347D7651
		#
		#
		# DEE4909389A4CD1E200A4A56A7AFD389
		export TA_AES_KEY=${string_array[1]}
		export TA_AES_IV=${string_array[4]}
		unset IFS
	fi
}
