#!/usr/bin/env bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------

# 1. we run this once to get the variables
# 2. we source the alias from a file
# Check if Docker is installed
# command -v docker checks if docker exists
# docker info confirms it's working (not just present)
# Both redirect output to /dev/null to suppress noise
check_docker() {
  command -v docker &>/dev/null && docker info &>/dev/null
}

# Check if Podman is installed
check_podman() {
  command -v podman &>/dev/null && podman info &>/dev/null
}

## returns 0 or 1. 0=installed 1=not found or not executable
is_docker_installed(){
	## Identify the location of docker executables
	dockerRun=$(which docker 2>/dev/null)
	[[ -x "$dockerRun" ]]
}
## returns 0 or 1. 0=installed 1=not found or not executable
is_podman_installed(){
	## Identify the location of podman executables
	podRun=$(which podman 2>/dev/null)
	[[ -x "$podRun" ]]
}

function createPodmanAlias {
	TA_LOCAL_TEMP_RUN_ONLY_ONCE_PODMAN_ALIAS_FLAG=${TA_LOCAL_TEMP_RUN_ONLY_ONCE_PODMAN_ALIAS_FLAG:-flase}
	if [[ $TA_LOCAL_TEMP_RUN_ONLY_ONCE_PODMAN_ALIAS_FLAG = true ]]; then
		return
	fi

	# if docker is installed, we use docker
	# if podman is installed, we use podman and alias docker to podman
	# if non of these are installed, we stop here
	if ( is_docker_installed && check_docker ); then
		export TA_LOCAL_TEMP_DOCKER_INSTALL=true
		export TA_LOCAL_TEMP_RUN_ONLY_ONCE_PODMAN_ALIAS_FLAG=true
		return
	fi

	if ( is_podman_installed && check_podman ); then
		export TA_LOCAL_TEMP_PODMAN_INSTALL=true
		export TA_LOCAL_TEMP_RUN_ONLY_ONCE_PODMAN_ALIAS_FLAG=true
		# Replace docker command with podman via function
		docker() {
			podman "$@"
		}
		return
	fi

	if ! check_podman && ! check_docker; then
  		to_terminal "Error: Docker or Podman must be installed and running before proceeding to install ${PRODUCT_NAME:-}."
  		exit 1
	fi

	# If Docker is used but not working, give specific help
	if ! check_docker; then
		to_terminal "Docker is installed but not running or not configured correctly."
		to_terminal "You may need to do one or more of the following:"
		to_terminal " - Start the Docker service: sudo systemctl start docker"
		to_terminal " - Create docker group: groupadd docker"
		to_terminal " - Add your user to the docker group: sudo usermod -aG docker \$USER"
		to_terminal " - Restart docker service: service docker restart"
		to_terminal " - Enable Docker to start at boot: sudo systemctl enable docker"
		to_terminal " - Log out and back in for group changes to take effect"
		exit 1
	fi
}








