#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# --------------------------------------------- 


TA_LOCAL_CONFIG=scripts/.configuration
TA_LOCAL_CONFIG_DEV=scripts/.configuration.dev
TA_CONFIG=

# if there is dev file, use it
if [ -f "$TA_LOCAL_CONFIG_DEV" ]; then
  TA_CONFIG=$TA_LOCAL_CONFIG_DEV
else
  TA_CONFIG=$TA_LOCAL_CONFIG
fi

check_ui_port_privilege() {
    # Check if UI port is set to 443 (privileged port)
    # returns 0 if port 443 is found
    # returns 1 if not found
    grep -q "^[[:space:]]*TA_EXTERNAL_UI_PORT[[:space:]]*=[[:space:]]*443" "$TA_CONFIG"
    return $?
}

update_ui_port(){
    # update TA_EXTERNAL_UI_PORT from 443 to 3000 for rootless run
    if check_ui_port_privilege; then
        sed -i "s/TA_EXTERNAL_UI_PORT=443/TA_EXTERNAL_UI_PORT=3000/g" "$TA_CONFIG"
    fi
}

enable_lingering(){
    loginctl enable-linger "$UID"
}

create_policy(){
    SYSTEM_POLICY_FILE="/etc/containers/policy.json"
    if [[ -f "$SYSTEM_POLICY_FILE" ]]; then
        log "Copying system Podman policy.json to $PODMAN_POLICY_FILE"
        mkdir -p "$(dirname "$PODMAN_POLICY_FILE")"
        cp "$SYSTEM_POLICY_FILE" "$PODMAN_POLICY_FILE"
    else
        log "Warning: System policy file $SYSTEM_POLICY_FILE not found. Podman may not work correctly."
    fi

}


# Podman requires lingering to be enabled for rootless containers to persist beyond the user's login session
should_linger_be_enabled(){
    local user="${1:-$USER}"  # Use provided user or default to current user
    # check if linux
    # Skip check if not Linux
    if [[ "$(uname -s)" != "Linux" ]]; then
        log "Not running on Linux. Skipping lingering check."
        return 0
    fi

    # Skip check if running as root (lingering not needed for rootful Podman)
    if [[ $EUID -eq 0 ]]; then
        log "Running as root. Lingering not required. Skipping check."
        return 0
    fi

    # Skip if Podman is not available 
    if ! command -v podman &>/dev/null; then
        log "Podman is not installed. Skipping lingering check."
        return 0
    fi
    local status=$(loginctl show-user "$user" --property=Linger --value 2>/dev/null)
    log "Lingering should be anabled, linger status is $status, if there is no status check login with 'loginctl show-user'"
    [[ "$status" == "yes" ]]
}

check_policy_json_present(){
    # Use XDG_CONFIG_HOME if set, otherwise default to ~/.config
    XDG_CONFIG_HOME="${XDG_CONFIG_HOME:-$HOME/.config}"
    PODMAN_POLICY_FILE="$XDG_CONFIG_HOME/containers/policy.json"
    # 0 true - file exists
    # 1 false - files does not exist
    [[ -f "$PODMAN_POLICY_FILE" ]]
    return $?
}

check_resources_enabled() {
    # Check if UI port is set to 443 (privileged port)
    # returns 0 if true is found
    # returns 1 if not found
    grep -q "^[[:space:]]*RESOURCES[[:space:]]*=[[:space:]]*true" "$TA_CONFIG"
    return $?
}

update_resources(){
    # update RESOURCES from true to false for rootless run
    if check_resources_enabled; then
        sed -i "s/RESOURCES=true/RESOURCES=false/g" "$TA_CONFIG"
    fi
}
is_mac_os() {
	if [[ "$OSTYPE" == "darwin"* ]]; then
		return 0
	else
		return 1
	fi
}



check_configuration_file_hash(){
    #read file `.configuration_file_hash` by line and check by file `.configuration`,`.env` and `.security_config`
    if [[ -f "./.configuration_file_hash" ]]; then
        IFS=$'\n' read -r -d '' -a lines < ./.configuration_file_hash
        for line in "${lines[@]}"; do
            file_hash=$(echo "$line" | awk '{print $1}')
            file_name=$(echo "$line" | awk '{print $2}')
            if is_mac_os; then
                current_hash=$(shasum -a 256 "$file_name" | awk '{print $1}')
            else
                current_hash=$(sha256sum "$file_name" | awk '{print $1}')
            fi
            
            if [[ "$file_hash" == "$current_hash" ]]; then
                log "File $file_name hash matches, no need to reconfigure."
                continue
            else
                status_header
                to_terminal "Configuration change detected - start reconfiguration"
                log "File $file_name hash = $file_hash does not match current hash $current_hash, reconfiguring..."
        		rm -f ./.configuration_file_hash
                return 1
            fi
        done
    else
        status_header
        to_terminal "Configuration change detected - start reconfiguration"
		log "Configuration file hash file (configuration_file_hash) not found, reconfiguring..."
        return 1
    fi

}

save_config_hash_in_file(){
    if [[ ! -f "./.configuration_file_hash" ]]; then
        touch ./.configuration_file_hash

        if is_mac_os; then
            shasum -a 256 ./.configuration ./.env ./.security_config | while read -r line; do
                echo "$line" >> ./.configuration_file_hash
            done
        else
            sha256sum ./.configuration ./.env ./.security_config | while read -r line; do
                echo "$line" >> ./.configuration_file_hash
            done
        fi

        log "Created configuration file hash file (.configuration_file_hash)"
    fi
}
