#!/usr/bin/env bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------

#run scripts at relative path
scriptdir="$(dirname "$0")"

source ./.security_config
source ./localTempPodman.sh
source ./spinner.sh
source ./imageHelper.sh
source ./configuration_utils.sh

DATA_DIRS=("../data" "../graph_data")
 
check_directory_permissions(){
	# Use SUDO_USER if running via sudo, otherwise current user
	local user_name=$(whoami)
	local exit_code=0
    for dir in "$@"; do
        # Check if directory exists
        if [[ ! -d "$dir" ]]; then
            log "Directory not found: $dir (required for container volume)"
            continue
        fi


		if is_mac_os; then
			log "Using mac stat to work out directory permmissions"
	        # Use macOS compatible stat command
			permissions=$(stat -f "%Lp" "$dir")
			owner=$(stat -f "%Su" "$dir")
			group=$(stat -f "%Sg" "$dir")
		else
			log "Using linux stat to work out directory permmissions"
			# Get directory metadata
			permissions=$(stat -c "%a" "$dir")
			owner=$(stat -c "%U" "$dir")
			group=$(stat -c "%G" "$dir")
		fi
	

        log "$dir info -> Permissions: $permissions, Owner: $owner, Group: $group, Current User: $user_name"

        # Check writability
        if [[ ! -w "$dir" ]]; then
            has_error=1
        fi

        if (( permissions < 777 )); then
            log "Directory permissions too restrictive: $permissions (should be at least 777)"
            has_error=1
        fi
    done
    # Suggest appropriate fixes only if errors occurred
    if (( has_error )); then
		set_spinner_error_msg "Issues with required directories (required diretories: data / graph_data):
		- Suggested fix; grant access using 'chmod -R ugo+rwx <directory>' to the required directories, may require root permission"
    fi

    return $has_error
}

getContainerExitCode(){
	CONTAINER_ID=${1:-}
	CONTAINER_TYPE=${2:-}
	# need container to check
    if [[ -z "$CONTAINER_ID" ]]; then
		set_spinner_warn_msg "Error: No container id provided for exit code check (container = $CONTAINER_TYPE). Did it start up?"
        return 1
    fi

    local container_info
    container_info=$(docker inspect "$CONTAINER_ID" --format='{{.State.Status}}|{{.State.ExitCode}}|{{.State.OOMKilled}}' 2>/dev/null)

	# If there is no container info then it might not exist, so something has gone wrong
    if [[ -z "$container_info" ]]; then
		set_spinner_warn_msg "Error: Failed to inspect $CONTAINER_TYPE container (id = $CONTAINER_ID), it should exist, please check container logs"
        return 1
    fi

    IFS='|' read -r state exit_code oom_killed <<< "$container_info"

	log "Container ID = $CONTAINER_ID type = $CONTAINER_TYPE state: $state, exit code: $exit_code"

   	if [[ "$state" == "exited" || "$state" == "stopped" ]]; then
		if [[ "$oom_killed" == "true" ]]; then
			set_spinner_error_msg "$CONTAINER_TYPE container (id = $CONTAINER_ID) was OOMKilled (Out of Memory)"
        	return 1
		elif [[ "$exit_code" -eq 137 ]]; then
			set_spinner_error_msg "$CONTAINER_TYPE container (id = $CONTAINER_ID) exited with code 137 (SIGKILL), possibly OOM"
        	return 1
		elif [ "$exit_code" -ne 0 ]; then
			# Check permissions on graph_data and data directories 
			set_spinner_error_msg "$CONTAINER_TYPE container (id = $CONTAINER_ID) exited with non-zero code $exit_code. Check container logs for more infomation as it may not have started correctly"
       	 	return 1
		fi
	fi

	if [[ "$state" != "running" ]]; then
		set_spinner_warn_msg "$CONTAINER_TYPE container (id = $CONTAINER_ID) is not running (state: $state)"
        return 1
    fi

    log "$CONTAINER_TYPE container (id = $CONTAINER_ID) is running and healthy."
	return 0
}

unhealthy_check_exit_code(){
	local HEALTHY=${1:-true}
	local CONTAINER_ID=${2:-}
	local CONTAINER_TYPE=${3:-}
	log "Checking if $CONTAINER_TYPE container (id = $CONTAINER_ID) is healthy healthy=$HEALTHY"
	if [[ "$HEALTHY" == "false"  ]]; then
		if ! check_directory_permissions "${DATA_DIRS[@]}"; then
			# Permission issue detected error message already set in check_directory_permissions
			return 1
		fi

		# exit code 6 is used in spinner for warning
	    if ! getContainerExitCode "$CONTAINER_ID" "$CONTAINER_TYPE"; then
			get_shared_vars_msg
        	# Propagate warning (exit 6) or error (exit 1) - msg are set in getContainerExitCode
			if [[ "$SPINNER_ERROR_MSG" == *"OOM"* ]] || [[ "$SPINNER_ERROR_MSG" == *"exited with"* ]] || [[ "$SPINNER_ERROR_MSG" == *"container name provided"* ]]; then
				log "Container failed, but directory permissions are OK. Original error: $SPINNER_ERROR_MSG"
				return 1  # Still fatal, but not due to permissions
			else
				return 6  # Warning (e.g., unhealthy but not fatal)
			fi
    	fi
		set_spinner_warn_msg "$CONTAINER_TYPE container (id = $CONTAINER_ID) is unhealthy, please check container logs"
	fi
}

function checkUIPodMain {
	HEALTHY=false
	for ((i = 0; i < 150; i++)); do
		ui_port=$TA_LOCAL_INTERNAL_UI_PORT
		log "Checking UI port => ${ui_port}"

		if [ -z $ui_port ]; then
			status_header
			to_terminal "${PRODUCT_NAME} UI pod could not be started."
			to_terminal "This could indicate an incompatibility issue between the existing environment variables and the new docker images."
			to_terminal "Please uninstall and download the latest zip from ${DOC_NAME} and try again."
			to_terminal "Select the operation...."
			prompt_and_read "
				\t1) Stop ${PRODUCT_NAME}
				\t2) Uninstall ${PRODUCT_NAME} (keep database data)
				\t3) Uninstall ${PRODUCT_NAME} (remove database data)
				\t4) Quit." CHOICE

			case $CHOICE in
				1)
					to_terminal -n "Stopping ${PRODUCT_NAME}..."
					./stopLocal.sh
					to_terminal "${PRODUCT_NAME} processes are stopped."
					;;
				2)
					to_terminal "Uninstalling ${PRODUCT_NAME}..."
					./cleanUpBeforeInstall.sh
					to_terminal "Finished Uninstalling ${PRODUCT_NAME}..."
					rm -rf ../.license_accepted
					rm -rf ../logs/*
					./copy_back_files.sh
					;;
				3)
					to_terminal -n "Uninstalling ${PRODUCT_NAME}..."
					./cleanUpBeforeInstall.sh
					rm -rf ../data
					rm -rf ../graph_data
					to_terminal "Finished Uninstalling ${PRODUCT_NAME}..."
					rm -rf ../.license_accepted
					rm -rf ../logs/*
					./copy_back_files.sh
					;;
				4) exit ;;

			esac
			exit

			break
		else
			HEALTHY=true
    		break
		fi

	done
	# Need all in case it has started and stopped
	ui=$(get_ui_id -a)
	unhealthy_check_exit_code "$HEALTHY" "$ui" "UI"
	local hc_exit=$?
	log "END OF UI PORT AND RUNNING CHECK"
	return $hc_exit
}

function checkDBPod {
	HEALTHY=false
	db_id=$(get_db_id -a)
	for ((i = 0; i < 150; i++)); do
		log "Checking DB network mode"
		db_network=$(docker container inspect $db_id --format '{{.HostConfig.NetworkMode}}')
		db_port=$(docker ps --format "{{.Image}} {{.Ports}}" | grep transformation-advisor-db | awk '{print $2}')

		if [[ "$db_network" == "host" ]]; then
			if [[ -z "$db_port" ]]; then
				log "DB container is in host network mode - and no ports found. Checking status"
				# Check if DB container is running
				db_status=$(docker ps --format "{{.Image}}|{{.State}}|{{.Exited}}" | grep transformation-advisor-db)
				IFS='|' read -r image db_state db_exited <<< "$db_status"
				log "DB status => image: $image state: $db_state exited: $db_exited"
				if [[ "$db_state" == "running" ]] && [[ "$db_exited" == "false" ]]; then
					HEALTHY=true
					break		
				fi
			fi 
		fi

		log "Checking DB port => ${db_port}"
		if [ -z "$db_port" ]; then
			break
		else
			HEALTHY=true
			break
		fi

	done
	# check that couchdb returns something from curl

	for i in {1..10}
	do
		sleep 3
		COUCH_URL="http://localhost:${TA_DB_PORT:-5984}/"
		COUCH_RETURN_STATUS=$(docker exec -i "$db_id" bash -c "curl -s -o /dev/null -w '%{http_code}' ${COUCH_URL}")
		log "CouchDB curl url =  $COUCH_URL"
		log "CouchDB curl status =  $COUCH_RETURN_STATUS"
		# The container has a port but couchdb didn't actually start up
		if [[ -z "$COUCH_RETURN_STATUS" ]] || [[ "$COUCH_RETURN_STATUS" == "000" ]]; then
			log "CouchDB container curl returned 000 or empty, will sleep and check again"
			HEALTHY=false
			continue
		fi
		HEALTHY=true
		break
	done


	# Need all in case it has started and stopped
	unhealthy_check_exit_code "$HEALTHY" "$db_id" "DB"
	local hc_exit=$?
	log "END OF DB RUNNING PORT CHECK"
	return $hc_exit
}

function checkLibertyOnline() {
	#	grab the external port from config
	ta_local_external_liberty_port=$(awk '/TA_EXTERNAL_SERVER_PORT/{print}' .env)

	# if ta_local_external_liberty_port contains <server_port>, e.g.TA_LOCAL_EXTERNAL_SERVER_PORT=<server_port>
	if [[ "${ta_external_liberty_port:-}" == *"<server_port>"* ]]; then
		TA_EXTERNAL_SERVER_PORT=$(docker ps | grep "$TA_LOCAL_INTERNAL_SERVER_PORT" | awk -F "->" '{print $1}' | awk -F ":" '{print $3}')
	else
		# we could simplified to always check docker status to get external liberty port
		# set this to current env
    	export "$ta_local_external_liberty_port"
	fi

	url=$PROTOCOL://${TA_LOCAL_RUNTIME_HOST_IP}:${TA_EXTERNAL_SERVER_PORT}/lands_advisor/advisor/healthcheck

	HEALTHY=false
	for i in {1..40}
	do
		# use quotes to avoid IP's dots removed
		x=$(curl  --max-time 5 --connect-timeout 2 --retry 0 -s -k "$url" | grep "{\"status\":\"OK\"}")
		# use curl silent mode, and if x is not empty, it means TA war is running
		if [ ! -z "$x" ]; then
			log "ADVISOR is running and health is OK"
			HEALTHY=true
			break
		fi
		sleep 2
	done
	# Need all in case it has started and stopped
	server=$(get_server_id -a)
	unhealthy_check_exit_code "$HEALTHY" "$server" "ADVISOR"
	local hc_exit=$?
	log "END OF ADVISOR HEALTH CHECK"
	return $hc_exit
}

function checkNeo4jStatus(){
	graph=$(docker ps -a | grep transformation-advisor-neo4j | awk '{print $1}')
	HEALTHY=false
	for i in {1..30}
	do
		NEO4J_STATUS=$(docker exec -i $graph bash -c "neo4j status")
		if [[ "$NEO4J_STATUS" == "Neo4j is running"* ]]; then
			log "Neo4j is running"
			HEALTHY=true
			break
		fi
		sleep 1
	done
	# Need all in case it has started and stopped
	graph=$(get_graph_id -a)
	unhealthy_check_exit_code "$HEALTHY" "$graph" "GRAPH"
	local hc_exit=$?
	log "END OF GRAPH HEALTH CHECK"
	return $hc_exit
}


function podsRunningCheck(){
	log "Checking pods are running"
	checkUIPodMain
	checkDBPod
	checkLibertyOnline
	checkNeo4jStatus
}

function checkUIPodRunning(){
	run_with_spinner checkUIPodMain \
		"UI status check" \
		"Checking Status" \
		"RUNNING" 
}

function checkDBPodRunning(){
	run_with_spinner checkDBPod \
		"DB status check" \
		"Checking Status" \
		"RUNNING"
}

function checkAdvisorPodRunning(){
	run_with_spinner checkLibertyOnline \
		"Advisor status check" \
		"Checking Status" \
		"RUNNING"
}

function checkNeo4jPodRunning(){
	run_with_spinner checkNeo4jStatus \
		"Neo4j status check" \
		"Checking Status" \
		"RUNNING"
}
