#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# --------------------------------------------- 

declare -x FRAME
declare -x FRAME_INTERVAL

FRAME=("-" "\\" "|" "/")
FRAME_INTERVAL=0.25

# Internal spinner state
SPINNER_PID=
SPINNER_MSG=
GREEN="\e[32m"
RESET="\e[0m"
RED="\e[31m"
YELLOW='\e[1;33m'
# temp file for spinner msg - allows for it to be set in sub-shell
SHARED_VARS_FILE=$(mktemp)
trap 'rm -f "$SHARED_VARS_FILE"' EXIT

# Duplicate FD 3 to FD 5 at the start of the script
exec 6>&4

trap cleanup_spinner SIGINT

# Internal cleanup
cleanup_spinner() {
    if [[ -n "$SPINNER_PID" ]]; then
        kill "$SPINNER_PID" 2>/dev/null
        wait "$SPINNER_PID" 2>/dev/null
        SPINNER_PID=
        SPINNER_MSG=
    fi
}

set_shared_var() {
    echo "$1=\"$2\"" >> "$SHARED_VARS_FILE"
}

set_spinner_error_msg(){
    set_shared_var SPINNER_ERROR_MSG "$1"
}

set_spinner_warn_msg(){
    set_shared_var SPINNER_WARN_MSG "$1"
}
# Spinner background loop
start_spinner() {
    local msg="$1"
    SPINNER_MSG="$msg"
    START_TIME=$(date +%s)  # Record start time
    local spinner_short="$2" #short name for sprinner task
    # Print full message, allows wrapping
    printf '\t%s\n' "$msg" >&6

    while :; do
        ELAPSED_TIME=$(( $(date +%s) - START_TIME ))
        for frame in "${FRAME[@]}"; do
            printf "\r\t > [ %s ] (%s sec) %s ... " "$frame" "$ELAPSED_TIME" "$spinner_short" >&6
            sleep "$FRAME_INTERVAL"
        done
    done &
    SPINNER_PID=$!
}

# Stop the spinner and print final message
stop_spinner() {
    local msg="$1"
    local end_msg="$2"
    local display_icon="$3"
    local color="${4:-}"
    local ELAPSED_TIME=$(( $(date +%s) - START_TIME ))

    cleanup_spinner

    local output
    printf -v output "\\r\t > [ $display_icon ] (%s sec) %s ... ${4:-}%s$RESET\n" "$ELAPSED_TIME" "$msg" "$end_msg"
    # avoids potential permissions error i.e. 'tee: /dev/fd/6: Permission denied'
    printf '%s' "$output" >&1        # Write to log (current fd 1)
    printf '%s\n' "$output" >&6        # Write to terminal (fd 6)
}

get_shared_vars_msg(){
    # Only source if file exists and is not empty
    if [[ -f "$SHARED_VARS_FILE" ]] && [[ -s "$SHARED_VARS_FILE" ]]; then
        source "$SHARED_VARS_FILE"
    fi
}

# Run a single command with spinner
run_with_spinner() {
    local cmd="$1"
    local msg="$2"
    local spinner_short="${3:-Running}"
    local success_msg="${4:-SUCCESS}"

    # Clear the shared variables file
    # Prevent incorect messages being set during spinner cmd
    > "$SHARED_VARS_FILE"

    if [[ -z "$cmd" || -z "$msg" ]]; then
        printf "Usage: run_with_spinner <command> <message>\\n" >&2
        return 1
    fi

    trap cleanup_spinner EXIT

    start_spinner "$msg" "$spinner_short"

    # run the command which allows us to capture any output/vars that might be set in the cmd
    eval "$cmd" 2>&1 | log_stdin

    exit_code=${PIPESTATUS[0]}

    get_shared_vars_msg

    if [[  $exit_code -eq 0 ]]; then
        stop_spinner "$spinner_short" "$success_msg" "\xE2\x9C\x93" ${GREEN} 
    elif [[ $exit_code -eq 6 ]]; then
        local warn_text="${SPINNER_WARN_MSG:-WARN}"
        stop_spinner "$msg" "WARN" "\xE2\x9C\x95" "${YELLOW}"
        # Log warning message if available
        if [[ -n "$warn_text" ]]; then
            echo "$warn_text" &> >(log_warn_stdin)
        fi
        # Sleep to allow message to be printed before next line
        sleep 1
    else
        local error_text="${SPINNER_ERROR_MSG:-Operation failed with exit code $exit_code}"
        stop_spinner "$msg" "FAILED" "\xE2\x9C\x95" "${RED}"
        # Always log the error
        echo "$error_text" &> >(log_error_stdin_with_notificattion)
        # give time for message to flush
        sleep 1
        exit
    fi
    trap - EXIT
}
