#!/usr/bin/env bash
# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# ---------------------------------------------

# Define default log directory and log file name
LOG_DIR="${LOG_DIR:-logs}"
LOG_FILE="${LOG_FILE:-${LOG_DIR}/output.$(date +'%Y-%m-%d').log}"
LOG_TO_CONSOLE="${LOG_TO_CONSOLE:-false}"
RESET="\e[0m"
RED="\e[31m"
YELLOW='\e[1;33m'

# Ensure log directory exists
ensure_log_dir() {
    if [ ! -d "$LOG_DIR" ]; then
        mkdir -p "$LOG_DIR" 2>/dev/null || {
            echo "ERROR: Cannot create log directory '$LOG_DIR'" >&5
            return 1
        }
    fi
}
# Log a message with a timestamp
log() {
    local timestamp
    timestamp=$(date +"%Y-%m-%d %T")
    echo -e  "[$timestamp] $*" >> "$LOG_FILE"  # Log to file
}

to_terminal(){
    local format="$1"
    local timestamp
    timestamp=$(date +"%Y-%m-%d %T")
    # If no format string is provided, default to "%s\n"
    if [[ -z "$format" ]]; then
        printf "%b\n" "$*">&4
        printf "%s%b\n" "[$timestamp] " "$*">>"$LOG_FILE" 
    else
        # Check if the first argument looks like a format string
        if [[ "$format" == *'%'* ]]; then
            # Shift arguments and use the format string
            shift
            # empty string for format - used for timestamp in log file
            printf "$format" "" "$@">&4
            printf "%s%s\n" "[$timestamp] " "$@">>"$LOG_FILE" 
        else
            # Default to adding a newline
            printf "\t%s\n" "$*">&4
            printf "%s%s\n" "[$timestamp] " "$@">>"$LOG_FILE" 
        fi
    fi
}

log_stdin() {
    while IFS= read -r line; do
        log "$line"
    done
}

log_notification(){
    local timestamp
    timestamp=$(date +"%Y-%m-%d %T")
    local level=${1:-}
    local random_warn_id=${2:-}
    local colour=${3:-}
    local content=${4:-}
    log "${colour}$level ($random_warn_id): $content${RESET}"
}

terminal_notification(){
    local format=${1:-}
    local level=${2:-}
    local random_warn_id=${3:-}
    to_terminal $format "$level: Check logs (logs/output.$(date +'%Y-%m-%d').log) for more detail id=($random_warn_id)"
}

log_error_notification(){
    local random_error_id=${1:-}
    local content=${2:-}
    log_notification "Error" "$random_error_id" "$RED" "$content"
}


terminal_error_notification(){
    local random_warn_id=${1:-}
    terminal_notification "\t%s${RED}%s${RESET}\n\n" "Error" "$random_warn_id"
}

log_warn_notification(){
    local random_warn_id=${1:-}
    local content=${2:-}
    log_notification "Warn" "$random_warn_id" "$YELLOW" "$content"
}

terminal_warn_notification(){
    local random_warn_id=${1:-}
    terminal_notification "\t%s${YELLOW}%s${RESET}\n\n" "Warn" "$random_warn_id"
}

terminal_notification_with_content(){
    local format=${1:-}
    local level=${2:-}
    local content=${3:-}
    local random_warn_id=${4:-}
    # Strips all leading/trailing whitespace including newlines
    content="$(printf '%s' "$content")"     
    to_terminal $format "$level: $content id=($random_warn_id)"
}

terminal_warn_notification_with_content(){
    local random_warn_id=${1:-}
    local content=${2:-}
    terminal_notification_with_content "\t%s${YELLOW}%s${RESET}\n" "Warn" "$content" "$random_warn_id"
}

terminal_error_notification_with_content(){
    local random_warn_id=${1:-}
    local content=${2:-}
    terminal_notification_with_content "\t%s${RED}%s${RESET}\n" "Error" "$content" "$random_warn_id"
}

log_error_stdin() {
    local content
    IFS= read -r -d '' content
    # ^-+$'\n'? is for opensll -quiet output, *"++++++"* is for openssl output
    if [[ "$content" =~ \+{5,} || -z "$content" ]]; then
        return;
    fi
    local random_error_id=$(echo $RANDOM | md5sum | head -c 5)
    terminal_error_notification "$random_error_id"
    log_error_notification "$random_error_id" "$content"
}

log_error_stdin_with_notificattion() {
    local content
    IFS= read -r -d '' content
    # ^-+$'\n'? is for opensll -quiet output, *"++++++"* is for openssl output
    if [[ "$content" =~ \+{5,} || -z "$content" ]]; then
        return;
    fi
    local random_error_id=$(echo $RANDOM | md5sum | head -c 5)
    terminal_error_notification_with_content "$random_error_id" "$content"
    log_error_notification "$random_error_id" "$content"
}

terminal_error() {
    local content=${1:-No error content provided}
    local id=${2:-No error id provided}
    # need %s for timstamp
    printf "\n\t${RED}%s: %s id=(%s)${RESET}\n\n" "Error" "$content" "$id">&4
    log_error_notification "$id" "$content"
}

terminal_warn() {
    local content=${1:-No warn content provided}
    local id=${2:-No warn id provided}
    # need %s for timstamp
    printf "\n\t${YELLOW}%s: %s id=(%s)${RESET}\n\n" "Warn" "$content" "$id">&4
    log_warn_notification "$id" "$content"
}

log_warn_stdin() {
    local content
    IFS= read -r -d '' content
    if [[ -z "$content" ]]; then
        return;
    fi
    local random_warn_id=$(echo $RANDOM | md5sum | head -c 5)
    terminal_warn_notification "$random_warn_id"
    log_warn_notification "$random_warn_id" "$content"
}

prompt_and_read(){
    local prompt="$1"
    local var_name="$2"
    local user_input
    # Display prompt to user (on stderr to avoid being redirected)
    to_terminal "" $prompt >&4
    # Read input
    # Return after 1 character of input
    read -r -n 1 user_input >&4
    # Log both prompt and input
    # log "PROMPT: $prompt"
    log "USER INPUT: $user_input"

    # Assign input to variable
    eval "$var_name=\"\$user_input\""
}

prompt_and_read_full(){
    local prompt="$1"
    local var_name="$2"
    local user_input
    # Display prompt to user (on stderr to avoid being redirected)
    to_terminal "" $prompt >&4
    # Read input
    # Read full input. Needed for areas where input is greater than selection
    read -r user_input >&4
    # Log both prompt and input
    # log "PROMPT: $prompt"
    log "USER INPUT: $user_input"

    # Assign input to variable
    eval "$var_name=\"\$user_input\""
}

line_break(){
    # Tab + 126 dashes is the max for the line break
    local max_dash_length=126
    local standard_tab_length=8
    local terminal_col_length=$COLUMNS
    local line_break_length;

    if [[ -z $terminal_col_length ]]; then
        log "terminal col length  is unset, will try get it from stty"
        terminal_col_length=$(stty size | cut -d' ' -f2)
    fi
    if [[ -z $terminal_col_length  ]]; then
        log "terminal col length  is unset, will try get it from tput"
        terminal_col_length=$(tput cols)
    fi

    if [[ $terminal_col_length -gt $max_dash_length ]]; then
        line_break_length=$(( max_dash_length - standard_tab_length ))
    else
        line_break_length=$(( terminal_col_length - standard_tab_length ))
    fi
    local dash_string=$(printf '%*s' "$line_break_length" | tr ' ' '-')
    to_terminal "$dash_string"
}

header(){
    printf "\n\t%b\n" "$*">&4
    log $*
    line_break
}

status_header(){
    header "Status"
}

ta_product(){
    printf "
    \n
    \t _____  _    
    \t|_   _|/ \   
    \t  | | / _ \  
    \t  | |/ ___ \ 
    \t  |_/_/   \_\\
  " | tee >(cat >>"$LOG_FILE") >(cat >&4) > /dev/null
}

ama_product(){
    printf "
    \n
    \t    _    __  __    _    
    \t   / \  |  \/  |  / \   
    \t  / _ \ | |\/| | / _ \  
    \t / ___ \| |  | |/ ___ \ 
    \t/_/   \_\_|  |_/_/   \_\\
    \n
  " | tee >(cat >>"$LOG_FILE") >(cat >&4) > /dev/null
}


# Optional: Start logging stdout/stderr to file
setup_background_logging() {
    ensure_log_dir || return 1

    # Save original stdout (1) and stderr (2)
    exec 4>&1 5>&2

    # Redirect stdout to log file
    exec 1>>"$LOG_FILE"

    # Pipe stderr to both log file and terminal
    exec 2> >(tee -a "$LOG_FILE" >&4)
    export LOG_FILE=$LOG_FILE

}


trap cleanup_logging EXIT INT TERM

cleanup_logging() {
    exec 1>&4 2>&5   # Restore stdout and stderr
    exec 4>&- 5>&-
}