#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# --------------------------------------------- 


# track podman machine state checks
default_meets_requirements=false
suitable_non_default_found=false
declare -a suitable_non_default_machines  # Array to store suitable non-default machine info


is_root_user() {
    [[ "$EUID" -eq 0 ]] && [[ -n "$SUDO_USER" ]]
    return $?
}

# Mac specific as linux doesn't need machine
get_podman_machine_list() {
    #If we are running as root we need to run the podman machine list as rootless
    if is_root_user; then
        # run command as user 
        su -l "$SUDO_USER" -c "podman machine list --format '{{.Name}} {{.CPUs}} {{.Memory}}'"
    else
        podman machine list --format '{{.Name}} {{.CPUs}} {{.Memory}}'
    fi
}

get_podman_machie_current() {
    if is_root_user; then
        # run command as user 
        su -l "$SUDO_USER" -c "podman machine info --format '{{.Host.CurrentMachine}}'"
    else
        podman machine info --format '{{.Host.CurrentMachine}}'
    fi
}

podman_machine_default_is_root() {
    local is_rootless
    
    if is_root_user; then
        is_rootless=$(su -l "$SUDO_USER" -c "podman machine inspect --format '{{.Rootful}}'")
    else
        is_rootless=$(podman machine inspect --format '{{.Rootful}}')
    fi

    [[ "$is_rootless" == "true" ]]
    return $?
}

check_podman_machine_resources(){
    # Get current machine name i.e default
    local current_machine_name=$(get_podman_machie_current)

    while read -r name cpus memory_str; do
        # Extract numeric part from memory (e.g., '4GiB' -> '4')
        if [[ $memory_str =~ [+-]?[0-9]+\.?[0-9]* ]]; then
            memory_gib=${BASH_REMATCH[0]}
            # Our min memory is 8GB
            # Convert GiB to GB (1 GiB = 1.073741824 GB)
            memory_gb=$(echo "$memory_gib * 1.073741824" | bc -l)
            memory_gb=$(printf "%.2f" "$memory_gb")
        else
            echo "Error: Cannot parse memory: $memory_str" >&2
            continue
        fi
        # check default machine meets requirements
        memory_ok=$(echo "$memory_gb >= 8.0" | bc -l 2>/dev/null || echo "0")
        # If the name from the machine list is the current machine name check requirments 
        if [[ "$current_machine_name" == "$name" ]]; then
            if [[ "$cpus" -ge 4 ]] && [[ "${memory_ok%.*}" -eq 1 ]]; then
                default_meets_requirements=true
                log "podman-machine OK: Current machine '$name' has $cpus CPUs and ${memory_gib}GiB (${memory_gb}GB) memory (meets requirements)"
            else
                log "podman-machine WARNING: Current machine '$name' does NOT meet requirements: $cpus CPUs, ${memory_gib}GiB (${memory_gb}GB) memory"
            fi
        else
            # Check non default machines to see if any of them meet the requirements - giving user as much info as possible
            if [[ "$cpus" -ge 4 ]] && [[ "${memory_ok%.*}" -eq 1 ]]; then
                suitable_non_default_found=true
                suitable_non_default_machines+=("$name:$cpus:$memory_gib:$memory_gb")
		    fi
            log "podman-machine INFO: Non-default machine '$name' has $cpus CPUs and ${memory_gib}GiB (${memory_gb}GB) memory"
        fi
    done < <(get_podman_machine_list)

    if [[ "$default_meets_requirements" == false ]]; then 
        if [[ "$suitable_non_default_found" == true ]]; then
            machine_list=""
            first_suitable=""
            for machine in "${suitable_non_default_machines[@]}"; do
                IFS=':' read -r mname mcpus mgib mgb <<< "$machine"
                machine_list+="'$mname' ($mcpus CPUs, ${mgib}GiB / ${mgb}GB), "
                # Suggest starting the first suitable one
                if [[ -z "$first_suitable" ]]; then
                    first_suitable="podman machine start $mname"
                fi
            done
            # Remove the trainiling comma left from list generation
            machine_list=${machine_list%, }
            terminal_warn_notification_with_content "podman-machine-non-default" "Current machine does not meet requirements, but the following non-default machines do: $machine_list"
            terminal_warn_notification_with_content "podman-machine-non-default-set" "Consider using one via: '$first_suitable', or update the current machine with: 'podman stop; podman machine set --cpus=4 --memory=8000; podman start'"
        else
            terminal_warn_notification_with_content "podman-machine-default" "Current machine does not meet requirements and no other suitable machine was found."
		    terminal_warn_notification_with_content "podman-machine-default-set" "Please update the current machine by running 'podman stop; podman machine set --cpus=4 --memory=8000; podman start'"
        fi
        exit 1       
    fi
}