#!/usr/bin/env bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2021
# ---------------------------------------------


#run scripts at relative path
scriptdir="$(dirname "$0")"
cd "$scriptdir"
source ./logging.sh
source ./spinner.sh
source ./input_validation.sh

source ./.security_config

###create the podman/docker alias
source ./createPodmanAlias.sh
createPodmanAlias

source ./enableSecurity.sh
# enable security environment, if conditions are met.
importSecurityEnv
source ./checkPodsRunning.sh
source ./imageHelper.sh
source ./secretHelper.sh
source ./findHost.sh
source ./container_manager_utils.sh
source ./launch_utils.sh
source ./configuration_utils.sh

header "Prerequisites"

function stopLaunch(){
	to_terminal "\n\t%s" "Interrupted, stopping launch of ${PRODUCT_NAME}"
	exit 1
}


function fullUninstall(){
	./cleanUpBeforeInstall.sh
	rm -rf ../data
	rm -rf ../graph_data
	rm -rf ./.neo4j_pass
	rm -rf ../.license_accepted
	./copy_back_files.sh
	removeVolumes
	rm -rf ../key
	rm -rf ../swidtag
}

function partialUninstall(){
	./cleanUpBeforeInstall.sh
	rm -rf ../.license_accepted
	./copy_back_files.sh
	rm -rf ../key
	rm -rf ../swidtag
}

# if does not contain "command not found", we have docker
if [[ ${TA_LOCAL_TEMP_DOCKER_INSTALL:-false} = true ]]; then
	to_terminal "Docker installed."
	echo ""
	sed -i 's/TA_READ_VALUE_FROM_ENV_VAR=false/TA_READ_VALUE_FROM_ENV_VAR=true/g' .env
	sed -i 's/TA_READ_VALUE_FROM_ENV_VAR=false/TA_READ_VALUE_FROM_ENV_VAR=true/g' .env_orig
	echo ""
elif [[ ${TA_LOCAL_TEMP_PODMAN_INSTALL:-false} = true ]]; then
	to_terminal "" "\tPodman installed.\n"
	echo ""
	echo ""
fi

reconfigure_local_config(){
	header "Re-initializing Configuration"
	removeConfigSecrets
	rm -f ./.configuration_file_hash
	run_with_spinner "./reloadConfig.sh" "Cleaning up before re-initializing ${PRODUCT_NAME} configuration" "Cleaning" "CLEANED"
	./copy_back_files.sh
	./installLocal.sh
	sleep 3
	save_config_hash_in_file
	./renameDockerName.sh &>/dev/null
}

image_and_pod_check_for_launch(){
	ui=$(get_ui_id)
	couch=$(get_db_id)
	server=$(get_server_id)
	neo4j=$(get_graph_id)
	if [[ "$image_status" == "all-present" ]]; then
		if ! ( host_matches_env_file && check_configuration_file_hash) ; then
			# there is a mis match in the host IP and the IP in our env file. Let's reconfigure
			reconfigure_local_config
			return 0
		fi

		if [ -z $server ] || [ -z $ui ] || [ -z $couch ] || [ -z $neo4j ]; then
			status_header
			to_terminal "${PRODUCT_NAME} images installed but not running."
			to_terminal "Please select option 5 to start ${PRODUCT_NAME} or option 8 if the config has changed."
		else
			# if TA is running, display the status

			# set the protocol, reset the protocol, as startDockerCompose set to something else
			PROTOCOL=http
			# translate into lower case
			TA_AUTH_ENABLE_TLS_lower_case=$(echo "$TA_AUTH_ENABLE_TLS" | tr '[:upper:]' '[:lower:]')

			if [[ $TA_AUTH_ENABLE_TLS_lower_case == 'true' ]]; then
				PROTOCOL=https
			fi

			# this will print dots like ... in a background job
			source ./helper.sh
			# showConfiguringTA "Checking Status.."
			header "Checking Status of pods"

			########### testing for ui, couchDB and liberty containers are running #######
			checkUIPodRunning
			checkDBPodRunning
			checkNeo4jPodRunning
			checkAdvisorPodRunning
			##############################################################################
			# Save config file
			save_config_hash_in_file

			displayTARunningStatus
		fi
	fi
}

image_and_pod_check_for_launch


#If an argument is passed in, then use it and dont wait for prompt response
if [[ ! -z ${1:-} ]]; then
	OPERATION=$1
	if [[ $OPERATION -gt 0 ]] && [[ $OPERATION -lt 10 ]] ; then
		log "COMMAND LINE OPERATION SELECTED: $1"
	else
		to_terminal "INVALID OPERATION SELECTED: $1"
	fi
else
	header "Select the operation"
	prompt_and_read "
	\t1) Install ${PRODUCT_NAME}\n
	\t2) Uninstall ${PRODUCT_NAME} (keep database data)\n
	\t3) Uninstall ${PRODUCT_NAME} (remove database data)\n
	\t4) Stop ${PRODUCT_NAME} \n
	\t5) Start ${PRODUCT_NAME} \n
	\t6) Check for latest ${PRODUCT_NAME} \n
	\t7) Working in an Air Gapped Environment\n
	\t8) Re-initialize Configuration\n
	\t9) Run Debug Report\n
	\tq) Quit " OPERATION
	basic_input_validation $OPERATION 1 9
fi

case $OPERATION in
1)
	run_with_spinner "./cleanUpBeforeInstall.sh" "Cleaning up before installing ${PRODUCT_NAME}" "Cleaning" "CLEANED"
	./installLocal.sh
	sleep 3
	save_config_hash_in_file
	./renameDockerName.sh&>/dev/null
	;;
2)
	run_with_spinner partialUninstall "Uninstalling ${PRODUCT_NAME}" "Uninstalling"
	#Remove after spinner so print can complete
	rm -rf ../logs
	;;
3)
	run_with_spinner fullUninstall "Uninstalling ${PRODUCT_NAME}" "Uninstalling"
	#Remove after spinner so print can complete
	rm -rf ../logs
	;;
4)
	# to_terminal  "Stopping ${PRODUCT_NAME} ..."
	run_with_spinner ./stopLocal.sh "Stopping ${PRODUCT_NAME}" "Stopping" "STOPPED"
	status_header
	to_terminal "${PRODUCT_NAME} processes are stopped."
	;;
5)
	header "Starting ${PRODUCT_NAME}"
	./startLocal.sh
	;;
6)
	# to_terminal  "Checking if newer version of ${PRODUCT_NAME} is available..."
	./checkforLatest.sh
	;;
7) 
	header "Air-gap Installation"
	./fileImages.sh 
	;;
8) 
	reconfigure_local_config
	;;

9)  
	local_info="${SHORT_NAME}LocalInfo/${SHORT_NAME}LocalInfo_$date.txt"
	message_log="${SHORT_NAME}LocalInfo/message.log_$date.txt"
	run_with_spinner "./getLocalEnvInfo.sh ${local_info} ${message_log}" "Generating debug report for ${PRODUCT_NAME}" "Creating report" "CREATED"
	line_break
	to_terminal "Local Debug log is available at ${local_info}"
	to_terminal "Local Message log is available at ${message_log}"
	line_break
	;;

q) exit ;;
esac

