#!/bin/bash

# ---------------------------------------------
# Licensed Materials - Property of IBM
# (C) Copyright IBM Corporation 2025
# ---------------------------------------------

function createSecret(){
  secret_to_creat=$1
  var_to_check=$2
  val_to_add=$3

  # The secret secret_to_creat does not exist, create a secreat
  secret_inspect=`podman secret inspect $secret_to_creat 2>/dev/null`
  if [[ "${secret_inspect}" == '[]' ]]; then

    # If the .env file includes var_to_check with a value, use its value to create secret
    val_from_env_file=`grep $var_to_check .env | cut -d'=' -f 2- | awk '!/^[[:space:]]*$/'`
    if [ ! -z "${val_from_env_file}" ]; then
      val="${val_from_env_file//[[:space:]]/}"
    else
      # If val_to_add has a value, use its value to create secret
      if [ ! -z "${val_to_add}" ]; then
        val=$val_to_add

      # Otherwise, generate a random value to create secret
      else
        val=$(openssl rand -base64 29 | tr -d "=+/" | cut -c1-25)
      fi
    fi
    log "[INFO] Create secret ${secret_to_creat} ......"
    printf $val | podman secret create "${secret_to_creat}" -

  # the secret exists already
  else 
    log "[INFO] Secret ${secret_to_creat} already created"
  fi
}

function createSecretForKeyAndCert(){
  secret_to_creat=$1
  var_to_check=$2
  val_to_add=$3

  # The secret secret_to_creat does not exist, create a secreat
  secret_inspect=`podman secret inspect $secret_to_creat 2>/dev/null`
  if [[ "${secret_inspect}" == '[]' ]]; then
      TA_AUTH_ENABLE_TLS_lower_case=$(echo "$TA_AUTH_ENABLE_TLS" | tr '[:upper:]' '[:lower:]')
      if [[ $TA_AUTH_ENABLE_TLS_lower_case == 'true' ]]; then
          createSecret "${secret_to_creat}" "${var_to_check}" "${val_to_add}"
      else
        log "[INFO] Create secret ${secret_to_creat} ......"
        printf " " | podman secret create "${secret_to_creat}" -
      fi
  # the secret exists already
  else 
    log "[INFO] Secret ${secret_to_creat} already created"
  fi
}

#
# Check if the TA secret exists and if not create it.
#
function checkAndCreateTaSecret() {
  createSecret "ta_db_user" "TA_DB_USER" ""
  createSecret "ta_db_password" "TA_LOCAL_DB_PASSWORD" "${TA_LOCAL_DB_PASSWORD}"
  createSecret "ta_db_nonadmin_user" "TA_DB_NONADMIN_USER" ""
  createSecret "ta_db_nonadmin_password" "TA_LOCAL_TA_DB_NONADMIN_PASSWORD" "${TA_LOCAL_TA_DB_NONADMIN_PASSWORD}"
  createSecret "ta_aes_iv" "TA_AES_IV" "${TA_AES_IV}"
  createSecret "ta_aes_key" "TA_AES_KEY" "${TA_AES_KEY}"
  createSecret "ta_neo4j_user" "NEO4J_USER" "neo4j"
  createSecret "ta_neo4j_pass" "NEO4J_PASS" "${TA_LOCAL_NEO4J_PASSWORD}"
  createSecret "ta_neo4j_auth" "NEO4J_AUTH" "neo4j/${TA_LOCAL_NEO4J_PASSWORD}"

  TA_AUTH_OIDC_CLIENT_ID=`grep TA_AUTH_OIDC_CLIENT_ID .security_config | cut -d'=' -f 2- | awk '!/^[[:space:]]*$/'`
  TA_AUTH_OIDC_CLIENT_SECRET=`grep TA_AUTH_OIDC_CLIENT_SECRET .security_config | cut -d'=' -f 2- | awk '!/^[[:space:]]*$/'`
  createSecret "ta_auth_oidc_client_id" "dummy" "${TA_AUTH_OIDC_CLIENT_ID}"
  createSecret "ta_auth_oidc_client_secret" "dummy" "${TA_AUTH_OIDC_CLIENT_SECRET}"

  createSecretForKeyAndCert "ta_private_key" "TA_PRIVATE_KEY" "${TA_PRIVATE_KEY}"
  createSecretForKeyAndCert "ta_public_key" "TA_PUBLIC_KEY" "${TA_PUBLIC_KEY}"
}

function removeConfigSecrets(){
  podman secret rm ta_auth_oidc_client_id 2>/dev/null
  podman secret rm ta_auth_oidc_client_secret 2>/dev/null
  podman secret rm ta_aes_iv 2>/dev/null
  podman secret rm ta_aes_key 2>/dev/null
  podman secret rm ta_private_key 2>/dev/null
  podman secret rm ta_public_key 2>/dev/null
}

function removeAllTaSecrets(){
  podman secret rm ta_db_user 2>/dev/null
  podman secret rm ta_db_password 2>/dev/null
  podman secret rm ta_db_nonadmin_user 2>/dev/null
  podman secret rm ta_db_nonadmin_password 2>/dev/null
  podman secret rm ta_private_key 2>/dev/null
  podman secret rm ta_public_key 2>/dev/null
  podman secret rm ta_neo4j_user 2>/dev/null
  podman secret rm ta_neo4j_pass 2>/dev/null
  podman secret rm ta_neo4j_auth 2>/dev/null
  removeConfigSecrets
}

tadbSecretOptions=(
--secret ta_public_key,target=/opt/couchdb/ta-couchdb-config/ta_public_key
--secret ta_private_key,target=/opt/couchdb/ta-couchdb-config/ta_private_key
--secret ta_db_user,target=/opt/couchdb/ta-couchdb-config/db_username
--secret ta_db_password,target=/opt/couchdb/ta-couchdb-config/secret
)

tagraphSecretOptions=(
--secret ta_public_key,target=/var/lib/neo4j/ta-neo4j-config/ta_public_key
--secret ta_private_key,target=/var/lib/neo4j/ta-neo4j-config/ta_private_key
--secret ta_neo4j_auth,target=/var/lib/neo4j/ta-neo4j-config/neo4j_auth
)

taserverSecretOptions=(
--secret ta_auth_oidc_client_id,target=/etc/ta-config/clientId
--secret ta_auth_oidc_client_secret,target=/etc/ta-config/clientSecret
--secret ta_db_nonadmin_user,target=/etc/ta-config/db_nonadmin_user
--secret ta_db_nonadmin_password,target=/etc/ta-config/db_nonadmin_secret
--secret ta_db_user,target=/etc/ta-config/db_username
--secret ta_db_password,target=/etc/ta-config/secret
--secret ta_public_key,target=/etc/ta-config/ta_public_key
--secret ta_private_key,target=/etc/ta-config/ta_private_key
--secret ta_neo4j_user,target=/etc/ta-config/neo4j_username
--secret ta_neo4j_pass,target=/etc/ta-config/neo4j_secret
--secret ta_neo4j_auth,target=/etc/ta-config/neo4j_auth
--secret ta_aes_iv,target=/etc/ta-config/ta_aes_iv
--secret ta_aes_key,target=/etc/ta-config/ta_aes_key
--secret ta_public_key,type=env,target=TA_PUBLIC_KEY
--secret ta_private_key,type=env,target=TA_PRIVATE_KEY
--secret ta_neo4j_auth,type=env,target=NEO4J_AUTH
--secret ta_neo4j_user,type=env,target=NEO4J_USER
--secret ta_neo4j_pass,type=env,target=NEO4J_PASS
)

tauiSecretOptions=(
--secret ta_auth_oidc_client_id,target=/etc/ta-ui-config/clientId
--secret ta_auth_oidc_client_secret,target=/etc/ta-ui-config/clientSecret
--secret ta_db_nonadmin_user,target=/etc/ta-ui-config/db_nonadmin_user
--secret ta_db_nonadmin_password,target=/etc/ta-ui-config/db_nonadmin_secret
--secret ta_db_user,target=/etc/ta-ui-config/db_username
--secret ta_db_password,target=/etc/ta-ui-config/secret
--secret ta_public_key,target=/etc/ta-ui-config/ta_public_key
--secret ta_private_key,target=/etc/ta-ui-config/ta_private_key
--secret ta_neo4j_auth,target=/etc/ta-ui-config/neo4j_auth
--secret ta_neo4j_user,target=/etc/ta-ui-config/neo4j_username
--secret ta_neo4j_pass,target=/etc/ta-ui-config/neo4j_secret
--secret ta_aes_iv,target=/etc/ta-ui-config/ta_aes_iv
--secret ta_aes_key,target=/etc/ta-ui-config/ta_aes_key
)