/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.common.runtime;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.compress.Compression;
import org.apache.kafka.common.errors.CoordinatorLoadInProgressException;
import org.apache.kafka.common.errors.NotCoordinatorException;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.ControlRecordType;
import org.apache.kafka.common.record.EndTransactionMarker;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.TransactionResult;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.coordinator.common.runtime.CoordinatorEvent;
import org.apache.kafka.coordinator.common.runtime.CoordinatorEventProcessor;
import org.apache.kafka.coordinator.common.runtime.CoordinatorExecutorImpl;
import org.apache.kafka.coordinator.common.runtime.CoordinatorLoader;
import org.apache.kafka.coordinator.common.runtime.CoordinatorMetrics;
import org.apache.kafka.coordinator.common.runtime.CoordinatorResult;
import org.apache.kafka.coordinator.common.runtime.CoordinatorRuntimeMetrics;
import org.apache.kafka.coordinator.common.runtime.CoordinatorShard;
import org.apache.kafka.coordinator.common.runtime.CoordinatorShardBuilderSupplier;
import org.apache.kafka.coordinator.common.runtime.CoordinatorTimer;
import org.apache.kafka.coordinator.common.runtime.PartitionWriter;
import org.apache.kafka.coordinator.common.runtime.Serializer;
import org.apache.kafka.coordinator.common.runtime.SnapshottableCoordinator;
import org.apache.kafka.deferred.DeferredEvent;
import org.apache.kafka.deferred.DeferredEventQueue;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.server.util.timer.Timer;
import org.apache.kafka.server.util.timer.TimerTask;
import org.apache.kafka.storage.internals.log.LogConfig;
import org.apache.kafka.storage.internals.log.VerificationGuard;
import org.apache.kafka.timeline.SnapshotRegistry;
import org.slf4j.Logger;

public class CoordinatorRuntime<S extends CoordinatorShard<U>, U>
implements AutoCloseable {
    static final int INITIAL_BUFFER_SIZE = 524288;
    private final String logPrefix;
    private final LogContext logContext;
    private final Logger log;
    private final Time time;
    private final Timer timer;
    private final Duration defaultWriteTimeout;
    private final ConcurrentHashMap<TopicPartition, CoordinatorContext> coordinators;
    private final CoordinatorEventProcessor processor;
    private final PartitionWriter partitionWriter;
    private final CoordinatorLoader<U> loader;
    private final CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier;
    private final CoordinatorRuntimeMetrics runtimeMetrics;
    private final CoordinatorMetrics coordinatorMetrics;
    private final Serializer<U> serializer;
    private final Compression compression;
    private final int appendLingerMs;
    private final ExecutorService executorService;
    private final AtomicBoolean isRunning = new AtomicBoolean(true);
    private volatile MetadataImage metadataImage = MetadataImage.EMPTY;

    private CoordinatorRuntime(String logPrefix, LogContext logContext, CoordinatorEventProcessor processor, PartitionWriter partitionWriter, CoordinatorLoader<U> loader, CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier, Time time, Timer timer, Duration defaultWriteTimeout, CoordinatorRuntimeMetrics runtimeMetrics, CoordinatorMetrics coordinatorMetrics, Serializer<U> serializer, Compression compression, int appendLingerMs, ExecutorService executorService) {
        this.logPrefix = logPrefix;
        this.logContext = logContext;
        this.log = logContext.logger(CoordinatorRuntime.class);
        this.time = time;
        this.timer = timer;
        this.defaultWriteTimeout = defaultWriteTimeout;
        this.coordinators = new ConcurrentHashMap();
        this.processor = processor;
        this.partitionWriter = partitionWriter;
        this.loader = loader;
        this.coordinatorShardBuilderSupplier = coordinatorShardBuilderSupplier;
        this.runtimeMetrics = runtimeMetrics;
        this.coordinatorMetrics = coordinatorMetrics;
        this.serializer = serializer;
        this.compression = compression;
        this.appendLingerMs = appendLingerMs;
        this.executorService = executorService;
    }

    private void throwIfNotRunning() {
        if (!this.isRunning.get()) {
            throw Errors.NOT_COORDINATOR.exception();
        }
    }

    private void enqueueLast(CoordinatorEvent event) {
        try {
            this.processor.enqueueLast(event);
        }
        catch (RejectedExecutionException ex) {
            throw new NotCoordinatorException("Can't accept an event because the processor is closed", (Throwable)ex);
        }
    }

    private void enqueueFirst(CoordinatorEvent event) {
        try {
            this.processor.enqueueFirst(event);
        }
        catch (RejectedExecutionException ex) {
            throw new NotCoordinatorException("Can't accept an event because the processor is closed", (Throwable)ex);
        }
    }

    CoordinatorContext maybeCreateContext(TopicPartition tp) {
        return this.coordinators.computeIfAbsent(tp, x$0 -> new CoordinatorContext((TopicPartition)x$0));
    }

    CoordinatorContext contextOrThrow(TopicPartition tp) throws NotCoordinatorException {
        CoordinatorContext context = this.coordinators.get(tp);
        if (context == null) {
            throw Errors.NOT_COORDINATOR.exception();
        }
        return context;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void withActiveContextOrThrow(TopicPartition tp, Consumer<CoordinatorContext> func) throws NotCoordinatorException, CoordinatorLoadInProgressException {
        block5: {
            CoordinatorContext context = this.contextOrThrow(tp);
            try {
                context.lock.lock();
                if (context.state == CoordinatorState.ACTIVE) {
                    func.accept(context);
                    break block5;
                }
                if (context.state == CoordinatorState.LOADING) {
                    throw Errors.COORDINATOR_LOAD_IN_PROGRESS.exception();
                }
                throw Errors.NOT_COORDINATOR.exception();
            }
            finally {
                context.lock.unlock();
            }
        }
    }

    public <T> CompletableFuture<T> scheduleWriteOperation(String name, TopicPartition tp, Duration timeout, CoordinatorWriteOperation<S, T, U> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of write operation {}.", (Object)name);
        CoordinatorWriteEvent<T> event = new CoordinatorWriteEvent<T>(name, tp, timeout, op);
        this.enqueueLast(event);
        return event.future;
    }

    public <T> List<CompletableFuture<T>> scheduleWriteAllOperation(String name, Duration timeout, CoordinatorWriteOperation<S, T, U> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of write all operation {}.", (Object)name);
        return this.coordinators.keySet().stream().map(tp -> this.scheduleWriteOperation(name, (TopicPartition)tp, timeout, op)).collect(Collectors.toList());
    }

    public <T> CompletableFuture<T> scheduleTransactionalWriteOperation(String name, TopicPartition tp, String transactionalId, long producerId, short producerEpoch, Duration timeout, CoordinatorWriteOperation<S, T, U> op, int apiVersion) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of transactional write operation {}.", (Object)name);
        return this.partitionWriter.maybeStartTransactionVerification(tp, transactionalId, producerId, producerEpoch, apiVersion).thenCompose(verificationGuard -> {
            CoordinatorWriteEvent event = new CoordinatorWriteEvent(name, tp, transactionalId, producerId, producerEpoch, (VerificationGuard)verificationGuard, timeout, op);
            this.enqueueLast(event);
            return event.future;
        });
    }

    public CompletableFuture<Void> scheduleTransactionCompletion(String name, TopicPartition tp, long producerId, short producerEpoch, int coordinatorEpoch, TransactionResult result, Duration timeout) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of transaction completion for {} with producer id={}, producer epoch={}, coordinator epoch={} and transaction result={}.", new Object[]{tp, producerId, producerEpoch, coordinatorEpoch, result});
        CoordinatorCompleteTransactionEvent event = new CoordinatorCompleteTransactionEvent(name, tp, producerId, producerEpoch, coordinatorEpoch, result, timeout);
        this.enqueueLast(event);
        return event.future;
    }

    public <T> CompletableFuture<T> scheduleReadOperation(String name, TopicPartition tp, CoordinatorReadOperation<S, T> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of read operation {}.", (Object)name);
        CoordinatorReadEvent<T> event = new CoordinatorReadEvent<T>(name, tp, op);
        this.enqueueLast(event);
        return event.future;
    }

    public <T> List<CompletableFuture<T>> scheduleReadAllOperation(String name, CoordinatorReadOperation<S, T> op) {
        this.throwIfNotRunning();
        this.log.debug("Scheduled execution of read all operation {}.", (Object)name);
        return this.coordinators.keySet().stream().map(tp -> this.scheduleReadOperation(name, (TopicPartition)tp, op)).collect(Collectors.toList());
    }

    private void scheduleInternalOperation(String name, TopicPartition tp, Runnable op) {
        this.log.debug("Scheduled execution of internal operation {}.", (Object)name);
        this.enqueueLast(new CoordinatorInternalEvent(name, tp, op));
    }

    public void scheduleLoadOperation(TopicPartition tp, int partitionEpoch) {
        this.throwIfNotRunning();
        this.log.info("Scheduling loading of metadata from {} with epoch {}", (Object)tp, (Object)partitionEpoch);
        this.maybeCreateContext(tp);
        this.scheduleInternalOperation("Load(tp=" + String.valueOf(tp) + ", epoch=" + partitionEpoch + ")", tp, () -> {
            CoordinatorContext context = this.maybeCreateContext(tp);
            context.lock.lock();
            try {
                if (context.epoch < partitionEpoch) {
                    context.epoch = partitionEpoch;
                    switch (context.state.ordinal()) {
                        case 0: 
                        case 4: {
                            context.transitionTo(CoordinatorState.LOADING);
                            break;
                        }
                        case 1: {
                            this.log.info("The coordinator {} is already loading metadata.", (Object)tp);
                            break;
                        }
                        case 2: {
                            this.log.info("The coordinator {} is already active.", (Object)tp);
                            break;
                        }
                        default: {
                            this.log.error("Cannot load coordinator {} in state {}.", (Object)tp, (Object)context.state);
                            break;
                        }
                    }
                } else {
                    this.log.info("Ignored loading metadata from {} since current epoch {} is larger than or equals to {}.", new Object[]{context.tp, context.epoch, partitionEpoch});
                }
            }
            finally {
                context.lock.unlock();
            }
        });
    }

    public void scheduleUnloadOperation(TopicPartition tp, OptionalInt partitionEpoch) {
        this.throwIfNotRunning();
        this.log.info("Scheduling unloading of metadata for {} with epoch {}", (Object)tp, (Object)partitionEpoch);
        this.scheduleInternalOperation("UnloadCoordinator(tp=" + String.valueOf(tp) + ", epoch=" + String.valueOf(partitionEpoch) + ")", tp, () -> {
            CoordinatorContext context = this.coordinators.get(tp);
            if (context != null) {
                context.lock.lock();
                try {
                    if (partitionEpoch.isEmpty() || context.epoch < partitionEpoch.getAsInt()) {
                        this.log.info("Started unloading metadata for {} with epoch {}.", (Object)tp, (Object)partitionEpoch);
                        try {
                            context.transitionTo(CoordinatorState.CLOSED);
                            this.log.info("Finished unloading metadata for {} with epoch {}.", (Object)tp, (Object)partitionEpoch);
                            this.coordinators.remove(tp, context);
                        }
                        catch (Throwable ex) {
                            try {
                                this.log.error("Failed to unload metadata for {} with epoch {} due to {}.", new Object[]{tp, partitionEpoch, ex.toString()});
                            }
                            catch (Throwable throwable) {
                                throw throwable;
                            }
                            finally {
                                this.coordinators.remove(tp, context);
                            }
                        }
                    }
                    this.log.info("Ignored unloading metadata for {} in epoch {} since current epoch is {}.", new Object[]{tp, partitionEpoch, context.epoch});
                }
                finally {
                    context.lock.unlock();
                }
            } else {
                this.log.info("Ignored unloading metadata for {} in epoch {} since metadata was never loaded.", (Object)tp, (Object)partitionEpoch);
            }
        });
    }

    public void onNewMetadataImage(MetadataImage newImage, MetadataDelta delta) {
        this.throwIfNotRunning();
        this.log.debug("Scheduling applying of a new metadata image with offset {}.", (Object)newImage.offset());
        this.metadataImage = newImage;
        ((ConcurrentHashMap.KeySetView)this.coordinators.keySet()).forEach(tp -> this.scheduleInternalOperation("UpdateImage(tp=" + String.valueOf(tp) + ", offset=" + newImage.offset() + ")", (TopicPartition)tp, () -> {
            CoordinatorContext context = this.coordinators.get(tp);
            if (context != null) {
                context.lock.lock();
                try {
                    if (context.state == CoordinatorState.ACTIVE) {
                        this.log.debug("Applying new metadata image with offset {} to {}.", (Object)newImage.offset(), tp);
                        context.coordinator.onNewMetadataImage(newImage, delta);
                    }
                    this.log.debug("Ignored new metadata image with offset {} for {} because the coordinator is not active.", (Object)newImage.offset(), tp);
                }
                finally {
                    context.lock.unlock();
                }
            } else {
                this.log.debug("Ignored new metadata image with offset {} for {} because the coordinator does not exist.", (Object)newImage.offset(), tp);
            }
        }));
    }

    @Override
    public void close() throws Exception {
        if (!this.isRunning.compareAndSet(true, false)) {
            this.log.warn("Coordinator runtime is already shutting down.");
            return;
        }
        this.log.info("Closing coordinator runtime.");
        Utils.closeQuietly(this.loader, (String)"loader");
        Utils.closeQuietly((AutoCloseable)this.timer, (String)"timer");
        Utils.closeQuietly((AutoCloseable)this.processor, (String)"event processor");
        this.coordinators.forEach((tp, context) -> {
            context.lock.lock();
            try {
                context.transitionTo(CoordinatorState.CLOSED);
            }
            catch (Throwable ex) {
                this.log.warn("Failed to unload metadata for {} due to {}.", new Object[]{tp, ex.getMessage(), ex});
            }
            finally {
                context.lock.unlock();
            }
        });
        this.coordinators.clear();
        this.executorService.shutdown();
        Utils.closeQuietly((AutoCloseable)this.runtimeMetrics, (String)"runtime metrics");
        this.log.info("Coordinator runtime closed.");
    }

    public List<TopicPartition> activeTopicPartitions() {
        if (this.coordinators == null || this.coordinators.isEmpty()) {
            return List.of();
        }
        return this.coordinators.entrySet().stream().filter(entry -> ((CoordinatorContext)entry.getValue()).state.equals((Object)CoordinatorState.ACTIVE)).map(Map.Entry::getKey).toList();
    }

    class CoordinatorContext {
        final ReentrantLock lock = new ReentrantLock();
        final TopicPartition tp;
        final LogContext logContext;
        final DeferredEventQueue deferredEventQueue;
        final EventBasedCoordinatorTimer timer;
        final CoordinatorExecutorImpl<S, U> executor;
        CoordinatorState state;
        int epoch;
        SnapshottableCoordinator<S, U> coordinator;
        HighWatermarkListener highWatermarklistener;
        BufferSupplier bufferSupplier;
        CoordinatorBatch currentBatch;

        private CoordinatorContext(TopicPartition tp) {
            this.tp = tp;
            this.logContext = new LogContext(String.format("[%s topic=%s partition=%d] ", CoordinatorRuntime.this.logPrefix, tp.topic(), tp.partition()));
            this.state = CoordinatorState.INITIAL;
            this.epoch = -1;
            this.deferredEventQueue = new DeferredEventQueue(this.logContext);
            this.timer = new EventBasedCoordinatorTimer(tp, this.logContext);
            this.executor = new CoordinatorExecutorImpl(this.logContext, tp, CoordinatorRuntime.this, CoordinatorRuntime.this.executorService, CoordinatorRuntime.this.defaultWriteTimeout);
            this.bufferSupplier = new BufferSupplier.GrowableBufferSupplier();
        }

        private void transitionTo(CoordinatorState newState) {
            if (!newState.canTransitionFrom(this.state)) {
                throw new IllegalStateException("Cannot transition from " + String.valueOf((Object)this.state) + " to " + String.valueOf((Object)newState));
            }
            CoordinatorState oldState = this.state;
            CoordinatorRuntime.this.log.debug("Transition from {} to {}.", (Object)this.state, (Object)newState);
            switch (newState.ordinal()) {
                case 1: {
                    this.state = CoordinatorState.LOADING;
                    SnapshotRegistry snapshotRegistry = new SnapshotRegistry(this.logContext);
                    this.coordinator = new SnapshottableCoordinator(this.logContext, snapshotRegistry, CoordinatorRuntime.this.coordinatorShardBuilderSupplier.get().withLogContext(this.logContext).withSnapshotRegistry(snapshotRegistry).withTime(CoordinatorRuntime.this.time).withTimer(this.timer).withExecutor(this.executor).withCoordinatorMetrics(CoordinatorRuntime.this.coordinatorMetrics).withTopicPartition(this.tp).build(), this.tp);
                    this.load();
                    break;
                }
                case 2: {
                    this.state = CoordinatorState.ACTIVE;
                    this.highWatermarklistener = new HighWatermarkListener();
                    CoordinatorRuntime.this.partitionWriter.registerListener(this.tp, this.highWatermarklistener);
                    this.coordinator.onLoaded(CoordinatorRuntime.this.metadataImage);
                    break;
                }
                case 4: {
                    this.state = CoordinatorState.FAILED;
                    this.unload();
                    break;
                }
                case 3: {
                    this.state = CoordinatorState.CLOSED;
                    this.unload();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Transitioning to " + String.valueOf((Object)newState) + " is not supported.");
                }
            }
            CoordinatorRuntime.this.runtimeMetrics.recordPartitionStateChange(oldState, this.state);
        }

        private void load() {
            if (this.state != CoordinatorState.LOADING) {
                throw new IllegalStateException("Coordinator must be in loading state");
            }
            CoordinatorRuntime.this.loader.load(this.tp, this.coordinator).whenComplete((summary, exception) -> CoordinatorRuntime.this.scheduleInternalOperation("CompleteLoad(tp=" + String.valueOf(this.tp) + ", epoch=" + this.epoch + ")", this.tp, () -> {
                CoordinatorContext context = CoordinatorRuntime.this.coordinators.get(this.tp);
                if (context != null) {
                    if (context.state != CoordinatorState.LOADING) {
                        CoordinatorRuntime.this.log.info("Ignored load completion from {} because context is in {} state.", (Object)context.tp, (Object)context.state);
                        return;
                    }
                    try {
                        if (exception != null) {
                            throw exception;
                        }
                        context.transitionTo(CoordinatorState.ACTIVE);
                        if (summary != null) {
                            CoordinatorRuntime.this.runtimeMetrics.recordPartitionLoadSensor(summary.startTimeMs(), summary.endTimeMs());
                            CoordinatorRuntime.this.log.info("Finished loading of metadata from {} with epoch {} in {}ms where {}ms was spent in the scheduler. Loaded {} records which total to {} bytes.", new Object[]{this.tp, this.epoch, summary.endTimeMs() - summary.startTimeMs(), summary.schedulerQueueTimeMs(), summary.numRecords(), summary.numBytes()});
                        }
                    }
                    catch (Throwable ex) {
                        CoordinatorRuntime.this.log.error("Failed to load metadata from {} with epoch {} due to {}.", new Object[]{this.tp, this.epoch, ex.toString()});
                        context.transitionTo(CoordinatorState.FAILED);
                    }
                } else {
                    CoordinatorRuntime.this.log.debug("Failed to complete the loading of metadata for {} in epoch {} since the coordinator does not exist.", (Object)this.tp, (Object)this.epoch);
                }
            }));
        }

        private void unload() {
            if (this.highWatermarklistener != null) {
                CoordinatorRuntime.this.partitionWriter.deregisterListener(this.tp, this.highWatermarklistener);
                this.highWatermarklistener = null;
            }
            this.timer.cancelAll();
            this.executor.cancelAll();
            this.deferredEventQueue.failAll((Throwable)Errors.NOT_COORDINATOR.exception());
            this.failCurrentBatch((Throwable)Errors.NOT_COORDINATOR.exception());
            if (this.coordinator != null) {
                try {
                    this.coordinator.onUnloaded();
                }
                catch (Throwable ex) {
                    CoordinatorRuntime.this.log.error("Failed to unload coordinator for {} due to {}.", new Object[]{this.tp, ex.getMessage(), ex});
                }
            }
            this.coordinator = null;
        }

        private void freeCurrentBatch() {
            this.currentBatch.lingerTimeoutTask.ifPresent(TimerTask::cancel);
            int maxBatchSize = CoordinatorRuntime.this.partitionWriter.config(this.tp).maxMessageSize();
            if (this.currentBatch.builder.buffer().capacity() <= maxBatchSize) {
                this.bufferSupplier.release(this.currentBatch.builder.buffer());
            } else if (this.currentBatch.buffer.capacity() <= maxBatchSize) {
                this.bufferSupplier.release(this.currentBatch.buffer);
            }
            this.currentBatch = null;
        }

        private void flushCurrentBatch() {
            if (this.currentBatch != null) {
                try {
                    if (this.currentBatch.builder.numRecords() == 0) {
                        CoordinatorRuntime.this.log.debug("Tried to flush an empty batch for {}.", (Object)this.tp);
                        this.failCurrentBatch(new IllegalStateException("Record batch was empty"));
                        return;
                    }
                    long flushStartMs = CoordinatorRuntime.this.time.milliseconds();
                    long offset = CoordinatorRuntime.this.partitionWriter.append(this.tp, this.currentBatch.verificationGuard, this.currentBatch.builder.build());
                    CoordinatorRuntime.this.runtimeMetrics.recordFlushTime(CoordinatorRuntime.this.time.milliseconds() - flushStartMs);
                    this.coordinator.updateLastWrittenOffset(offset);
                    if (offset != this.currentBatch.nextOffset) {
                        CoordinatorRuntime.this.log.error("The state machine of the coordinator {} is out of sync with the underlying log. The last written offset returned is {} while the coordinator expected {}. The coordinator will be reloaded in order to re-synchronize the state machine.", new Object[]{this.tp, offset, this.currentBatch.nextOffset});
                        this.transitionTo(CoordinatorState.FAILED);
                        this.transitionTo(CoordinatorState.LOADING);
                        throw Errors.NOT_COORDINATOR.exception();
                    }
                    this.deferredEventQueue.add(offset, (DeferredEvent)this.currentBatch.deferredEvents);
                    this.freeCurrentBatch();
                }
                catch (Throwable t) {
                    CoordinatorRuntime.this.log.error("Writing records to {} failed due to: {}.", (Object)this.tp, (Object)t.getMessage());
                    this.failCurrentBatch(t);
                    throw t;
                }
            }
        }

        private void maybeFlushCurrentBatch(long currentTimeMs) {
            if (this.currentBatch != null && (this.currentBatch.builder.isTransactional() || this.currentBatch.appendTimeMs - currentTimeMs >= (long)CoordinatorRuntime.this.appendLingerMs)) {
                this.flushCurrentBatch();
            }
        }

        private void failCurrentBatch(Throwable t) {
            if (this.currentBatch != null) {
                this.coordinator.revertLastWrittenOffset(this.currentBatch.baseOffset);
                this.currentBatch.deferredEvents.complete(t);
                this.freeCurrentBatch();
            }
        }

        private void maybeAllocateNewBatch(long producerId, short producerEpoch, VerificationGuard verificationGuard, long currentTimeMs) {
            if (this.currentBatch == null) {
                LogConfig logConfig = CoordinatorRuntime.this.partitionWriter.config(this.tp);
                int maxBatchSize = logConfig.maxMessageSize();
                long prevLastWrittenOffset = this.coordinator.lastWrittenOffset();
                ByteBuffer buffer = this.bufferSupplier.get(Math.min(524288, maxBatchSize));
                MemoryRecordsBuilder builder = new MemoryRecordsBuilder(buffer, 2, CoordinatorRuntime.this.compression, TimestampType.CREATE_TIME, 0L, currentTimeMs, producerId, producerEpoch, 0, producerId != -1L, false, -1, maxBatchSize);
                Optional<TimerTask> lingerTimeoutTask = Optional.empty();
                if (CoordinatorRuntime.this.appendLingerMs > 0) {
                    lingerTimeoutTask = Optional.of(new TimerTask(CoordinatorRuntime.this.appendLingerMs){

                        public void run() {
                            CoordinatorRuntime.this.enqueueFirst(new CoordinatorInternalEvent("FlushBatch", CoordinatorContext.this.tp, () -> {
                                if (this.isCancelled()) {
                                    return;
                                }
                                CoordinatorRuntime.this.withActiveContextOrThrow(CoordinatorContext.this.tp, CoordinatorContext::flushCurrentBatch);
                            }));
                        }
                    });
                    CoordinatorRuntime.this.timer.add(lingerTimeoutTask.get());
                }
                this.currentBatch = new CoordinatorBatch(CoordinatorRuntime.this.log, prevLastWrittenOffset, currentTimeMs, maxBatchSize, verificationGuard, buffer, builder, lingerTimeoutTask);
            }
        }

        private void append(long producerId, short producerEpoch, VerificationGuard verificationGuard, List<U> records, boolean replay, boolean isAtomic, DeferredEvent event) {
            if (this.state != CoordinatorState.ACTIVE) {
                throw new IllegalStateException("Coordinator must be active to append records");
            }
            if (records.isEmpty()) {
                if (this.currentBatch != null && this.currentBatch.builder.numRecords() > 0) {
                    this.currentBatch.deferredEvents.add(event);
                } else if (this.coordinator.lastCommittedOffset() < this.coordinator.lastWrittenOffset()) {
                    this.deferredEventQueue.add(this.coordinator.lastWrittenOffset(), (DeferredEvent)DeferredEventCollection.of(CoordinatorRuntime.this.log, event));
                } else {
                    event.complete(null);
                }
            } else {
                long currentTimeMs = CoordinatorRuntime.this.time.milliseconds();
                if (producerId != -1L) {
                    isAtomic = true;
                    this.flushCurrentBatch();
                }
                this.maybeAllocateNewBatch(producerId, producerEpoch, verificationGuard, currentTimeMs);
                ArrayList<SimpleRecord> recordsToAppend = new ArrayList<SimpleRecord>(records.size());
                for (Object record : records) {
                    recordsToAppend.add(new SimpleRecord(currentTimeMs, CoordinatorRuntime.this.serializer.serializeKey(record), CoordinatorRuntime.this.serializer.serializeValue(record)));
                }
                if (isAtomic) {
                    int estimatedSize = AbstractRecords.estimateSizeInBytes((byte)this.currentBatch.builder.magic(), (CompressionType)CoordinatorRuntime.this.compression.type(), recordsToAppend);
                    if (estimatedSize > this.currentBatch.builder.maxAllowedBytes()) {
                        throw new RecordTooLargeException("Message batch size is " + estimatedSize + " bytes in append to partition " + String.valueOf(this.tp) + " which exceeds the maximum configured size of " + this.currentBatch.maxBatchSize + ".");
                    }
                    if (!this.currentBatch.builder.hasRoomFor(estimatedSize)) {
                        this.flushCurrentBatch();
                        this.maybeAllocateNewBatch(producerId, producerEpoch, verificationGuard, currentTimeMs);
                    }
                }
                for (int i = 0; i < records.size(); ++i) {
                    boolean hasRoomFor;
                    Object recordToReplay = records.get(i);
                    SimpleRecord recordToAppend = (SimpleRecord)recordsToAppend.get(i);
                    if (!isAtomic && !(hasRoomFor = this.currentBatch.builder.hasRoomFor(recordToAppend.timestamp(), recordToAppend.key(), recordToAppend.value(), recordToAppend.headers()))) {
                        this.flushCurrentBatch();
                        this.maybeAllocateNewBatch(producerId, producerEpoch, verificationGuard, currentTimeMs);
                    }
                    try {
                        if (replay) {
                            this.coordinator.replay(this.currentBatch.nextOffset, producerId, producerEpoch, recordToReplay);
                        }
                        this.currentBatch.builder.append(recordToAppend);
                        ++this.currentBatch.nextOffset;
                        continue;
                    }
                    catch (Throwable t) {
                        CoordinatorRuntime.this.log.error("Replaying record {} to {} failed due to: {}.", new Object[]{recordToReplay, this.tp, t.getMessage()});
                        this.currentBatch.deferredEvents.add(event);
                        this.failCurrentBatch(t);
                        return;
                    }
                }
                this.currentBatch.deferredEvents.add(event);
                this.maybeFlushCurrentBatch(currentTimeMs);
            }
        }

        private void completeTransaction(long producerId, short producerEpoch, int coordinatorEpoch, TransactionResult result, DeferredEvent event) {
            if (this.state != CoordinatorState.ACTIVE) {
                throw new IllegalStateException("Coordinator must be active to complete a transaction");
            }
            this.flushCurrentBatch();
            long prevLastWrittenOffset = this.coordinator.lastWrittenOffset();
            try {
                this.coordinator.replayEndTransactionMarker(producerId, producerEpoch, result);
                long flushStartMs = CoordinatorRuntime.this.time.milliseconds();
                long offset = CoordinatorRuntime.this.partitionWriter.append(this.tp, VerificationGuard.SENTINEL, MemoryRecords.withEndTransactionMarker((long)CoordinatorRuntime.this.time.milliseconds(), (long)producerId, (short)producerEpoch, (EndTransactionMarker)new EndTransactionMarker(result == TransactionResult.COMMIT ? ControlRecordType.COMMIT : ControlRecordType.ABORT, coordinatorEpoch)));
                CoordinatorRuntime.this.runtimeMetrics.recordFlushTime(CoordinatorRuntime.this.time.milliseconds() - flushStartMs);
                this.coordinator.updateLastWrittenOffset(offset);
                this.deferredEventQueue.add(offset, (DeferredEvent)DeferredEventCollection.of(CoordinatorRuntime.this.log, event));
            }
            catch (Throwable t) {
                this.coordinator.revertLastWrittenOffset(prevLastWrittenOffset);
                event.complete(t);
            }
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    public static enum CoordinatorState {
        INITIAL{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return false;
            }
        }
        ,
        LOADING{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == INITIAL || state == FAILED;
            }
        }
        ,
        ACTIVE{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == ACTIVE || state == LOADING;
            }
        }
        ,
        CLOSED{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return true;
            }
        }
        ,
        FAILED{

            @Override
            boolean canTransitionFrom(CoordinatorState state) {
                return state == LOADING || state == ACTIVE;
            }
        };


        abstract boolean canTransitionFrom(CoordinatorState var1);
    }

    class CoordinatorWriteEvent<T>
    implements CoordinatorEvent,
    DeferredEvent {
        public static final long NOT_QUEUED = -1L;
        final TopicPartition tp;
        final String name;
        final String transactionalId;
        final long producerId;
        final short producerEpoch;
        final VerificationGuard verificationGuard;
        final CoordinatorWriteOperation<S, T, U> op;
        final CompletableFuture<T> future;
        final Duration writeTimeout;
        private OperationTimeout operationTimeout = null;
        CoordinatorResult<T, U> result;
        private final long createdTimeMs;
        private long deferredEventQueuedTimestamp;

        CoordinatorWriteEvent(String name, TopicPartition tp, Duration writeTimeout, CoordinatorWriteOperation<S, T, U> op) {
            this(name, tp, null, -1L, -1, VerificationGuard.SENTINEL, writeTimeout, op);
        }

        CoordinatorWriteEvent(String name, TopicPartition tp, String transactionalId, long producerId, short producerEpoch, VerificationGuard verificationGuard, Duration writeTimeout, CoordinatorWriteOperation<S, T, U> op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.transactionalId = transactionalId;
            this.producerId = producerId;
            this.producerEpoch = producerEpoch;
            this.verificationGuard = verificationGuard;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
            this.writeTimeout = writeTimeout;
            this.deferredEventQueuedTimestamp = -1L;
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    this.result = this.op.generateRecordsAndResult(context.coordinator.coordinator());
                    context.append(this.producerId, this.producerEpoch, this.verificationGuard, this.result.records(), this.result.replayRecords(), this.result.isAtomic(), this);
                    if (!this.future.isDone()) {
                        this.operationTimeout = new OperationTimeout(this.tp, this, this.writeTimeout.toMillis());
                        CoordinatorRuntime.this.timer.add((TimerTask)this.operationTimeout);
                        this.deferredEventQueuedTimestamp = CoordinatorRuntime.this.time.milliseconds();
                    }
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            CompletableFuture<Void> appendFuture;
            if (this.future.isDone()) {
                return;
            }
            long purgatoryTimeMs = CoordinatorRuntime.this.time.milliseconds() - this.deferredEventQueuedTimestamp;
            CompletableFuture<Void> completableFuture = appendFuture = this.result != null ? this.result.appendFuture() : null;
            if (exception == null) {
                if (appendFuture != null) {
                    this.result.appendFuture().complete(null);
                }
                this.future.complete(this.result.response());
            } else {
                if (appendFuture != null) {
                    this.result.appendFuture().completeExceptionally(exception);
                }
                this.future.completeExceptionally(exception);
            }
            if (this.operationTimeout != null) {
                this.operationTimeout.cancel();
                this.operationTimeout = null;
            }
            if (this.deferredEventQueuedTimestamp != -1L) {
                CoordinatorRuntime.this.runtimeMetrics.recordEventPurgatoryTime(purgatoryTimeMs);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorWriteEvent(name=" + this.name + ")";
        }
    }

    public static interface CoordinatorWriteOperation<S, T, U> {
        public CoordinatorResult<T, U> generateRecordsAndResult(S var1) throws KafkaException;
    }

    class CoordinatorCompleteTransactionEvent
    implements CoordinatorEvent,
    DeferredEvent {
        final TopicPartition tp;
        final String name;
        final long producerId;
        final short producerEpoch;
        final int coordinatorEpoch;
        final TransactionResult result;
        final Duration writeTimeout;
        private OperationTimeout operationTimeout = null;
        final CompletableFuture<Void> future;
        private final long createdTimeMs;
        private long deferredEventQueuedTimestamp;

        CoordinatorCompleteTransactionEvent(String name, TopicPartition tp, long producerId, short producerEpoch, int coordinatorEpoch, TransactionResult result, Duration writeTimeout) {
            this.name = name;
            this.tp = tp;
            this.producerId = producerId;
            this.producerEpoch = producerEpoch;
            this.coordinatorEpoch = coordinatorEpoch;
            this.result = result;
            this.writeTimeout = writeTimeout;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
            this.deferredEventQueuedTimestamp = -1L;
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    context.completeTransaction(this.producerId, this.producerEpoch, this.coordinatorEpoch, this.result, this);
                    if (!this.future.isDone()) {
                        this.operationTimeout = new OperationTimeout(this.tp, this, this.writeTimeout.toMillis());
                        CoordinatorRuntime.this.timer.add((TimerTask)this.operationTimeout);
                        this.deferredEventQueuedTimestamp = CoordinatorRuntime.this.time.milliseconds();
                    }
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (this.future.isDone()) {
                return;
            }
            long purgatoryTimeMs = CoordinatorRuntime.this.time.milliseconds() - this.deferredEventQueuedTimestamp;
            if (exception == null) {
                this.future.complete(null);
            } else {
                this.future.completeExceptionally(exception);
            }
            if (this.operationTimeout != null) {
                this.operationTimeout.cancel();
                this.operationTimeout = null;
            }
            if (this.deferredEventQueuedTimestamp != -1L) {
                CoordinatorRuntime.this.runtimeMetrics.recordEventPurgatoryTime(purgatoryTimeMs);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorCompleteTransactionEvent(name=" + this.name + ")";
        }
    }

    class CoordinatorReadEvent<T>
    implements CoordinatorEvent {
        final TopicPartition tp;
        final String name;
        final CoordinatorReadOperation<S, T> op;
        final CompletableFuture<T> future;
        T response;
        private final long createdTimeMs;

        CoordinatorReadEvent(String name, TopicPartition tp, CoordinatorReadOperation<S, T> op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.future = new CompletableFuture();
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                CoordinatorRuntime.this.withActiveContextOrThrow(this.tp, context -> {
                    this.response = this.op.generateResponse(context.coordinator.coordinator(), context.coordinator.lastCommittedOffset());
                    this.complete(null);
                });
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (exception == null) {
                this.future.complete(this.response);
            } else {
                this.future.completeExceptionally(exception);
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "CoordinatorReadEvent(name=" + this.name + ")";
        }
    }

    public static interface CoordinatorReadOperation<S, T> {
        public T generateResponse(S var1, long var2) throws KafkaException;
    }

    class CoordinatorInternalEvent
    implements CoordinatorEvent {
        final TopicPartition tp;
        final String name;
        final Runnable op;
        private final long createdTimeMs;

        CoordinatorInternalEvent(String name, TopicPartition tp, Runnable op) {
            this.tp = tp;
            this.name = name;
            this.op = op;
            this.createdTimeMs = CoordinatorRuntime.this.time.milliseconds();
        }

        @Override
        public TopicPartition key() {
            return this.tp;
        }

        @Override
        public void run() {
            try {
                this.op.run();
            }
            catch (Throwable t) {
                this.complete(t);
            }
        }

        @Override
        public void complete(Throwable exception) {
            if (exception != null) {
                CoordinatorRuntime.this.log.error("Execution of {} failed due to {}.", new Object[]{this.name, exception.getMessage(), exception});
            }
        }

        @Override
        public long createdTimeMs() {
            return this.createdTimeMs;
        }

        public String toString() {
            return "InternalEvent(name=" + this.name + ")";
        }
    }

    class HighWatermarkListener
    implements PartitionWriter.Listener {
        static final long NO_OFFSET = -1L;
        private final AtomicLong lastHighWatermark = new AtomicLong(-1L);

        HighWatermarkListener() {
        }

        public long lastHighWatermark() {
            return this.lastHighWatermark.get();
        }

        @Override
        public void onHighWatermarkUpdated(TopicPartition tp, long offset) {
            CoordinatorRuntime.this.log.debug("High watermark of {} incremented to {}.", (Object)tp, (Object)offset);
            if (this.lastHighWatermark.getAndSet(offset) == -1L) {
                CoordinatorRuntime.this.enqueueFirst(new CoordinatorInternalEvent("HighWatermarkUpdate", tp, () -> {
                    long newHighWatermark = this.lastHighWatermark.getAndSet(-1L);
                    CoordinatorContext context = CoordinatorRuntime.this.coordinators.get(tp);
                    if (context != null) {
                        context.lock.lock();
                        try {
                            if (context.state == CoordinatorState.ACTIVE) {
                                CoordinatorRuntime.this.log.debug("Updating high watermark of {} to {}.", (Object)tp, (Object)newHighWatermark);
                                context.coordinator.updateLastCommittedOffset(newHighWatermark);
                                context.deferredEventQueue.completeUpTo(newHighWatermark);
                                CoordinatorRuntime.this.coordinatorMetrics.onUpdateLastCommittedOffset(tp, newHighWatermark);
                            }
                            CoordinatorRuntime.this.log.debug("Ignored high watermark updated for {} to {} because the coordinator is not active.", (Object)tp, (Object)newHighWatermark);
                        }
                        finally {
                            context.lock.unlock();
                        }
                    } else {
                        CoordinatorRuntime.this.log.debug("Ignored high watermark updated for {} to {} because the coordinator does not exist.", (Object)tp, (Object)newHighWatermark);
                    }
                }));
            }
        }
    }

    static class DeferredEventCollection
    implements DeferredEvent {
        private final Logger log;
        private final List<DeferredEvent> events = new ArrayList<DeferredEvent>();

        public DeferredEventCollection(Logger log) {
            this.log = log;
        }

        public void complete(Throwable t) {
            for (DeferredEvent event : this.events) {
                try {
                    event.complete(t);
                }
                catch (Throwable e) {
                    this.log.error("Completion of event {} failed due to {}.", new Object[]{event, e.getMessage(), e});
                }
            }
        }

        public boolean add(DeferredEvent event) {
            return this.events.add(event);
        }

        public int size() {
            return this.events.size();
        }

        public String toString() {
            return "DeferredEventCollection(events=" + String.valueOf(this.events) + ")";
        }

        public static DeferredEventCollection of(Logger log, DeferredEvent ... deferredEvents) {
            DeferredEventCollection collection = new DeferredEventCollection(log);
            for (DeferredEvent deferredEvent : deferredEvents) {
                collection.add(deferredEvent);
            }
            return collection;
        }
    }

    class OperationTimeout
    extends TimerTask {
        private final TopicPartition tp;
        private final DeferredEvent event;

        public OperationTimeout(TopicPartition tp, DeferredEvent event, long delayMs) {
            super(delayMs);
            this.event = event;
            this.tp = tp;
        }

        public void run() {
            String name = this.event.toString();
            CoordinatorRuntime.this.scheduleInternalOperation("OperationTimeout(name=" + name + ", tp=" + String.valueOf(this.tp) + ")", this.tp, () -> this.event.complete((Throwable)new TimeoutException(name + " timed out after " + this.delayMs + "ms")));
        }
    }

    private static class CoordinatorBatch {
        final long baseOffset;
        final long appendTimeMs;
        final int maxBatchSize;
        final VerificationGuard verificationGuard;
        final ByteBuffer buffer;
        final MemoryRecordsBuilder builder;
        final Optional<TimerTask> lingerTimeoutTask;
        final DeferredEventCollection deferredEvents;
        long nextOffset;

        CoordinatorBatch(Logger log, long baseOffset, long appendTimeMs, int maxBatchSize, VerificationGuard verificationGuard, ByteBuffer buffer, MemoryRecordsBuilder builder, Optional<TimerTask> lingerTimeoutTask) {
            this.baseOffset = baseOffset;
            this.nextOffset = baseOffset;
            this.appendTimeMs = appendTimeMs;
            this.maxBatchSize = maxBatchSize;
            this.verificationGuard = verificationGuard;
            this.buffer = buffer;
            this.builder = builder;
            this.lingerTimeoutTask = lingerTimeoutTask;
            this.deferredEvents = new DeferredEventCollection(log);
        }
    }

    class EventBasedCoordinatorTimer
    implements CoordinatorTimer<Void, U> {
        final Logger log;
        final TopicPartition tp;
        final Map<String, TimerTask> tasks = new HashMap<String, TimerTask>();

        EventBasedCoordinatorTimer(TopicPartition tp, LogContext logContext) {
            this.tp = tp;
            this.log = logContext.logger(EventBasedCoordinatorTimer.class);
        }

        @Override
        public void schedule(String key, long delay, TimeUnit unit, boolean retry, CoordinatorTimer.TimeoutOperation<Void, U> operation) {
            this.schedule(key, delay, unit, retry, 500L, operation);
        }

        @Override
        public void schedule(final String key, long delay, TimeUnit unit, final boolean retry, final long retryBackoff, final CoordinatorTimer.TimeoutOperation<Void, U> operation) {
            TimerTask task = new TimerTask(unit.toMillis(delay)){

                public void run() {
                    String eventName = "Timeout(tp=" + String.valueOf(EventBasedCoordinatorTimer.this.tp) + ", key=" + key + ")";
                    CoordinatorWriteEvent event = new CoordinatorWriteEvent(eventName, EventBasedCoordinatorTimer.this.tp, CoordinatorRuntime.this.defaultWriteTimeout, coordinator -> {
                        EventBasedCoordinatorTimer.this.log.debug("Executing write event {} for timer {}.", (Object)eventName, (Object)key);
                        if (!EventBasedCoordinatorTimer.this.tasks.remove(key, (Object)this)) {
                            throw new RejectedExecutionException("Timer " + key + " was overridden or cancelled");
                        }
                        return operation.generateRecords();
                    });
                    event.future.exceptionally(ex -> {
                        if (ex instanceof RejectedExecutionException) {
                            EventBasedCoordinatorTimer.this.log.debug("The write event {} for the timer {} was not executed because it was cancelled or overridden.", (Object)event.name, (Object)key);
                            return null;
                        }
                        if (ex instanceof NotCoordinatorException || ex instanceof CoordinatorLoadInProgressException) {
                            EventBasedCoordinatorTimer.this.log.debug("The write event {} for the timer {} failed due to {}. Ignoring it because the coordinator is not active.", new Object[]{event.name, key, ex.getMessage()});
                            return null;
                        }
                        if (retry) {
                            EventBasedCoordinatorTimer.this.log.info("The write event {} for the timer {} failed due to {}. Rescheduling it. ", new Object[]{event.name, key, ex.getMessage()});
                            EventBasedCoordinatorTimer.this.schedule(key, retryBackoff, TimeUnit.MILLISECONDS, true, retryBackoff, operation);
                        } else {
                            EventBasedCoordinatorTimer.this.log.error("The write event {} for the timer {} failed due to {}. Ignoring it. ", new Object[]{event.name, key, ex.getMessage()});
                        }
                        return null;
                    });
                    EventBasedCoordinatorTimer.this.log.debug("Scheduling write event {} for timer {}.", (Object)event.name, (Object)key);
                    try {
                        CoordinatorRuntime.this.enqueueLast(event);
                    }
                    catch (NotCoordinatorException ex2) {
                        EventBasedCoordinatorTimer.this.log.info("Failed to enqueue write event {} for timer {} because the runtime is closed. Ignoring it.", (Object)event.name, (Object)key);
                    }
                }
            };
            this.log.debug("Registering timer {} with delay of {}ms.", (Object)key, (Object)unit.toMillis(delay));
            TimerTask prevTask = this.tasks.put(key, task);
            if (prevTask != null) {
                prevTask.cancel();
            }
            CoordinatorRuntime.this.timer.add(task);
        }

        @Override
        public void scheduleIfAbsent(String key, long delay, TimeUnit unit, boolean retry, CoordinatorTimer.TimeoutOperation<Void, U> operation) {
            if (!this.tasks.containsKey(key)) {
                this.schedule(key, delay, unit, retry, 500L, operation);
            }
        }

        @Override
        public void cancel(String key) {
            TimerTask prevTask = this.tasks.remove(key);
            if (prevTask != null) {
                prevTask.cancel();
            }
        }

        public void cancelAll() {
            Iterator<Map.Entry<String, TimerTask>> iterator = this.tasks.entrySet().iterator();
            while (iterator.hasNext()) {
                iterator.next().getValue().cancel();
                iterator.remove();
            }
        }

        public int size() {
            return this.tasks.size();
        }
    }

    public static class Builder<S extends CoordinatorShard<U>, U> {
        private String logPrefix;
        private LogContext logContext;
        private CoordinatorEventProcessor eventProcessor;
        private PartitionWriter partitionWriter;
        private CoordinatorLoader<U> loader;
        private CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier;
        private Time time = Time.SYSTEM;
        private Timer timer;
        private Duration defaultWriteTimeout;
        private CoordinatorRuntimeMetrics runtimeMetrics;
        private CoordinatorMetrics coordinatorMetrics;
        private Serializer<U> serializer;
        private Compression compression;
        private int appendLingerMs;
        private ExecutorService executorService;

        public Builder<S, U> withLogPrefix(String logPrefix) {
            this.logPrefix = logPrefix;
            return this;
        }

        public Builder<S, U> withLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        public Builder<S, U> withEventProcessor(CoordinatorEventProcessor eventProcessor) {
            this.eventProcessor = eventProcessor;
            return this;
        }

        public Builder<S, U> withPartitionWriter(PartitionWriter partitionWriter) {
            this.partitionWriter = partitionWriter;
            return this;
        }

        public Builder<S, U> withLoader(CoordinatorLoader<U> loader) {
            this.loader = loader;
            return this;
        }

        public Builder<S, U> withCoordinatorShardBuilderSupplier(CoordinatorShardBuilderSupplier<S, U> coordinatorShardBuilderSupplier) {
            this.coordinatorShardBuilderSupplier = coordinatorShardBuilderSupplier;
            return this;
        }

        public Builder<S, U> withTime(Time time) {
            this.time = time;
            return this;
        }

        public Builder<S, U> withTimer(Timer timer) {
            this.timer = timer;
            return this;
        }

        public Builder<S, U> withDefaultWriteTimeOut(Duration defaultWriteTimeout) {
            this.defaultWriteTimeout = defaultWriteTimeout;
            return this;
        }

        public Builder<S, U> withCoordinatorRuntimeMetrics(CoordinatorRuntimeMetrics runtimeMetrics) {
            this.runtimeMetrics = runtimeMetrics;
            return this;
        }

        public Builder<S, U> withCoordinatorMetrics(CoordinatorMetrics coordinatorMetrics) {
            this.coordinatorMetrics = coordinatorMetrics;
            return this;
        }

        public Builder<S, U> withSerializer(Serializer<U> serializer) {
            this.serializer = serializer;
            return this;
        }

        public Builder<S, U> withCompression(Compression compression) {
            this.compression = compression;
            return this;
        }

        public Builder<S, U> withAppendLingerMs(int appendLingerMs) {
            this.appendLingerMs = appendLingerMs;
            return this;
        }

        public Builder<S, U> withExecutorService(ExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        public CoordinatorRuntime<S, U> build() {
            if (this.logPrefix == null) {
                this.logPrefix = "";
            }
            if (this.logContext == null) {
                this.logContext = new LogContext(this.logPrefix);
            }
            if (this.eventProcessor == null) {
                throw new IllegalArgumentException("Event processor must be set.");
            }
            if (this.partitionWriter == null) {
                throw new IllegalArgumentException("Partition write must be set.");
            }
            if (this.loader == null) {
                throw new IllegalArgumentException("Loader must be set.");
            }
            if (this.coordinatorShardBuilderSupplier == null) {
                throw new IllegalArgumentException("State machine supplier must be set.");
            }
            if (this.time == null) {
                throw new IllegalArgumentException("Time must be set.");
            }
            if (this.timer == null) {
                throw new IllegalArgumentException("Timer must be set.");
            }
            if (this.runtimeMetrics == null) {
                throw new IllegalArgumentException("CoordinatorRuntimeMetrics must be set.");
            }
            if (this.coordinatorMetrics == null) {
                throw new IllegalArgumentException("CoordinatorMetrics must be set.");
            }
            if (this.serializer == null) {
                throw new IllegalArgumentException("Serializer must be set.");
            }
            if (this.compression == null) {
                this.compression = Compression.NONE;
            }
            if (this.appendLingerMs < 0) {
                throw new IllegalArgumentException("AppendLinger must be >= 0");
            }
            if (this.executorService == null) {
                throw new IllegalArgumentException("ExecutorService must be set.");
            }
            return new CoordinatorRuntime<S, U>(this.logPrefix, this.logContext, this.eventProcessor, this.partitionWriter, this.loader, this.coordinatorShardBuilderSupplier, this.time, this.timer, this.defaultWriteTimeout, this.runtimeMetrics, this.coordinatorMetrics, this.serializer, this.compression, this.appendLingerMs, this.executorService);
        }
    }
}

