/*
 * Decompiled with CFR 0.152.
 */
package org.shredzone.acme4j.toolbox;

import jakarta.annotation.Nullable;
import jakarta.annotation.ParametersAreNonnullByDefault;
import java.net.URL;
import java.security.KeyPair;
import java.security.PublicKey;
import java.util.Map;
import javax.crypto.SecretKey;
import org.jose4j.jwk.EllipticCurveJsonWebKey;
import org.jose4j.jwk.JsonWebKey;
import org.jose4j.jwk.PublicJsonWebKey;
import org.jose4j.jwk.RsaJsonWebKey;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.lang.JoseException;
import org.shredzone.acme4j.exception.AcmeException;
import org.shredzone.acme4j.exception.AcmeProtocolException;
import org.shredzone.acme4j.toolbox.JSONBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ParametersAreNonnullByDefault
public final class JoseUtils {
    private static final Logger LOG = LoggerFactory.getLogger(JoseUtils.class);

    private JoseUtils() {
    }

    public static JSONBuilder createJoseRequest(URL url, KeyPair keypair, @Nullable JSONBuilder payload, @Nullable String nonce, @Nullable String kid) {
        try {
            PublicJsonWebKey jwk = PublicJsonWebKey.Factory.newPublicJwk(keypair.getPublic());
            JsonWebSignature jws = new JsonWebSignature();
            jws.getHeaders().setObjectHeaderValue("url", url);
            if (kid != null) {
                jws.getHeaders().setObjectHeaderValue("kid", kid);
            } else {
                jws.getHeaders().setJwkHeaderValue("jwk", jwk);
            }
            if (nonce != null) {
                jws.getHeaders().setObjectHeaderValue("nonce", nonce);
            }
            jws.setPayload(payload != null ? payload.toString() : "");
            jws.setAlgorithmHeaderValue(JoseUtils.keyAlgorithm(jwk));
            jws.setKey(keypair.getPrivate());
            jws.sign();
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} {}", (Object)(payload != null ? "POST" : "POST-as-GET"), (Object)url);
                if (payload != null) {
                    LOG.debug("  Payload: {}", (Object)payload);
                }
                LOG.debug("  JWS Header: {}", (Object)jws.getHeaders().getFullHeaderAsJsonString());
            }
            JSONBuilder jb = new JSONBuilder();
            jb.put("protected", jws.getHeaders().getEncodedHeader());
            jb.put("payload", jws.getEncodedPayload());
            jb.put("signature", jws.getEncodedSignature());
            return jb;
        }
        catch (JoseException ex) {
            throw new AcmeProtocolException("Failed to sign a JSON request", ex);
        }
    }

    public static Map<String, Object> createExternalAccountBinding(String kid, PublicKey accountKey, SecretKey macKey, URL resource) throws AcmeException {
        try {
            PublicJsonWebKey keyJwk = PublicJsonWebKey.Factory.newPublicJwk(accountKey);
            JsonWebSignature innerJws = new JsonWebSignature();
            innerJws.setPayload(keyJwk.toJson());
            innerJws.getHeaders().setObjectHeaderValue("url", resource);
            innerJws.getHeaders().setObjectHeaderValue("kid", kid);
            innerJws.setAlgorithmHeaderValue(JoseUtils.macKeyAlgorithm(macKey));
            innerJws.setKey(macKey);
            innerJws.setDoKeyValidation(false);
            innerJws.sign();
            JSONBuilder outerClaim = new JSONBuilder();
            outerClaim.put("protected", innerJws.getHeaders().getEncodedHeader());
            outerClaim.put("signature", innerJws.getEncodedSignature());
            outerClaim.put("payload", innerJws.getEncodedPayload());
            return outerClaim.toMap();
        }
        catch (JoseException ex) {
            throw new AcmeException("Could not create external account binding", ex);
        }
    }

    public static Map<String, Object> publicKeyToJWK(PublicKey key) {
        try {
            return PublicJsonWebKey.Factory.newPublicJwk(key).toParams(JsonWebKey.OutputControlLevel.PUBLIC_ONLY);
        }
        catch (JoseException ex) {
            throw new IllegalArgumentException("Bad public key", ex);
        }
    }

    public static byte[] thumbprint(PublicKey key) {
        try {
            PublicJsonWebKey jwk = PublicJsonWebKey.Factory.newPublicJwk(key);
            return jwk.calculateThumbprint("SHA-256");
        }
        catch (JoseException ex) {
            throw new IllegalArgumentException("Bad public key", ex);
        }
    }

    public static String keyAlgorithm(JsonWebKey jwk) {
        if (jwk instanceof EllipticCurveJsonWebKey) {
            EllipticCurveJsonWebKey ecjwk = (EllipticCurveJsonWebKey)jwk;
            switch (ecjwk.getCurveName()) {
                case "P-256": {
                    return "ES256";
                }
                case "P-384": {
                    return "ES384";
                }
                case "P-521": {
                    return "ES512";
                }
            }
            throw new IllegalArgumentException("Unknown EC name " + ecjwk.getCurveName());
        }
        if (jwk instanceof RsaJsonWebKey) {
            return "RS256";
        }
        throw new IllegalArgumentException("Unknown algorithm " + jwk.getAlgorithm());
    }

    public static String macKeyAlgorithm(SecretKey macKey) {
        if (!"HMAC".equals(macKey.getAlgorithm())) {
            throw new IllegalArgumentException("Bad algorithm: " + macKey.getAlgorithm());
        }
        int size = macKey.getEncoded().length * 8;
        switch (size) {
            case 256: {
                return "HS256";
            }
            case 384: {
                return "HS384";
            }
            case 512: {
                return "HS512";
            }
        }
        throw new IllegalArgumentException("Bad key size: " + size);
    }
}

