/*******************************************************************************
 * 		Licensed Materials - Property of IBM
 *   		xxxx-xxx
 * 		© Copyright IBM Corporation 2014. All rights reserved.
 * 		US Government Users Restricted Rights - Use, duplication or disclosure
 * 		restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office.
 *******************************************************************************/
package samples;

import java.util.Date;

import com.ibm.debug.pdt.codecoverage.core.results.CCAbstractException;
import com.ibm.debug.pdt.codecoverage.core.results.CCMessageUtilities;
import com.ibm.debug.pdt.codecoverage.core.results.CCResultException;
import com.ibm.debug.pdt.codecoverage.core.results.CCResultsFactory;
import com.ibm.debug.pdt.codecoverage.core.results.ICCFile;
import com.ibm.debug.pdt.codecoverage.core.results.ICCFlowPoint;
import com.ibm.debug.pdt.codecoverage.core.results.ICCPart;
import com.ibm.debug.pdt.codecoverage.core.results.ICCResult;
import com.ibm.debug.pdt.codecoverage.core.results.ICCTestcase;

/**
 * Illustrates how obtain code coverage data from one or more files that are passed in
 * as arguments to main()
 * If using a command line then enter the following<br>
 * {@code java -cp ".;".;./plugins/com.ibm.debug.pdt.codecoverage.core.results_<version>/lib/ccapi.jar" .jar" samples.TestResultDriver "path to cc
 * data file or directory"}
 *
 */
@SuppressWarnings("nls")
public class TestResultDriver {

	public static void main(final String[] args) {

		long startTime = System.currentTimeMillis();

		CCResultsFactory.setExceptionLogging();
		ICCResult results = null;
		try {
			results = CCResultsFactory.getInstance().createResult(args);
		} catch (CCResultException e) {
			for(CCAbstractException ie : e.getExceptions())
				System.out.println(ie.getMessage());
			results = e.getResult();
			e.printStackTrace();
		}


		System.out.println("imported in " + (System.currentTimeMillis() - startTime) + " ms" );

		System.out.println("name: " + results.getName());
		System.out.println("elapsed: " + results.getInfo().getElapsedTime());

		System.out.println("number of testcases " + results.getTestcases().length);
		System.out.println("number of messages " + results.getMessages().length);
		for (String msg : results.getMessages()) {
			System.out.println(CCMessageUtilities.getMessage(msg));
		}
		System.out.println("number of files = " + results.getNumFiles());

		int totalLines = 0;
		int totalHit = 0;

		for (ICCFile file : results.getFiles()) {
			System.out.println("getName() : " + file.getName());
			System.out.println("getBaseName() : " + file.getBaseName());
			System.out.println("getBaseFileName() : " + file.getBaseFileName());
			System.out.println("getQualifiedName() : " + file.getQualifiedName());
			for (String msg : file.getMessages()) {
				System.out.println(CCMessageUtilities.getMessage(msg));
			}

			int lines = file.getLines(false).length;
			totalLines += lines;
			int hitLines = file.getLines(true).length;
			totalHit += hitLines;

			if (hitLines > 0) {
				try {
					System.out.println("File : " + file.getBaseName() + " executable lines : " + lines + " hit lines : " + hitLines + " hit " + file.getPercentCoverage() + "%");
				} catch (ArithmeticException e) {
					e.printStackTrace();
				}
			}


			if (file.getFile() == null) {
				System.out.println("no real file check for stream");
				try {
					if (file.getStream() != null)
						System.out.println(file.getStream().toString());
				} catch (Exception e) {
					e.printStackTrace();
				}
			} else
				System.out.println(file.getFile().getPath());


			for(ICCTestcase testcase : file.getTestcases()) {
				System.out.println("    testcase: " + testcase.getName());
				long testStartTime = testcase.getStartTime();
				Date startDate = new Date(testStartTime);
				System.out.println("    start time: " + startDate);
				System.out.println("    elapsed time: " + testcase.getElapsedTime());
			}
			System.out.print("    Lines[" + file.getLines(false).length + "] " );
			for(Integer line : file.getLines(false)) {
				System.out.print(line + ",");
			}
			System.out.print("LAST");
			System.out.println();
			System.out.print("    hit lines[" + file.getLines(true).length + "] " );
			for(Integer line : file.getLines(true)) {
				System.out.print(line + ",");
			}
			System.out.print("LAST");
			System.out.println();
			ICCTestcase[] testcases = file.getTestcases();
			for (ICCTestcase iccTestcase : testcases) {
				System.out.println("    Testcase name: " + iccTestcase.getName());
				System.out.print("        hit lines[" + file.getHitLines(iccTestcase).length + "] ");
				for(Integer line : file.getHitLines(iccTestcase))
					System.out.print(line + ",");
				System.out.print("LAST");
				System.out.println();
			}

			System.out.println();
			ICCTestcase[] tcs = file.getTestcases(27);
			if (tcs.length > 0)
				System.out.println("hit line 27" );

			ICCFlowPoint[] functions = file.getFlowPoints();
			for (ICCFlowPoint function : functions) {
				System.out.println("    Function: " + function.getName() + " @ " + function.getLine());
				System.out.println("        First line / Last line / #executable / #hit " + function.getLine() + " / " + function.getLastLine() + " / " + function.getNumExecutableLines() + " / " + function.getNumHitLines());
				System.out.println("has nested functions " + function.isNestedFlowPoints());
				if (function.isNestedFlowPoints()) {
					for(ICCFlowPoint flowpoint : function.getFlowPoints())
						System.out.println(" nested flow point ->" + flowpoint.getName());
				}
			}
		}

		System.out.println(" Total lines : " + totalLines);
		System.out.println(" Total hit lines : " + totalHit);
		System.out.println(" % hit : " + results.getPercentCoverage());
		System.out.println("total elapsed " + (System.currentTimeMillis() - startTime) + " ms" );


		// now check for parts
		System.out.println("Write out parts");
		for (ICCPart part : results.getParts()) {
			System.out.println(part.getName() + " coverage :" + part.getPercentCoverage());
			System.out.println("language:" + part.getLanguage());

			for (ICCFile file :part.getFiles()) {
				System.out.println("    > " + file.getName());
			}
		}

	}
}
