/*                               REXX                                */
/*********************************************************************/
/*                                                                   */
/* Licensed Materials - Property of IBM                              */
/* Copyright IBM Corporation 2011, 2013. All Rights Reserved.        */
/*                                                                   */
/* Note to U.S. Government Users Restricted Rights:                  */
/* Use, duplication or disclosure restricted by GSA ADP Schedule     */
/* Contract with IBM Corp.                                           */
/*                                                                   */
/*********************************************************************/
/*                                                                   */
/*                                                                   */
/* NAME := BLZXMIT                                                   */
/*                                                                   */
/* DESCRIPTIVE NAME := EWM XMIT a load module to a sequential dsn    */
/*                                                                   */
/* FUNCTION := This module creates an IEBCOPY XMIT data set from a   */
/*             load module member. It also removes the timestamp to  */
/*             stop hasmmap errors.                                  */
/*                                                                   */
/* OUTPUT := None                                                    */
/*                                                                   */
/* Change History                                                    */
/*                                                                   */
/* Who   When     What                                               */
/* ----- -------- -------------------------------------------------- */
/* LD    12/08/13 Initial version                                    */
/*                                                                   */
/*********************************************************************/
/*                                                                   */
/* XMIT file format is documented in                                 */
/* TSO/E customization (SA22-7783-xx)                                */
/* chapter 'Customizing TRANSMIT and RECEIVE',                       */
/* section 'Text Units and Text Unit Pointer Lists' and              */
/* section 'Format of Transmitted Data'                              */
/*                                                                   */
/* An XMIT file (80-byte lines) consists of control & data records.  */
/* A record is stored in 1 or more segments (max 255 bytes).         */
/* Within each segment, information is stored as a text unit.        */
/*                                                                   */
/* ----+----1----+----2----+----3----+----4----+----5----+----6----+ */
/*   INMR01             MVS040      ONNO      MVS040      ONNO       */
/* 5ECDDDFF0400005110000DEEFFF110000DDDD100000DEEFFF100000DDDD120000 */
/* 6095490102010100101064520400201046556010106452040020104655604010E */
/* |||     | | | |                                                   */
/* |||     | | | +-> text unit - data                                */
/* |||     | | +---> text unit - length                              */
/* |||     | +-----> text unit - # length/data pairs                 */
/* |||     +-------> text unit - type                                */
/* ||+-> record type                                                 */
/* |+--> Segment & record flags                                      */
/* +---> segment length                                              */
/*                                                                   */
/*===================================================================*/
/* user variables....................................................*/
  parse arg da' 'logfile
  parse arg dsn '(' member ')'

  Call OUTTRAP stem.
  XmitDsn = dsn"."member".XMIT"
  cmd = "xmit "MVSVAR(SYSNAME)"."USERID()" da ('"da"')
         outda('"XmitDsn"') NOLOG"
  address tso cmd
  If rc <> 0 Then
  Do
    Call lineout logfile,"ERROR",1
    Do xmit = 1 to stem.0
      Call lineout logfile, stem.xmit
    End
  End
  Else
  Do
    /*-1-*/                    /* Timestamp replacement value        */
    XmitDate='##############'

    /*-2-*/                                      /* temporary DD name */
    LongTime = TIME('L')
    Parse var LongTime hh':'mm':'ss'.'msecs
    DDname='$$'msecs

    /* system code. */

    /* drop Lines. */
    parse value ReadXmit(DDname XmitDsn) with rc error1 error2
    If rc <> 0 Then
    Do
      Call lineout logfile,"ERROR",1
      Call lineout logfile,error1
      Call lineout logfile,error2
    End
    Else
    Do
      Call UpdateSegment(XmitDate)

      parse value WriteSegment(DDname) with rc error1
      If rc <> 0 Then
      Do
        Call lineout logfile,"ERROR",1
        Call lineout logfile,error1
      End
      Else
      Do
        /* all is OK! */
        Call lineout logfile,"OK",1
      End
    End
  End

  XX = OUTTRAP('OFF')
  call lineout  /* close the logfile */

return rc

/*===================================================================*/
ReadXmit: PROCEDURE EXPOSE Lines. /* read first lines of file        */
/* -input-                                                           */
/* DDname: DD name to use for file allocation                        */
/* Dsn   : DSN to allocate                                           */
/*                                                                   */
/* -output-                                                          */
/* (Lines., contains first lines of XMIT file)                       */
/* (DDname, is now an allocated file)                                */
/*===================================================================*/
  Arg DDname Dsn

  QuoteDsn = "'"Dsn"'"
  error1=""
  error2=""

  cRC = BPXWDYN("alloc dd("DDname") da("QuoteDsn") shr")
  If cRC > 0 Then
  Do
    error1=">> ERROR: ALLOCATE ended with rc" cRC
    error2=">>       " Dsn
    Say error1
    Say error2
    Rc = BPXWDYN("free FILE("DDname")")
    return 12 error1 error2
  End    /* allocate error */

  Address MVS "EXECIO 4 DISKR" DDname "(STEM Lines. FINIS"
  cRC = rc                    /* 4 80-byte lines cover the (max) 263 */
  If cRC > 0 Then             /* bytes needed for 2 segment headers  */
  Do
    error1=">> ERROR: EXECIO (read) ended with rc" cRC
    error2=">>       " Dsn
    Say error1
    Say error2
    Rc = BPXWDYN("free FILE("DDname")")
    return 12 error1 error2
  End    /* read error */

Return 0 error1 error2  /* ReadXmit */


/*===================================================================*/
UpdateSegment: PROCEDURE EXPOSE Lines. /* update 1st XMIT Segment   */
/* -input-                                                           */
/* XmitDate : new default date                                       */
/* (Lines., included with expose)                                    */
/*                                                                   */
/* -output-                                                          */
/* (Lines., selected data fields have been updated)                  */
/*===================================================================*/
  Arg XmitDate

  INMDATE = '1024'x                              /* TIMESTAMP        */

  Stream = ''                   /* string all 80-byte lines together */
  Do T = 1 to Lines.0
    Stream = Stream || Lines.T
  End    /* loop */

  sFlags = substr(Stream,2,1)                       /* segment flags */
  If bitand(sFlags,'40'x) <> '40'x Then    /* 40x -> last segment    */
  Do
    Say ">> WARNING: header record spans multiple segments,"
    Say "            but only the first one is updated"
  End    /* multi segment */

  sLength=c2d(left(Stream,1))                     /* Segment length */
  Ptr=9                                        /* skip record header */

  Do while Ptr < sLength      /* read all text units in this segment */
    parse value GetTextUnit(Ptr) with Ptr uType uLength uPair .

    If uPair > 1 Then                      /* multi-data text unit ? */
    Do
      Do T = 1 to uPair                     /* skip to next TextUnit */
        Ptr=Ptr + 2 + c2d(substr(Stream,Ptr,2))
      End    /* loop */
    End    /* uPair > 1 */
    Else
    Do                               /* only 1 or 0 length/data pair */
      Select                         /* act upon selected unit types */
        When uType = INMDATE Then
          Stream=overlay(XmitDate,Stream,Ptr,uLength)
        Otherwise
          nop
      End    /* select */

      Ptr=Ptr+uLength                        /* go to next text unit */
    End    /* uPair = 0/1 */
  End    /* Do while */

  Do T = 1 to Lines.0           /* put updates back in 80-byte lines */
    Lines.T = substr(Stream,(T-1)*80+1,80)
  End    /* loop */
Return    /* UpdateSegment */


/*===================================================================*/
GetTextUnit: PROCEDURE EXPOSE Stream /* get info on next text unit   */
/* -input-                                                           */
/* Ptr: starting point of text unit                                  */
/* (Stream, included with expose)                                    */
/*                                                                   */
/* -output-                                                          */
/* Ptr    : starting point of data inside this unit If uPair <= 1,   */
/*          otherwise starting point of length/data pairs            */
/* uType  : type of data inside this unit                            */
/* uLength: length of actual data inside this unit If uPair <= 1,    */
/*          otherwise length of first length/data pair               */
/* uPair  : # of length/data pairs                                   */
/*                                                                   */
/*       ONNO  DDNAME  $$XMIT$$                                      */
/* 000000DDDD00CCDCDC0055EDCE55                                      */
/* 02030465560644514508BB7493BB                                      */
/* | |(1)|  (2)     (3)                                              */
/* | | | |                                                           */
/* | | | +-> position of Ptr If uPair <= 1                           */
/* | | +---> uLength (= 4), position of Ptr If uPair > 1             */
/* | +-----> uPair  (= 3)                                            */
/* +-------> uType  (= 0002x)                                        */
/*===================================================================*/
  Arg Ptr

  uType = substr(Stream,Ptr,2)                               /* type */
  If uType = '0000'x Then    /* INMR02 records start with a dummy    */
    uPair = 0                /* 0000 0001. This is NOT documented !! */
  Else
    uPair = c2d(substr(Stream,Ptr+2,2))    /* # of length/data pairs */
  Select
    When uPair = 0 Then
    Do                                          /* empty text unit ? */
      uLength = 0
      Ptr = Ptr + 4
    End    /* empty text unit */
    When uPair = 1 Then
    Do                                        /* 1 length/data pairs */
      uLength = c2d(substr(Stream,Ptr+4,2))
      Ptr = Ptr + 6
    End    /* 1 length/data pair */
    Otherwise                          /* multiple length/data pairs */
    Do                                        /* 1 length/data pairs */
      uLength=c2d(substr(Stream,Ptr+4,2)) /* shows length of 1st one */
      Ptr = Ptr + 4
    End
  /* leave further processing to caller */
  End    /* select */
Return Ptr uType uLength uPair    /* GetTextUnit */


/*===================================================================*/
WriteSegment: PROCEDURE EXPOSE Lines. /* save updates                */
/* -input-                                                           */
/* DDname: DD name by which the XMIT file is allocated               */
/* (Lines., included with expose)                                    */
/*                                                                   */
/* -output-                                                          */
/* (DDname, file is updated and freed)                               */
/*===================================================================*/
  Arg DDname

  error1=""
  error2=""

  Do T = 1 to Lines.0
                                          /* position on correct line */
    Address MVS "EXECIO 1 DISKRU" DDname "(SKIP"
    cRC = rc
    If cRC > 0 Then
    Do
      error1=">> ERROR: EXECIO (positioning on line" T") ended with rc" cRC
      Say error1
      Address MVS "EXECIO 0 DISKRU" DDname "(FINIS"
      Rc = BPXWDYN("free FILE("DDname")")
      Return 12 error1
    End    /* positioning error */

    push Lines.T                               /* write 80-byte line */
    Address MVS "EXECIO 1 DISKW" DDname
    cRC = rc
    If cRC > 0 Then
    Do
      error1=">> ERROR: EXECIO (writing line" T") ended with rc" cRC
      Say error1
      Address MVS "EXECIO 0 DISKRU" DDname "(FINIS"
      Rc = BPXWDYN("free FILE("DDname")")
      Return 12 error1
    End    /* write error */
  End    /* loop */

  Address MVS "EXECIO 0 DISKRU" DDname "(FINIS"    /* close file */
  cRC = rc
  If cRC > 0 Then
  Do
    error1=">> ERROR: EXECIO (close) ended with rc" cRC
    Say error1
    Rc = BPXWDYN("free FILE("DDname")")
    Return 12 error1
  End    /* close error */

  Address TSO "FREE FILE("DDname")"
Return 0 error1  /* WriteSegment */