/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2005, 2010. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
dojo.provide("com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution"); //$NON-NLS-1$

dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.ProbabilityDistribution"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.ProbabilityDistributionType"); //$NON-NLS-1$

(function() {
	
var ProbabilityDistribution			= com.ibm.team.tempo.shared.client.internal.simulation.ProbabilityDistribution;
var ProbabilityDistributionType		= com.ibm.team.tempo.shared.client.internal.simulation.ProbabilityDistributionType;

dojo.declare("com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution", ProbabilityDistribution, { //$NON-NLS-1$

	name: ProbabilityDistributionType.TRIANGULAR,

	constructor: function(lb, m, ub) {
		this.lowerbound = lb;
		this.middle = m;
		this.upperbound = ub;
	},

	// ---- api ------------------------------------------------------------------------------------------------------------

	randomCost: function(rand) {
		var cost = 0.0;
		if (this.upperbound == this.lowerbound) return this.upperbound;
		var delta = this.middle-this.lowerbound;
		var a1 = delta/(this.upperbound-this.lowerbound);
		if (rand <= a1) cost = Math.sqrt(rand/a1)*delta+this.lowerbound;
		else cost = this.upperbound - Math.sqrt((1.0-rand)/(1.0-a1))*(this.upperbound-this.middle);
		return cost;
	},
	
	scale: function(r) {	
		// Gives a trianular distribution of a variable scaled by r
		// r can be negative
		// if r=-1.0 then we get the negative of the distribution
		var m;
		var lb;
		var ub;
		if (r >=0.0) {
			m = this.middle*r;
			lb = this.lowerbound*r;
			ub = this.upperbound*r;
		} else { // r is negative, need to interchange this.lowerbound and this.upperbound
			m = this.middle*r;
			lb = this.upperbound*r;
			ub = this.lowerbound*r;
		}
		var result = new com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution(lb, m, ub);
		return result;
	},
	
	convolute: function(tdvector){
		//gives a triangular distribution that is an approximation of the sum of the random variables
		//represented by tdvector
		//This would be a convolution integral of the distributions
		//The gammas of the sum of the variables is the sum of the gammas
		
		var gammaOut = new Array(4);
		gammaOut[0]=1.0; // not used
		gammaOut[1]=0.0;
		gammaOut[2]=0.0;
		gammaOut[3]=0.0;
		for (var i = 0; i < tdvector.length; i++) {
			var gamma = tdvector[i].gammas();
			for (var j = 1; j < gamma.length; j++)
				gammaOut[j] += gamma[j];
		}		
		return this.fromGammas(gammaOut);
	},
	
	fromGammas: function(gamma) {
		//Returns a triangular distribution that matches the mean (gamma[1]), 
		//variance(gamma[2]) and skewness(gamma[3]) provided
		//gamma[2], and gamma[3] are the second and third moments around the mean (gamma[1])
		var x = 0.0; 
		var y = 0.0;
		// Condition for the distribution to be representable by a triangular distribution: discriminant >= 0.0
		var discriminant = 8.0*gamma[2]*gamma[2]*gamma[2] - 25.0*gamma[3]*gamma[3];
		if (discriminant < 0.0) {
			//System.out.println("Cannot represent the distribution as a triangular distribution"); //$NON-NLS-1$
			//System.out.println("variance=  "+gamma[2]+"  skewness=  "+gamma[3]+"   discriminant=  "+discriminant); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			//System.out.println("We will try the closest (right) triangular distribution"); //$NON-NLS-1$
			//So we will find the closest right triangle to fit the data (maximum skewness for a triangular distribution)
			y = 3.0*Math.sqrt(gamma[2]/2.0)*Math.signum(gamma[3]);
			x = Math.abs(y);
		}   
		else {	
			//Start with the upper and lower limit for y
			var ymax = 3.0*Math.sqrt(gamma[2]/2.0);
			var ymin = -ymax;
			
			//Ensure that the upper and lower limit satisfy the conditions
			var fy = this.F2(ymax, gamma);
			
			//assert (fy <= 0.0): "The cubic should be non-positive at ymax"; //$NON-NLS-1$
			fy = this.F2(ymin, gamma);
			
			//assert (fy >=0.0): "The cubic should be non-negative at ymin"; //$NON-NLS-1$
			// search for root using binary search.
			
			for (var i = 0; i < 100; i++) {
				y = (ymax+ymin)/2.0;
				fy = this.F2(y, gamma);
				if (fy <= 0.0) ymax = y;
				if (fy >= 0.0) ymin = y;
				if (ymax == ymin) break;
			}
			//assert (Math.signum(y) == Math.signum(gamma[3])): "y should have the same sign as skewness"; //$NON-NLS-1$
			x = Math.sqrt(6*gamma[2]-y*y/3.0);
		}
		var m = gamma[1] - 2.0*y/3.0;
		var u = m+x+y;
		var l = m-x+y;
		//assert ((u >= m) && (m >= l)):  "The triangular distribution should be valid"; //$NON-NLS-1$
		return new com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution(l, m, u);
	},
	
	/**
	 * @param y
	 * @param gamma
	 * @return The cubic to be satisfied by y in fromMoments
	 */
	F2: function(y, gamma) {
		return (4.0*y*y*y - 54.0*gamma[2]*y+135.0*gamma[3]);
	},
	
	gammas: function(){
		//gives the mean and the moments around the mean
		var gamma = new Array(4);
		gamma[0]=1.0;
		gamma[1]=(this.lowerbound+this.middle+this.upperbound)/3.0; //mean
		gamma[2]=(this.upperbound*this.upperbound+this.middle*this.middle+this.lowerbound*this.lowerbound-this.upperbound*this.middle-this.middle*this.lowerbound-this.lowerbound*this.upperbound)/18.0; //variance
		gamma[3]=((this.upperbound-2.0*this.middle+this.lowerbound)*(this.upperbound+this.middle-2.0*this.lowerbound)*(2.0*this.upperbound-this.middle-this.lowerbound))/270.0; //skewness
		return gamma;
	},
	
	fromMoments: function(p) {	
		//Returns a triangular distribution that matches the first three moments given
		//First Normalize the moments, in case they are not.  p[0] should be  = 1.0;
		var p1 = p[1]/p[0];
		var p2 = p[2]/p[0];
		var p3 = p[3]/p[0];
		var gamma1 = p1;  //mean
		var gamma2 = p2 -p1*p1; //variance
		var gamma3 = p3 - 3.0*p1*p2 + 2.0*p1*p1*p1;  // skewness
		var x = 0.0; 
		var y = 0.0;
		// Condition for the distribution to be representable by a triangular distribution: discriminant >= 0.0
		var discriminant = 8.0*gamma2*gamma2*gamma2 - 25.0*gamma3*gamma3;
		if (discriminant < 0.0) {
			//System.out.println("Cannot represent the distribution as a triangular distribution"); //$NON-NLS-1$
			//System.out.println("variance=  "+gamma2+"  skewness=  "+gamma3+"   discriminant=  "+discriminant); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			//System.out.println("We will try the closest (right) triangular distribution"); //$NON-NLS-1$
			//So we will find the closest right triangle to fit the data (maximum skewness for a triangular distribution)
			y = 3.0*Math.sqrt(gamma2/2.0)*Math.signum(gamma3);
			x = Math.abs(y);
		}   
		else {	
			//Start with the upper and lower limit for y
			var ymax = 3.0*Math.sqrt(gamma2/2.0);
			var ymin = -ymax;
			//Ensure that the upper and lower limit satisfy the conditions
			var fy = this.F3(ymax, gamma2, gamma3);
			//assert (fy <= 0.0): "The cubic should be non-positive at ymax"; //$NON-NLS-1$
			
			fy = this.F3(ymin, gamma2, gamma3);
			//assert (fy >=0.0): "The cubic should be non-negative at ymin"; //$NON-NLS-1$
			
			// search for root using binary search.
			for (var i = 0; i < 100; i++) {
				y = (ymax+ymin)/2.0;
				fy = this.F3(y, gamma2, gamma3);
				if (fy <= 0.0) ymax = y;
				if (fy >= 0.0) ymin = y;
				if (ymax == ymin) break;
			}
			//assert (Math.signum(y) == Math.signum(gamma3)): "y should have the same sign as skewness"; //$NON-NLS-1$
			x = Math.sqrt(6*gamma2-y*y/3.0);
		}
		var m = gamma1 - 2.0*y/3.0;
		var u = m+x+y;
		var l = m-x+y;
		//assert ((u >= m) && (m >= l)):  "The triangular distribution should be valid"; //$NON-NLS-1$
		return new com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution(l, m, u);
	},
	
	/**
	 * @param y
	 * @param gamma2
	 * @param gamma3
	 * @return The cubic to be satisfied by y in fromMoments
	 */
	F3: function(y, gamma2, gamma3) {
		return (4.0*y*y*y - 54.0*gamma2*y+135.0*gamma3);
	},

	// ---- implementation -------------------------------------------------------------------------------------------------

	__sentinel: null // terminates this class definition
});

com.ibm.team.tempo.shared.client.internal.simulation.TriangularDistribution.NAME = ProbabilityDistributionType.TRIANGULAR;

})();

