/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2005, 2010. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
dojo.provide("com.ibm.team.tempo.shared.client.internal.simulation.MonteCarloGraphNode"); //$NON-NLS-1$

dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.CostDistribution"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.CostDistributionRpt"); //$NON-NLS-1$

(function() {
	
var CostDistribution				= com.ibm.team.tempo.shared.client.internal.simulation.CostDistribution;
var CostDistributionRpt				= com.ibm.team.tempo.shared.client.internal.simulation.CostDistributionRpt;

dojo.declare("com.ibm.team.tempo.shared.client.internal.simulation.MonteCarloGraphNode", null, { //$NON-NLS-1$

	constructor: function(graphNode, n, isWithNodeDistributions) {
		this.modeTbl = {};
	
		this.modeCost = -1.0;
		this.totCost = 0.0;
		this.totSlack = 0.0;
		this.targetCost = 0.0;
		this.totSlackPlusTotCost = 0.0;
		this.effectiveAbilityToCompleteCount = 0;
		this.ableToCompleteCount = 0;
		this.onCritPathCount = 0;
		this.onTargetCount = 0;
	
		this.isWithNodeDistributions = false;
		this.costDistribution = new CostDistribution(n);
		this.targetCost = graphNode.getTargetCost();
		this.id = graphNode.getId();
		
		this.estimate = graphNode.getEstimate();
		this.n = n;
		this.isWithNodeDistributions = isWithNodeDistributions;
		this.cumulativeCosts = [];
		this.avgCumulativeCost = -1.0;
	},

	// ---- api ------------------------------------------------------------------------------------------------------------

	getId: function() {
		return this.id;
	},	
	
	getOnTargetCount: function() {
		return this.onTargetCount;
	},
	
	getAbleToCompleteCount: function() {
		return this.ableToCompleteCount;
	},
	
	getOnCritPathCount: function() {
		return this.onCritPathCount;
	},

	getModeCost: function() {
		if (this.modeCost < 0.0) {
			this.computeMode();
		}
		return this.modeCost;
	},

	getAvgCost: function() {
		return this.totCost/this.n;
	},
	
	getAvgSlack: function() {
		return this.totSlack/this.n;
	},

	getAvgSlackPlusTotCost: function() {
		return this.totSlackPlusTotCost/this.n;
	},
	
	updateSlack: function(slack) {
		this.totSlack = this.totSlack + slack;
	},

	updateSlackPlusTotCost: function(slackPlusTotCost) {
		this.totSlackPlusTotCost = this.totSlackPlusTotCost + slackPlusTotCost;
	},

	getProbabilityOfOnTime: function() {
		return (this.getOnTargetCount()-0.0) / (this.n-0.0);
	},

	getProbabilityToComplete: function() {
		return (this.getAbleToCompleteCount()-0.0) / (this.n-0.0);
	},

	/**
	 * @return The probability that this node will complete within the targeted 
	 * duration since the beginning (i.e. node cost + previous node cost + slack time).
	 */
	getEffectiveProbabilityToComplete: function() {
		return (this.effectiveAbilityToCompleteCount-0.0) / (this.n-0.0);
	},
	
	getCostDistribution: function() {
		return this.costDistribution;
	},

	getTargetCost: function() {
		return this.targetCost;
	},
	
	sortNumber: function (a, b) {
		return a-b;
	},
	
	getCumulativeCostAt: function (percentile) {
		if (this.cumulativeCosts.length == 0) {
			return -1.0;
		}
		this.cumulativeCosts.sort(this.sortNumber);
		var index = Math.max((percentile*this.cumulativeCosts.length - 1).toFixed(), 0);
		return this.cumulativeCosts[index];
	},
	
	getAvgCumulativeCost: function () {
		if (this.avgCumulativeCost < 0.0) {
			var avg = 0.0;
			for (var i = 0; i < this.cumulativeCosts.length; i++) {
				avg += this.cumulativeCosts[i];
			}
			this.avgCumulativeCost = avg/this.cumulativeCosts.length;
		}
		return this.avgCumulativeCost;
	},
	
	showResults: function() {
	    var L = this.estimate.lowerBound;
		var N = this.estimate.middle;
		var H = this.estimate.upperBound;
		
		var buf = ""; //$NON-NLS-1$
		buf += " NodeId=" + this.getId(); //$NON-NLS-1$
		buf += " L/N/H=" + L + "/" + N + "/" + H + " target=" + this.targetCost; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		buf += " mode=" + this.getModeCost(); //$NON-NLS-1$
		buf += " critPathCt=" + this.getOnCritPathCount(); //$NON-NLS-1$
		buf += " targetCt=" + this.getOnTargetCount(); //$NON-NLS-1$
		buf += " avgSlack=" + this.getAvgSlack(); //$NON-NLS-1$
		//buf += " probOnTime=" + this.getProbabilityOfOnTime(); //$NON-NLS-1$
				
		buf += " completion time bounded at=" + this.getCumulativeCostAt(0.95); //$NON-NLS-1$
		buf += " avg completion time=" + this.getAvgCumulativeCost(); //$NON-NLS-1$
		
		buf += " probToComplete=" + this.getProbabilityToComplete(); //$NON-NLS-1$
		buf += " effProbToComplete=" + this.getEffectiveProbabilityToComplete(); //$NON-NLS-1$
		
//		if (this.isWithNodeDistributions) this.showDistribution();
		return buf;
	},
	
	updateAbleToCompleteCount: function(overallCost, overallTargetCost) {		
		if (overallCost <= overallTargetCost) {
			this.ableToCompleteCount++;
		}
	},

	/**
	 * 
	 * @param effectiveCost The total cost for this node which may include any slack times.
	 * @param effectiveTargetCost
	 */
	updateEffectiveAbilityToCompleteCount: function(cumulativeCost, effectiveTargetCost) {		
		if (cumulativeCost <= effectiveTargetCost) {
			this.effectiveAbilityToCompleteCount++ ;
		}
		this.cumulativeCosts.push(cumulativeCost);
	},

	updateCostFields: function(cost) {
		if (this.isWithNodeDistributions) this.costDistribution.update(cost);
		this.updateModeTbl(cost);	
		if (cost <= this.targetCost) this.onTargetCount++;	
		this.totCost = this.totCost + cost;
	},
	
	incrementOnCritPathCount: function() {
		this.onCritPathCount++;
	},
		
	resetCounters: function() {
		this.effectiveAbilityToCompleteCount = 0;
		this.ableToCompleteCount = 0;
		this.onCritPathCount = 0;
		this.onTargetCount = 0;
		this.modeTbl= {};
		this.modeCost = Number.NaN;
		this.cumulativeCosts = [];
		this.avgCumulativeCost = -1.0;
	},
	
	updateModeTbl: function(cost) {
//		String costStr = String.valueOf(Math.round(cost));
		var costStr = Math.round(cost) + ""; //$NON-NLS-1$
		if (this.modeTbl[costStr] != null) {
			var count = Math.round(Number(this.modeTbl[costStr]));
			this.modeTbl[costStr]= count++;
		} else {
			this.modeTbl[costStr]= 1;
		}
	},
	
	/**
	 * Compute which cost appears most frequently.
	 */
	computeMode: function() {
		var prevCount = 0;
		for (var key in this.modeTbl) {
			var count = this.modeTbl[key];
			if (count > prevCount)
				this.modeCost = Number(key);
			prevCount = count;
		}
	},
		
	showDistribution: function() {
		var buf = ""; //$NON-NLS-1$
		try {
			var rpt = new CostDistributionRpt(this.costDistribution);
			rpt.run();
		} catch (e) {
//			System.out.println(e.getMessage());
//			console.log(e); // Suppressed 4/19/2009
		}	
		
		return buf;
	},
	
	// Added by PMM
	setComputedProbabilityOfCompletionPercentage: function(computedProbabilityOfCompletionPercentage) {
		this.computedProbabilityOfCompletionPercentage = computedProbabilityOfCompletionPercentage;
	},
	
	getComputedProbabilityOfCompletionPercentage: function() {
		return this.computedProbabilityOfCompletionPercentage;
	},
	
	setComputedAverageSlack: function(computedAverageSlack) {
		this.computedAverageSlack = computedAverageSlack;
	},
	
	getComputedAverageSlack: function() {
		return this.averageSlack;
	},
	
	setComputedAverageTime: function(computedAverageTime) {
		this.computedAverageTime = computedAverageTime;
	},
	
	getComputedAverageTime: function() {
		return this.computedAverageTime;
	},


	// ---- implementation -------------------------------------------------------------------------------------------------

	__sentinel: null // terminates this class definition
});
	
})();
