/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2005, 2010. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
dojo.provide("com.ibm.team.tempo.shared.client.internal.simulation.GraphOptimization"); //$NON-NLS-1$

dojo.require("com.ibm.team.apt.shared.client.plancheck.PlanCheck"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.SimpleSchedule"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.ScheduleByOwner"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.CostDistribution"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.OptimizationResults"); //$NON-NLS-1$
dojo.require("com.ibm.team.tempo.shared.client.internal.simulation.MonteCarloGraphNode"); //$NON-NLS-1$

dojo.require("dojo.string"); //$NON-NLS-1$
dojo.require("dojo.i18n"); //$NON-NLS-1$
dojo.requireLocalization("com.ibm.team.tempo.shared.client.internal", "SimulationMessages"); //$NON-NLS-1$ //$NON-NLS-2$

(function() {
	
var Run								= com.ibm.team.apt.shared.client.plancheck.PlanCheck.Run;
var SimpleSchedule					= com.ibm.team.tempo.shared.client.internal.simulation.SimpleSchedule;
var ScheduleByOwner					= com.ibm.team.tempo.shared.client.internal.simulation.ScheduleByOwner;
var CostDistribution				= com.ibm.team.tempo.shared.client.internal.simulation.CostDistribution;
var OptimizationResults				= com.ibm.team.tempo.shared.client.internal.simulation.OptimizationResults;
var MonteCarloGraphNode				= com.ibm.team.tempo.shared.client.internal.simulation.MonteCarloGraphNode;

var bind= dojo.string.substitute;
var Messages= dojo.i18n.getLocalization("com.ibm.team.tempo.shared.client.internal", "SimulationMessages"); //$NON-NLS-1$ //$NON-NLS-2$

dojo.declare("com.ibm.team.tempo.shared.client.internal.simulation.GraphOptimization", null, { //$NON-NLS-1$

	constructor: function(schedule, overallTargetCost) {
		this.ownerGraphNodeTbl = null;
		this.graph = null;
		this.scheduleResult = null;
		this.overallTargetCost = 0.0;
		this.overallCost = 0.0;	
		
		if (schedule instanceof SimpleSchedule) {
			this.scheduleResult = schedule.run();
			SimpleSchedule.updateDependencies(this.scheduleResult);
			this.graph = schedule.getGraph();
			this.overallTargetCost = overallTargetCost;
		}

		if (schedule instanceof ScheduleByOwner) {
			this.graph = schedule.getGraph();
			this.scheduleResult = schedule.getScheduleResult();
			this.overallTargetCost = overallTargetCost;
			this.ownerGraphNodeTbl = this.graph.getOwnerGraphNodeTbl();
		}
	},

	// ---- api ------------------------------------------------------------------------------------------------------------

	/**
	 * Compute the total N times, summarizing the results.
	 * @param graph
	 * @param n Number of times cost is computed.
	 * @return OptimizationResults consisting of N, mean, average of the squares and 
	 *         the average exponential cost.  
	 * @throws Exception
	 */
	simulate: function(n, monitor) {
		return this.__doSimulate(n, false, true, monitor);
	},

	getOverallCost: function() {
		return this.overallCost;
	},

	getLookupCount: function(graph) {
		return graph.getLookupCount();
	},

	// ---- implementation -------------------------------------------------------------------------------------------------
	
	__doSimulate: function(n, showNodes, showNodeDistributions, monitor) {
		monitor.beginTask(Messages['plan_monitorMonteCarlo'], n); //$NON-NLS-1$
		var monteCarloNodeTbl= this.__populateMonteCarloNodes(this.graph.getGraphNodeTbl(), n, showNodeDistributions);
		var costDistribution= new CostDistribution(n);
		var cost = 0.0;

		var lookupCount = 0;
		
		var i= 0;

		var runner= new Run(dojo.hitch(this, function() {
			for (; i < n && !monitor.shouldYield(); i++) {
				if (monitor.isCanceled())
					return Run.DONE;

				this.overallCost = 0.;
				this.graph.resetNodes();
				this.graph.run();
	
				if (this.scheduleResult == null) {
					this.__updateGraphMetrics(this.graph, monteCarloNodeTbl);
				} else { 
					if (this.ownerGraphNodeTbl == null) {
						this.__updateSimpleScheduledMetrics(this.scheduleResult, monteCarloNodeTbl);
					} else { 
						this.__updateOwnerScheduledMetrics(monteCarloNodeTbl, this.ownerGraphNodeTbl);
					}
				}
				
				this.__updateCriticalPathNodeCount(this.graph, monteCarloNodeTbl);
				
				cost = this.graph.getCritPathTotalCost();
				costDistribution.update(cost);
				 
				lookupCount = lookupCount + this.getLookupCount(this.graph);
				
				if (showNodes)
					this.graph.show();
				monitor.worked(1);
			}
			
			if (i < n) {
				return Run.CONTINUE;
			}

			runner._result= new OptimizationResults(costDistribution, lookupCount);	
			runner._result.setMonteCarloNodeTbl(monteCarloNodeTbl);

			return Run.DONE;
		}));
		
		return runner;
	},

	__populateMonteCarloNodes: function(graphNodeTbl, n, showNodeDistributions) {
		var monteCarloNodeTbl= [];

		for (var key in graphNodeTbl) {
			monteCarloNodeTbl[key]= new MonteCarloGraphNode(graphNodeTbl[key], n, showNodeDistributions);
		}
		
		return monteCarloNodeTbl;
	},
	
	__updateCriticalPathNodeCount: function(graph, monteCarloNodeTbl) {
		var graphNode = graph.getCriticalPathTailNode();
		
		while (graphNode != null) {
			var monteCarloGraphNode = monteCarloNodeTbl[graphNode.getId()];
			monteCarloGraphNode.incrementOnCritPathCount();
			graphNode = graphNode.getConstraintPredecessor();
		}
	},

	__updateGraphMetrics: function(graph, monteCarloNodeTbl) {
		var graphNodeTbl = graph.getGraphNodeTbl();
		
		for (var nodeId in graphNodeTbl) {
			var graphNode = graphNodeTbl[nodeId];
			var monteCarloGraphNode = monteCarloNodeTbl[nodeId]; 
			this.__updateMetrics(graphNode, monteCarloGraphNode);
		}	
	},
	
	__updateSimpleScheduledMetrics: function(scheduleResult, monteCarloNodeTbl) {
		for (var i = 0; i < scheduleResult.length; i++) {
			var graphNode = scheduleResult[i];
			
			var monteCarloGraphNode = monteCarloNodeTbl[graphNode.getId()]; 
			this.__updateMetrics(graphNode, monteCarloGraphNode);
			this.overallCost = this.overallCost + graphNode.getTotalCost();
//System.out.println("node="+ graphNode + " overall="+overallCost + " totCost=" + graphNode.getTotalCost()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			monteCarloGraphNode.updateAbleToCompleteCount(this.overallCost, this.overallTargetCost);
			monteCarloGraphNode.updateEffectiveAbilityToCompleteCount(this.overallCost, graphNode.getEffectiveTargetCost());
		}	
	},
	
	__updateOwnerScheduledMetrics: function(monteCarloNodeTbl, ownerGraphNodeTbl) {
		var overallCost = 0.0;
		
		for (var ownerId in ownerGraphNodeTbl) {
			var overallUserCost = 0.;
			var ownerNodeLst = ownerGraphNodeTbl[ownerId];
			for (var j = 0; j < ownerNodeLst.length; j++) {
				var graphNode = ownerNodeLst[j];
				var monteCarloGraphNode = monteCarloNodeTbl[graphNode.getId()]; 
				this.__updateMetrics(graphNode, monteCarloGraphNode);
				overallUserCost = overallUserCost + graphNode.getTotalCost() + graphNode.getSlack();
//	System.out.println("node="+ graphNode + " overall="+overallNodeCost + " totCost=" + graphNode.getTotalCost() + " slack=" +graphNode.getSlack()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				monteCarloGraphNode.updateAbleToCompleteCount(overallUserCost, this.overallTargetCost);
				monteCarloGraphNode.updateEffectiveAbilityToCompleteCount(overallUserCost, graphNode.getEffectiveTargetCost());
			}
			
			overallCost = Math.max(overallCost, overallUserCost);			
		}
	},
	
	__updateMetrics: function(graphNode, monteCarloGraphNode) {
		monteCarloGraphNode.updateCostFields(graphNode.getCost());	

		monteCarloGraphNode.updateSlack(graphNode.getSlack());
		monteCarloGraphNode.updateSlackPlusTotCost(graphNode.getTotalCost() + graphNode.getSlack());
//System.out.println("node="+ graphNode + " slack="+graphNode.getSlack() + " totCost=" + graphNode.getTotalCost()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	},

	__sentinel: null // terminates this class definition
});
	
})();

