/* REXX **************************************************************
 *                                                                   *
 *  Licensed Materials - Property of IBM                             *
 *  Copyright IBM Corporation 2014, 2017. All Rights Reserved.       *
 *                                                                   *
 *  Note to U.S. Government Users Restricted Rights:                 *
 *  Use, duplication or disclosure restricted by GSA ADP Schedule    *
 *  Contract with IBM Corp.                                          *
 *                                                                   *
 *********************************************************************
 *                                                                   *
 *  --------------                                                   *
 *  DtlJobComplete                                                   *
 *  --------------                                                   *
 *                                                                   *
 *  Wait for the DTL batch compile job to complete.                  *
 *                                                                   *
 *  Function:                                                        *
 *                                                                   *
 *    Waits for the DTL batch job to complete. The batch job is      *
 *    complete when the @dtl.completed file exists in the build      *
 *    done directory.                                                *
 *                                                                   *
 *  Format:                                                          *
 *                                                                   *
 *    DtlDsnCreate __BLD(bld)_COM(com)_(_options___________________  *
 *                                                                   *
 *  Parameters:                                                      *
 *                                                                   *
 *    BLD(bld)                                                       *
 *      (Required) - Build definition ID as provided by the built-   *
 *      in ant property ${buildDefinitionId}.                        *
 *                                                                   *
 *    BLD(bld)                                                       *
 *      (Required) - Working directory for common files as defined   *
 *      by the ${dir.com} ant property.                              *
 *                                                                   *
 *  Options:                                                         *
 *                                                                   *
 *    None                                                           *
 *                                                                   *
 *  Return Codes:                                                    *
 *                                                                   *
 *    RC =  0  Command successful                                    *
 *          4  Parameter error                                       *
 *          8  Functional error                                      *
 *                                                                   *
 *  External References:                                             *
 *                                                                   *
 *    None                                                           *
 *                                                                   *
 *  Notes:                                                           *
 *                                                                   *
 *    None                                                           *
 *                                                                   *
 *-------------------------------------------------------------------*
 *                                                                   *
 *  Changes:                                                         *
 *                                                                   *
 *    Date       Rel   PTM/APAR Init Description                     *
 *    ---------- ----- -------- ---- ------------------------------- *
 *    2014-11-05 4.0.6          djr  Creation                        *
 *--------------------------------------------------------EWM 4.0.7--*
 *--------------------------------------------------------EWM 5.0.0--*
 *--------------------------------------------------------EWM 5.0.1--*
 *--------------------------------------------------------EWM 5.0.2--*
 *--------------------------------------------------------EWM 6.0.0--*
 *--------------------------------------------------------EWM 6.0.1--*
 *--------------------------------------------------------EWM 6.0.2--*
 *--------------------------------------------------------EWM 6.0.3--*
 *--------------------------------------------------------EWM 6.0.4--*
 *    2017-05-16 6.0.4          djr  EE integration                  *
 *--------------------------------------------------------EWM n.n.n--*
 *--------------------------------------------------------EWM n.n.n--*
 *********************************************************************/
parse source . typecall execfn execdd execds . execadr execenv exectkn .
parse arg args
parse value args with 'BLD(' bld ')' .
parse value args with 'COM(' com ')' .
parse value args with ' ( ' options
parse value args 'TRACE(O)' with 'TRACE(' trc ')' .
exectrace = trace(trc)
signal on failure
signal on error
signal on syntax
signal on novalue
signal on halt
address sh

if bld = '' then do
  say execfn||': BLD parameter required'
  call exit 4
end

if com = '' then do
  say execfn||': COM parameter required'
  call exit 4
end

/*--  Initialization  --------------------------------------*
 *  Set initial value for internal processing variables.    *
 *----------------------------------------------------------*/
bldPfx   = substr(bld,1,min(length(bld)+1,lastpos('.',bld))-1)
bldDir   = bldPfx||'.done'

fileOk   = 0
fileName = '@dtl.completed'
filePath = com||'/'||bldDir

waitLoop = 4 /*360*/
waitSecs = 5

/*--  Process Options  -------------------------------------*
 *  Set processing overrides for user-specified options.    *
 *----------------------------------------------------------*/
do while options \= ''
  parse var options option options
  parse var option optequ '=' optval
  parse var option optpar '(' optval . ')'

  select
    when left(option,2) = '${' then nop
    when optpar = 'TRACE'      then nop

    otherwise do
      say execfn||': Invalid option' option
      call exit 4
    end
  end
end

/*--  Main Process  ----------------------------------------*
 *  Wait for the DTL batch compile job to complete.         *
 *----------------------------------------------------------*/
call directory filePath

do i = 1 to waitLoop
  if exists(fileName) \= '' then do
    fileOk = 1
    leave
  end
  call sleep waitSecs
end

if \fileOk then do
  say date('O') time('L') 'Timeout waiting for file:' fileName
  call exit 8
end

call exit 0

/*--  Exit  ------------------------------------------------*
 *  Universal exit.  Retrieve RC and return to caller.      *
 *----------------------------------------------------------*/
exit:
arg rc
exit rc

/*--  Error trapping  --------------------------------------*
 *  Encountered an unrecoverable error.  Terminate exec.    *
 *----------------------------------------------------------*/
halt:
novalue:
syntax:
error:
failure:
call exit 8
